#!/usr/bin/env python
import boto3
import glob
import re
import json
import sys
import os
import subprocess
from shutil import copyfile, rmtree
from cryptography.hazmat.primitives import serialization as crypto_serialization
from cryptography.hazmat.primitives.asymmetric import rsa
from cryptography.hazmat.backends import default_backend as crypto_default_backend


def clean(s3_bucket):
    # kill kops'd resources
    if "NAME" in os.environ:
        try:
            subprocess.call("kops delete cluster $NAME --yes", shell=True)
        except Exception as e:
            print("type error: " + str(e))
    else:
        print("You must have cluster_name set at $NAME")
        quit()
    # kill terraform'd resources
    try:
        subprocess.call("terraform destroy -force", shell=True)
    except Exception as e:
        print("type error: " + str(e))

    # kill local resources
    for folder in ["kubernetes", ".terraform", "generated"]:
        rmtree(folder, True)
    # kill state bucket will all versions
    delete_s3_bucket(s3_bucket)
    # kill remaining local resources
    for file in ["backend.tf", "variables.tf", "vars.env"]:
        os.remove(file)
    for ext in ['.pem', '.pub', '.out']:
        files = glob.glob("./*" + ext)
        for file in files:
            os.remove(file)
    return


def create_ssh_keypair(cluster_name):
    key_name = cluster_name + "_node_key"
    key = rsa.generate_private_key(
        backend=crypto_default_backend(),
        public_exponent=65537,
        key_size=4096
    )
    private_key = key.private_bytes(
        crypto_serialization.Encoding.PEM,
        crypto_serialization.PrivateFormat.PKCS8,
        crypto_serialization.NoEncryption())
    public_key = key.public_key().public_bytes(
        crypto_serialization.Encoding.OpenSSH,
        crypto_serialization.PublicFormat.OpenSSH
    )
    # save keypair
    if os.path.exists(key_name + ".pem") or os.path.exists(key_name + ".pub"):
        print("[WARN] KeyFiles Already Exist! Not Overwriting")
    else:
        print(private_key.decode('utf-8'), file=open(key_name + ".pem", "w"))
        print(public_key.decode('utf-8'), file=open(key_name + ".pub", "w"))
    return key_name


def create_s3_bucket(s3_bucket, region):
    # connect to s3 and create bucket
    s3 = boto3.client('s3')
    try:
        s3.create_bucket(
            Bucket=s3_bucket,
            ACL='private',
            CreateBucketConfiguration={
                'LocationConstraint': region
            }
        )
    except s3.exceptions.ClientError as e:
        print("[WARN] " + str(e))

    # enable versioning on bucket
    s3.put_bucket_versioning(
        Bucket=s3_bucket,
        VersioningConfiguration={
            'Status': 'Enabled'
        }
    )

    # output name of bucket
    print("[INFO] Bucket Name: {}".format(s3_bucket))
    return s3_bucket


def delete_s3_bucket(s3_bucket):
    s3 = boto3.client('s3')
    response = s3.list_objects_v2(Bucket=s3_bucket)
    if 'Contents' in response:
        for item in response['Contents']:
            print('deleting file', item['Key'])
            s3.delete_object(Bucket=s3_bucket, Key=item['Key'])
            while response['KeyCount'] == 1000:
                response = s3.list_objects_v2(
                    Bucket=s3_bucket,
                    StartAfter=response['Contents'][0]['Key'],
                )
                for item in response['Contents']:
                    print('deleting file', item['Key'])
                    s3.delete_object(Bucket=s3_bucket, Key=item['Key'])
    s3.put_bucket_versioning(
        Bucket=s3_bucket,
        VersioningConfiguration={
            'Status': 'Suspended'
        }
    )
    subprocess.call("sh bin/s3_delete.sh " + s3_bucket, shell=True)
    s3.delete_bucket(Bucket=s3_bucket)


def extract_bucket_name(file_name):
    with open(file_name, 'rt') as in_file:
        for line in in_file:
            if "bucket" in line:
                return re.findall('"([^"]*)"', line.strip())[0]


def generate_kops_terraform(cluster_name, cluster_domain, s3_bucket, key_name):
    if os.path.exists("kubernetes"):
        print("[ERROR] Cluster already exists! Use `kops edit cluster`")
    else:
        os.system(
            "kops create cluster\
            --node-count 3\
            --node-size c5.xlarge\
            --master-size m4.xlarge\
            --master-zones \"$(terraform output availability_zones | tr -d '\n')\"\
            --zones \"$(terraform output availability_zones | tr -d '\n')\"\
            --subnets \"$(terraform output private_subnets | tr -d '\n')\"\
            --utility-subnets \"$(terraform output utility_subnets | tr -d '\n')\"\
            --dns-zone \"" + cluster_domain + "\"\
            --vpc \"$(terraform output vpc_id | tr -d '\n')\"\
            --network-cidr \"$(terraform output cidr_block | tr -d '\n')\"\
            --networking \"calico\"\
            --ssh-public-key \"" + key_name + ".pub\"\
            --name \"" + cluster_name + "." + cluster_domain + "\"\
            --state \"s3://" + s3_bucket + "\"\
            --topology \"private\"\
            --bastion \
            --cloud-labels kubernetes.io/cluster/" + cluster_name + "." + cluster_domain + "=owned \
            --yes"
        )
        print(
            "export NAME=" + config_in['cluster_name'] + "." + config_in['cluster_domain'] + "\n"
            "export KOPS_STATE_STORE=s3://" + config_in['bucket_name'] + "\n", file=open("vars.env", "w+")
        )


def ingest_config(filename):
    with open(filename, 'r') as F:
        config = json.loads(F.read() or '{}')

    # convert FQDN to string - dot to dash
    s3_bucket = "{0}-state-store".format(re.sub("[\W\d]+", "-", config.get('cluster_domain').lower().strip()))
    config['bucket_name'] = s3_bucket

    return config


def init_terraform():
    os.system("terraform init")
    os.system("terraform plan -out plan.out")
    os.system("terraform apply plan.out")


def install_sshkey(key_name):
    pem_file = key_name + ".pem"
    user_ssh_path = os.path.expanduser('~/.ssh/')
    try:
        copyfile(pem_file, user_ssh_path + pem_file)
        # pem files need restrictive attributes
        os.chmod((user_ssh_path + pem_file), 0o600)
        print("[INFO] " + pem_file + " has been installed to " + user_ssh_path)
    except OSError as e:
        print("[WARN]" + str(e))
        print("[WARN] Unable to copy " + pem_file + " to " + user_ssh_path + "!\n Try Manually with CHMOD 0600")


def validate_route53(domain_name):
    route53 = boto3.client('route53')
    r53_zones = route53.list_hosted_zones()
    for n in r53_zones.get('HostedZones'):
        if domain_name in n.values():
            print("[INFO] SUCCESS! Zone: " + domain_name + " exists in this account's Route53")
            return
    print("[ERROR] Zone: " + domain_name + " does not exist in this account's Route53!!")
    quit()


def write_terraform_vars(azs, cluster_name, cluster_region, cluster_env, cluster_domain):
    # probably best to not mess with this mess >
    print(
        "// **DO NOT MODIFY** \n"
        "// This File is Managed by init.py\n"
        "\n"
        "variable \"cluster_region\" {\n"
        "  description = \"Region to deploy this cluster\"\n"
        "  default = \"" + cluster_region + "\"\n"
        "}\n"
        "\n"
        "variable \"cluster_azs\" {\n"
        "  description = \"Availability Zones to deploy resources across. Must be an ODD NUMBER for k8s!\"\n"
        "  default = " + str(azs).replace("'", "\"") + "\n"
        "  type = \"list\"\n"
        "}\n"
        "\n"
        "variable \"cluster_name\" {\n"
        "  description = \"Name for this cluster. Ex: west-coast\"\n"
        "  default = \"" + cluster_name + "\"\n"
        "}\n"
        "\n"
        "variable \"cluster_domain\" {\n"
        "  description = \"Route53 zone for this cluster\"\n"
        "  default = \"" + cluster_domain + "\"\n"
        "}\n"
        "\n"
        "variable \"vpc_cidr\" {\n"
        "  description = \"Master CIDR for the entire cluster/VPC\"\n"
        "  default = \"10.11.0.0/16\"\n"
        "}\n"
        "\n"
        "variable \"public_subnets\" {\n"
        "   default = [\"10.11.32.0/20\", \"10.11.48.0/20\", \"10.11.64.0/20\"]\n"
        "type = \"list\"\n"
        "}\n"
        "\n"
        "variable \"private_subnets\" {\n"
        "    default = [\"10.11.160.0/20\", \"10.11.176.0/20\", \"10.11.192.0/20\"]\n"
        "type = \"list\""
        "}\n"
        "\n"
        "variable \"elasticache_subnets\" {\n"
        "    default = [\"10.11.96.0/20\", \"10.11.112.0/20\", \"10.11.128.0/20\"]\n"
        "type = \"list\"\n"
        "}\n"
        "\n"
        "variable \"environment_name\" {\n"
        "  description = \"Name for the environment/stage. Ex: dev|staging|prod\"\n"
        "  default = \"" + cluster_env + "\"\n"
        "}\n", file=open("modules/base_vpc/variables.tf", "w+")
    )


def write_terraform_backend(s3_bucket, key_name, region):
    # probably best to not mess with this mess >
    print(
        "// **DO NOT MODIFY** \n"
        "// This File is Managed by init.py\n"
        "\n"
        "terraform {\n"
        "\tbackend \"s3\" {\n"
        "\t\tbucket = \"" + s3_bucket + "\"\n"
        "\t\tkey    = \"" + key_name + "\"\n"
        "\t\tregion = \"" + region + "\"\n"
        "\t}\n"
        "}\n", file=open("backend.tf", "w+")
    )


if len(sys.argv) > 1:
    args = sys.argv[1:]
    if "nukem" in args:
        config_in = ingest_config(sys.argv[1])
        bucket_name = extract_bucket_name("backend.tf")
        clean(bucket_name)
    else:
        nargs = len(sys.argv)

        # read in config file, store params
        config_in = ingest_config(sys.argv[1])

        # validate Route53 Zone exists
        validate_route53(config_in['cluster_domain'] + '.')

        # create state store bucket
        create_s3_bucket(config_in['bucket_name'], config_in['cluster_region'])

        # create ssh keypair and install on host
        ssh_key_name = create_ssh_keypair(config_in['cluster_name'])
        install_sshkey(ssh_key_name)

        # append ".tfstate" to cluster_name for tf key
        tf_backend_key = config_in['cluster_name'] + ".tfstate"

        # write out backend and vars info for Terraform state bucket
        write_terraform_backend(config_in['bucket_name'], tf_backend_key, config_in['cluster_region'])
        write_terraform_vars(config_in['availability_zones'], config_in['cluster_name'], config_in['cluster_region'],
                             config_in['cluster_env'], config_in['cluster_domain'])

        # initialize terraform
        init_terraform()

        # generate kops terraform
        generate_kops_terraform(config_in['cluster_name'], config_in['cluster_domain'], config_in['bucket_name'],
                                ssh_key_name)

        print(
            "\n\nYour k8s cluster has been built!\n"
            "**Before You Continue, Set These Variables:**\n"
            "export NAME=" + config_in['cluster_name'] + "." + config_in['cluster_domain'] + "\n"
            "export KOPS_STATE_STORE=s3://" + config_in['bucket_name'] + "\n"
            "OR\n"
            ". ./vars.env"
        )

        print(
            "To Modify your Kubernetes Cluster:\n"
            "kops edit cluster $NAME\n"
            "kops update cluster $NAME\n"
            "**Validate Your Changes**\n"
            "kops update cluster $NAME --yes\n"
            "After Terraform finishes, verify with:\n"
            "kops validate cluster $NAME\n"
        )
else:
    print("[ERROR] Requires config file location as first parameter")
