package api

import (
	"context"
	"fmt"
	"log"
	"time"

	"code.justin.tv/extensions/smart/services/smart/data"
	"code.justin.tv/extensions/smart/services/smart/rpc"
	"code.justin.tv/extensions/smart/services/smart/util"
)

const (
	OverrideSizeCap = 10 * 1024
	MessageSizeCap  = 5 * 1024
)

var SizeExceptions = map[string]bool{
	"qh13438dnohbfdnbkvo1n9hnsofrel": true, // Pretzel Rocks: Whats Playing
	"vtf6kwleslxtobyiyd8flzwspenf0g": true, // TRN Rocket League Live Tracker
	"zfh2irvx2jb4s60f02jq0ajm8vwgka": true, // Streamlabs
	"23mcx0kd450kqgzrvirylw57ogjxrv": true, // Streamlabs (Bits)
}

// PostMessage checks a message for auth and size and then stores and sends it to pubsub
func (a API) PostMessage(ctx context.Context, req *smartrpc.PostMessageRequest) (*smartrpc.Topics, error) {
	topic := req.Topics
	targets := topic.Targets

	if req.Auth == nil {
		return nil, smartrpc.NewMappedTwirpError("Twitch-Ext-Auth header not present in request", smartrpc.ErrNoAuthProvided)
	}

	err := a.Validator.CheckTokenPermission(
		ctx, topic.ExtensionId,
		req.Auth.ExtAuthToken,
		topic.ChannelId,
		"send", targets, nil)
	if err != nil {
		return nil, smartrpc.NewMappedTwirpError(err.Error(), smartrpc.ErrNotAuthorized)
	}

	message := req.Message
	messageSize, err := checkSize(message.Content, topic.ExtensionId, SizeExceptions)
	if err != nil {
		util.TrackExtensionPubsubSent(
			topic.ChannelId, topic.ExtensionId,
			"", "", 0, "", messageSize)
		return topic, smartrpc.NewMappedTwirpError("Message body too large", smartrpc.ErrMessageTooLong)
	}

	for _, target := range targets {
		topicString := topicToString(topic.ChannelId, topic.ExtensionId, target)
		sequencedMessage, err := a.Store.SequenceMessage(
			topicString, unsequencedMessageToMessage(message))
		if err != nil {
			log.Println(err.Error())
			return nil, smartrpc.NewMappedTwirpError(fmt.Sprintf("%s and other topics were not sent to.", target), smartrpc.ErrMemcacheFail)
		} else {
			err = a.PubSub.Publish(topicString, sequencedMessage)
			if err != nil {
				log.Println(err.Error())
				return nil, smartrpc.NewMappedTwirpError(fmt.Sprintf("%s and other topics were not sent to.", target), smartrpc.ErrPubSubFail)
			} else {
				util.TrackExtensionPubsubSent(
					topic.ChannelId, topic.ExtensionId,
					target, topicString, sequencedMessage.Sequence.Number,
					sequencedMessage.Sequence.Start, messageSize)
			}
		}
	}

	return topic, nil
}

func checkSize(content []string, extID string, exceptions map[string]bool) (int, error) {
	messageSize := 0
	for _, s := range content {
		messageSize = messageSize + len(s)
	}
	sizeCap := MessageSizeCap
	_, ok := exceptions[extID]
	if ok {
		sizeCap = OverrideSizeCap
	}
	if messageSize > sizeCap {
		return messageSize, smartrpc.NewMappedTwirpError("Message body too large", smartrpc.ErrMessageTooLong)
	}
	return messageSize, nil
}

func unsequencedMessageToMessage(mes *smartrpc.UnsequencedMessage) *data.Message {
	return &data.Message{ContentType: mes.ContentType, Content: mes.Content, TimeSent: time.Now()}
}
