package main

import (
	"context"
	"fmt"
	"log"
	"net/http/httptest"
	"testing"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"github.com/twitchtv/twirp"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/gds/gds/golibs/config"

	"code.justin.tv/extensions/smart/services/smart"
	"code.justin.tv/extensions/smart/services/smart/clients/pubsub"
	pubsubMocks "code.justin.tv/extensions/smart/services/smart/clients/pubsub/mocks"
	"code.justin.tv/extensions/smart/services/smart/clients/validator"
	validatorMocks "code.justin.tv/extensions/smart/services/smart/clients/validator/mocks"
	"code.justin.tv/extensions/smart/services/smart/rpc"
	"code.justin.tv/extensions/smart/services/smart/store"
	"code.justin.tv/extensions/smart/services/smart/store/memcached"
)

func buildServerAndClient(str store.Store, val validator.Client, pubs pubsub.Client, cfg config.Config) (*httptest.Server, smartrpc.Smart) {
	var server *httptest.Server
	url, ok := cfg.TryGetString("test.server.url")
	if !ok {
		stats, _ := statsd.NewNoopClient()
		server = httptest.NewServer(smart.BuildServer(stats, 3000, cfg, str, val, pubs, false))
		url = server.URL
	}

	log.Printf("Connectying to server at %s", url)
	httpClient := twitchclient.NewHTTPClient(twitchclient.ClientConf{
		Host: url,
	})
	client := smartrpc.NewSmartProtobufClient(url, httpClient)

	return server, client
}

func TestIntegration(t *testing.T) {
	if testing.Short() {
		t.Skip("Skipping integration tests because --short was specified")
	}

	awsConfig := aws.NewConfig().
		WithRegion("us-west-2")

	awsSession, err := session.NewSession(awsConfig)
	if err != nil {
		log.Fatalf("Couldn't create an AWS session: %v", err)
	}

	cfg, _ := createConfig(awsSession)

	server, client := buildServerAndClient(
		memcached.NewMemcachedStore(0, "127.0.0.1:11211"), &validatorMocks.AlwaysPass{}, &pubsubMocks.AlwaysPass{}, cfg)
	defer func() {
		if server != nil {
			server.Close()
		}
	}()
	extensionID := fmt.Sprint("t-", time.Now().Unix())
	log.Println(extensionID)
	t.Run("test get last message when no message yet exist", func(t *testing.T) {
		req := &smartrpc.Topic{
			ChannelId:   "test",
			ExtensionId: extensionID,
			Target:      "test",
		}

		_, err := client.GetLatestMessage(context.Background(), req)

		assertError(t, err, "Message not found", "404")
	})
	t.Run("test get message when no message yet exist", func(t *testing.T) {
		req := &smartrpc.SingleMessageRequest{
			Topic: &smartrpc.Topic{
				ChannelId:   "test",
				ExtensionId: extensionID,
				Target:      "test",
			},
			SequenceNumber: 1,
		}

		_, err := client.GetMessage(context.Background(), req)

		assertError(t, err, "Message not found", "404")
	})
	t.Run("test post message", func(t *testing.T) {
		req := &smartrpc.PostMessageRequest{
			Auth: &smartrpc.Token{ExtAuthToken: "test"},
			Topics: &smartrpc.Topics{
				ChannelId:   "test",
				ExtensionId: extensionID,
				Targets:     []string{"test"},
			},
			Message: &smartrpc.UnsequencedMessage{
				Content:     []string{"test"},
				ContentType: "test",
			},
		}

		topics, err := client.PostMessage(context.Background(), req)
		assert.Nil(t, err)
		assert.Len(t, topics.Targets, 0)
	})
	t.Run("test get message", func(t *testing.T) {
		req := &smartrpc.SingleMessageRequest{
			Topic: &smartrpc.Topic{
				ChannelId:   "test",
				ExtensionId: extensionID,
				Target:      "test",
			},
			SequenceNumber: 1,
		}

		message, err := client.GetMessage(context.Background(), req)
		assert.Nil(t, err)
		assert.NotNil(t, message)
		assert.Equal(t, uint64(1), message.Sequence.Number)
		assert.Equal(t, "test", message.Content[0])
		assert.Equal(t, "test", message.ContentType)
	})
	t.Run("test get last message", func(t *testing.T) {
		req := &smartrpc.Topic{
			ChannelId:   "test",
			ExtensionId: extensionID,
			Target:      "test",
		}

		message, err := client.GetLatestMessage(context.Background(), req)
		assert.Nil(t, err)
		assert.NotNil(t, message)
		assert.Equal(t, uint64(1), message.Sequence.Number)
		assert.Equal(t, "test", message.Content[0])
		assert.Equal(t, "test", message.ContentType)
	})
	t.Run("test post message on existent", func(t *testing.T) {
		req := &smartrpc.PostMessageRequest{
			Auth: &smartrpc.Token{ExtAuthToken: "test"},
			Topics: &smartrpc.Topics{
				ChannelId:   "test",
				ExtensionId: extensionID,
				Targets:     []string{"test"},
			},
			Message: &smartrpc.UnsequencedMessage{
				Content:     []string{"test"},
				ContentType: "test",
			},
		}

		topics, err := client.PostMessage(context.Background(), req)
		assert.Nil(t, err)
		assert.Len(t, topics.Targets, 0)
	})
	t.Run("test get message when the message does not exist", func(t *testing.T) {
		req := &smartrpc.SingleMessageRequest{
			Topic: &smartrpc.Topic{
				ChannelId:   "test",
				ExtensionId: extensionID,
				Target:      "test",
			},
			SequenceNumber: 99,
		}

		_, err := client.GetMessage(context.Background(), req)
		assertError(t, err, "Message not found", "404")
	})
	server.Close()
}

func assertError(t *testing.T, err error, message, code string) {
	require.NotNil(t, err)
	inner, ok := err.(twirp.Error)
	require.True(t, ok, "error Must be a twirp error")
	assert.Equal(t, message, inner.Msg())
	assert.Equal(t, code, inner.Meta("error_code"))
}
