package main

import (
	"fmt"
	"log"
	"os"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/client"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/extensions/smart/services/smart"
	"code.justin.tv/extensions/smart/services/smart/clients/pubsub"
	pubsubMocks "code.justin.tv/extensions/smart/services/smart/clients/pubsub/mocks"
	"code.justin.tv/extensions/smart/services/smart/clients/validator"
	validatorMocks "code.justin.tv/extensions/smart/services/smart/clients/validator/mocks"
	"code.justin.tv/extensions/smart/services/smart/store/memcached"
	"code.justin.tv/extensions/smart/services/smart/util"
	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/gds/gds/golibs/config"
	"code.justin.tv/gds/gds/golibs/config/builders"
)

const (
	localDebugPath        = "config/local_debug.config.json"
	defaultHystrixTimeout = 3000
	appName               = "smart"
)

func createMetrics(cfg config.Config, app, env string) (statsd.Statter, error) {
	method := cfg.GetString("stats.method", "statsd")
	switch method {
	case "noop":
		return statsd.NewNoopClient()
	case "statsd":
		if env != "test" {
			return statsd.NewBufferedClient(
				cfg.GetString("STATSD_HOST", "statsd.internal.justin.tv:8125"),
				fmt.Sprintf("%s.%s", app, env), time.Second, 0)
		}

	}
	log.Printf("Unrecognized stat method: %s. Using noop client.", method)
	return statsd.NewNoopClient()
}

func createConfig(cp client.ConfigProvider) (config.Config, config.RefreshController) {
	addDefaults := loadEnvVariables()
	defaults := builders.StandardDefaults{
		ComponentName: "smart",
		Env:           getEnv(),
		Name:          "ext",
		Sources: []*builders.Source{
			{Type: builders.File, Refresh: 10 * time.Second}},
		AdditionalDefaults: addDefaults}
	cfg, ctrl := defaults.CreateConfigAndSchedule(cp)

	log.Printf("\nConfiguration:\n%v", cfg.List().PrintTable())
	return cfg, ctrl
}
func getEnv() string {
	val := os.Getenv("ENV")
	if val != "" {
		return val
	}
	return "dev"
}

func loadEnvVariables() map[string]interface{} {
	envVariables := []string{
		"VALIDATOR_URL", "PUBSUB_URL", "MEMCACHE_HOST", "MEMCACHE_READ_TIMEOUT_MS", "STATSD_HOST",
	}
	addDefaults := map[string]interface{}{}
	for _, v := range envVariables {
		val := os.Getenv(v)
		if val != "" {
			addDefaults[v] = val
		} else {
			log.Printf("the env variable %s is missing", v)
		}
	}
	return addDefaults
}
func createValidator(cfg config.Config) validator.Client {
	host := cfg.GetString("VALIDATOR_URL", "http://localhost:8000")
	shouldPass := cfg.GetBool("test.validator.shouldPass", false)
	if shouldPass {
		return validatorMocks.AlwaysPass{}
	}
	val, err := validator.NewClient(host)
	if err != nil {
		panic(err)
	}
	return val
}

func createPubSub(cfg config.Config, stats statsd.Statter) pubsub.Client {
	host := cfg.GetString("PUBSUB_URL", "http://localhost:8000")
	shouldPass := cfg.GetBool("test.pubsub.shouldPass", false)
	if shouldPass {
		return pubsubMocks.AlwaysPass{}
	}
	val, err := pubsub.NewClient(stats, host)
	if err != nil {
		panic(err)
	}
	return val
}

func setupSpade(env string, stats statsd.StatSender) {
	if env == "production" || env == "dev" {
		util.SetupSpadeClient(stats)
		util.SetSpadeEnvironment(env)
	}
}

func main() {

	awsConfig := aws.NewConfig().WithRegion("us-west-2").WithCredentialsChainVerboseErrors(true)

	awsSession, err := session.NewSession(awsConfig)
	if err != nil {
		panic(err)
	}

	cfg, ctrl := createConfig(awsSession)
	defer ctrl.UnscheduleAll()

	env := cfg.GetString("app.env", "dev")
	statter, err := createMetrics(cfg, appName, env)
	if err != nil {
		log.Printf("Error Creating stats client. Using noop client.")
	}
	defer statter.Close()
	setupSpade(env, statter)

	twitchserver.AddDefaultSignalHandlers()
	hosts := strings.Split(cfg.GetString("MEMCACHE_HOST", "127.0.0.1:11211"), ",")
	// timeout of zero falls back to library default
	timeout := time.Duration(cfg.GetInt("MEMCACHE_READ_TIMEOUT_MS", 0)) * time.Millisecond
	store := memcached.NewMemcachedStore(timeout, hosts...)
	val := createValidator(cfg)
	pubs := createPubSub(cfg, statter)
	server := smart.BuildServer(statter, defaultHystrixTimeout, cfg, store, val, pubs, cfg.GetBool("app.shouldLog", false))
	err = twitchserver.ListenAndServe(server, &twitchserver.ServerConfig{Addr: cfg.GetString("app.port", ":8000")})
	if err != nil {
		err = fmt.Errorf("ListenAndServe failed with: %s", err)
		log.Print(err)
	}
}
