package data

import (
	"encoding/json"
	"time"
)

// Sequence contains information regarding the start of the sequence and the Number in the sequence
type Sequence struct {
	Number uint64 `json:"number"`
	Start  string `json:"start"`
}

// Message contains all information about a message
type Message struct {
	ContentType string    `json:"content_type"`
	Content     []string  `json:"content"`
	TimeSent    time.Time `json:"time_sent"`
	Sequence    Sequence  `json:"sequence"`
}

// MarshalJSON marshals a message with the correct time format
func (u *Message) MarshalJSON() ([]byte, error) {
	type Alias Message
	return json.Marshal(&struct {
		TimeSent string `json:"time_sent"`
		*Alias
	}{
		TimeSent: u.TimeSent.Format(time.RFC3339),
		Alias:    (*Alias)(u),
	})
}

// UnmarshalJSON unmarshals a message with the correct time format
func (u *Message) UnmarshalJSON(data []byte) error {
	type Alias Message
	aux := &struct {
		TimeSent string `json:"time_sent"`
		*Alias
	}{
		Alias: (*Alias)(u),
	}
	if err := json.Unmarshal(data, &aux); err != nil {
		return err
	}
	timestamp, err := time.Parse(time.RFC3339, aux.TimeSent)
	u.TimeSent = timestamp
	return err
}
