package data

import (
	"encoding/json"
	"fmt"
	"time"
)

const (
	bufferLength = 1
)

// MessageStore wraps a encodableMessageStore to prevent direct modification of the message store
type MessageStore struct {
	en encodableMessageStore
}
type encodableMessageStore struct {
	Messages []*Message `json:"messages"`
	Sequence Sequence   `json:"sequence"`
}

// NewMessageStore creates an empty store with an initialized start time.
func NewMessageStore() *MessageStore {
	return &MessageStore{en: encodableMessageStore{[]*Message{}, Sequence{0, time.Now().Format(time.RFC3339)}}}
}

// MessageStoreFromBytes creates a message store by unmarshaling bytes into an encodableMessageStore
func MessageStoreFromBytes(bytes []byte) (*MessageStore, error) {
	mesStore := encodableMessageStore{}

	err := json.Unmarshal(bytes, &mesStore)
	if err != nil {
		return nil, err
	}
	return &MessageStore{mesStore}, nil
}

// ToBytes marshals the encodableMessageStore
func (m *MessageStore) ToBytes() ([]byte, error) {
	return json.Marshal(m.en)
}

// GetMessage retuns the message in the store with the sequenceNumber or an error if it does not exist
func (m *MessageStore) GetMessage(sequenceNumber uint64) (*Message, error) {
	if sequenceNumber <= m.en.Sequence.Number && sequenceNumber > subUint(m.en.Sequence.Number, bufferLength) {
		for _, mes := range m.en.Messages {
			if mes.Sequence.Number == sequenceNumber {
				return mes, nil
			}
		}
	}
	return nil, fmt.Errorf("Message with sequence number %d not found", sequenceNumber)
}

func subUint(x, y uint64) uint64 {
	if y > x {
		return 0
	}
	return x - y
}

// GetLastMessage returns the message with the most recent sequence number or an error if it is not stored
func (m *MessageStore) GetLastMessage() (*Message, error) {
	return m.GetMessage(m.en.Sequence.Number)
}

func (m *MessageStore) nextSequence() Sequence {
	m.en.Sequence.Number = m.en.Sequence.Number + 1
	return m.en.Sequence
}

// AddMessage sequences and stores a message while managing the store size
func (m *MessageStore) AddMessage(newMessage *Message) *Message {
	seq := m.nextSequence()
	newMessage.Sequence = seq
	length := len(m.en.Messages)
	if length == bufferLength {
		added := false
		for i, mes := range m.en.Messages {
			if mes.Sequence.Number <= seq.Number-bufferLength && !added {
				m.en.Messages[i] = newMessage
				added = true
			}
		}
		if !added {
			m.en.Messages = append(m.en.Messages, newMessage)
		}
	} else if length > bufferLength {
		newMessages := make([]*Message, 0, bufferLength)
		for _, mes := range m.en.Messages {
			if mes.Sequence.Number > seq.Number-bufferLength {
				newMessages = append(newMessages, mes)
			}
		}
		newMessages = append(newMessages, newMessage)
		m.en.Messages = newMessages
	} else {
		m.en.Messages = append(m.en.Messages, newMessage)
	}
	return newMessage
}
