package data

import (
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
)

func TestNewMessageStore(t *testing.T) {
	now := time.Now().Format(time.RFC3339)
	store := NewMessageStore()
	assert.Len(t, store.en.Messages, 0)
	assert.Zero(t, store.en.Sequence.Number)
	assert.Equal(t, store.en.Sequence.Start, now)
}

func TestMessageStoreFromByte(t *testing.T) {
	bytes := []byte(`{"messages":[{"time_sent":"2006-01-02T15:04:05Z","content_type":"test","content":["test"],"sequence":{"number":4,"start":"startTime"}}],"sequence":{"number":4,"start":"startTime"}}`)
	store, err := MessageStoreFromBytes(bytes)

	assert.Nil(t, err)
	assert.Len(t, store.en.Messages, 1)
	seq := store.en.Sequence
	assert.Equal(t, seq.Number, uint64(4))
	assert.Equal(t, seq.Start, "startTime")
	seq = store.en.Messages[0].Sequence
	assert.EqualValues(t, seq.Number, uint(4))
	assert.Equal(t, seq.Start, "startTime")
}

func TestMessageStoreFromAndToBytes(t *testing.T) {
	bytes := []byte(`{"messages":[{"time_sent":"2006-01-02T15:04:05Z","content_type":"test","content":["test"],"sequence":{"number":4,"start":"startTime"}}],"sequence":{"number":4,"start":"startTime"}}`)
	store, err := MessageStoreFromBytes(bytes)
	assert.Nil(t, err)
	postBytes, err := store.ToBytes()
	assert.Nil(t, err)
	assert.EqualValues(t, bytes, postBytes)
}

func TestAddMessage(t *testing.T) {
	store := NewMessageStore()
	// Test for 0 to bufferLength+1 before add

	for i := 1; i <= bufferLength+10; i++ {
		mes := store.AddMessage(&Message{ContentType: "test", Content: []string{"test"}, TimeSent: time.Now()})
		seq := mes.Sequence
		assert.EqualValues(t, seq.Number, uint(i))
		seq = store.en.Sequence
		assert.EqualValues(t, seq.Number, uint(i))
	}
	assert.Len(t, store.en.Messages, bufferLength)
}

func TestAddMessageCoruptedSequencing(t *testing.T) {
	if bufferLength > 1 {
		store := &MessageStore{encodableMessageStore{[]*Message{
			&Message{"test", []string{"test"}, time.Now(), Sequence{4, time.Now().Format(time.RFC3339)}},
			&Message{"test", []string{"test"}, time.Now(), Sequence{4, time.Now().Format(time.RFC3339)}},
		}, Sequence{4, time.Now().Format(time.RFC3339)}}}

		for i := 1; i <= bufferLength-1; i++ {
			store.AddMessage(&Message{ContentType: "test", Content: []string{"test"}, TimeSent: time.Now()})

		}
		assert.Len(t, store.en.Messages, bufferLength+1)
		store.AddMessage(&Message{ContentType: "test", Content: []string{"test"}, TimeSent: time.Now()})
		assert.Len(t, store.en.Messages, bufferLength)
	}
}

func TestGetMessage(t *testing.T) {
	store := &MessageStore{encodableMessageStore{[]*Message{
		&Message{"test", []string{"test"}, time.Now(), Sequence{8, time.Now().Format(time.RFC3339)}},
		&Message{"test", []string{"test"}, time.Now(), Sequence{9, time.Now().Format(time.RFC3339)}},
	}, Sequence{9, time.Now().Format(time.RFC3339)}}}
	t.Run("when the message requested it too old", func(t *testing.T) {
		m, err := store.GetMessage(1)
		assert.NotNil(t, err)
		assert.Nil(t, m)
	})
	t.Run("when the message requested is in range but not saved", func(t *testing.T) {
		m, err := store.GetMessage(7)
		assert.NotNil(t, err)
		assert.Nil(t, m)
	})
	t.Run("when the message requested is in range", func(t *testing.T) {
		m, err := store.GetMessage(9)
		assert.Nil(t, err)
		assert.NotNil(t, m)

	})
	t.Run("when the message requested is too new", func(t *testing.T) {
		m, err := store.GetMessage(10)
		assert.NotNil(t, err)
		assert.Nil(t, m)
	})
}

func TestGetLastMessage(t *testing.T) {
	store := &MessageStore{encodableMessageStore{[]*Message{}, Sequence{9, time.Now().Format(time.RFC3339)}}}
	t.Run("When the message is not saved", func(t *testing.T) {
		m, err := store.GetLastMessage()
		assert.NotNil(t, err)
		assert.Nil(t, m)
	})
	store.en.Messages = []*Message{
		&Message{"test", []string{"test"}, time.Now(), Sequence{8, time.Now().Format(time.RFC3339)}},
		&Message{"test", []string{"test"}, time.Now(), Sequence{9, time.Now().Format(time.RFC3339)}},
	}
	t.Run("When the message is saved", func(t *testing.T) {
		m, err := store.GetLastMessage()
		assert.Nil(t, err)
		assert.NotNil(t, m)
	})

}
