package smartrpc

import (
	"code.justin.tv/foundation/twitchclient"
	"github.com/twitchtv/twirp"
)

const (
	errorKey = "error_key"

	ErrNoAuthProvided  = "ErrNoAuthProvided"
	ErrNotAuthorized   = "ErrNotAuthorized"
	ErrMessageTooLong  = "ErrMessageTooLong"
	ErrMemcacheFail    = "ErrMemcacheFail"
	ErrPubSubFail      = "ErrPubSubFail"
	ErrMessageNotFound = "ErrMessageNotFound"
)

func NewMappedTwirpError(msg string, errorType string) twirp.Error {
	switch errorType {
	case ErrNoAuthProvided:
		return twirp.InvalidArgumentError("auth", msg).WithMeta(errorKey, ErrNoAuthProvided)
	case ErrNotAuthorized:
		return twirp.NewError(twirp.PermissionDenied, msg).WithMeta(errorKey, ErrNotAuthorized)
	case ErrMessageTooLong:
		return twirp.NewError(twirp.OutOfRange, msg).WithMeta(errorKey, ErrMessageTooLong)
	case ErrMemcacheFail:
		return twirp.InternalError(msg).WithMeta(errorKey, ErrMemcacheFail)
	case ErrPubSubFail:
		return twirp.InternalError(msg).WithMeta(errorKey, ErrPubSubFail)
	case ErrMessageNotFound:
		return twirp.NewError(twirp.NotFound, msg).WithMeta(errorKey, ErrMessageNotFound)

	}
	return twirp.InternalError("Unrecognized error")
}

func TwirpErrorToPublicError(err twirp.Error) error {
	errorType := err.Meta(errorKey)
	switch errorType {
	case ErrNoAuthProvided:
		return &twitchclient.Error{StatusCode: 401, Message: "Twitch-Ext-Auth header not present in request"}
	case ErrNotAuthorized:
		return &twitchclient.Error{StatusCode: 403, Message: err.Msg()}
	case ErrMessageTooLong:
		return &twitchclient.Error{StatusCode: 422, Message: "Message body too large"}
	case ErrMemcacheFail:
		return &twitchclient.Error{StatusCode: 500, Message: "Unable to send message"}
	case ErrPubSubFail:
		return &twitchclient.Error{StatusCode: 500, Message: "Unable to send message"}
	case ErrMessageNotFound:
		return &twitchclient.Error{StatusCode: 404, Message: "Message not found"}

	}
	return twirp.InternalError("Something went wrong")
}
