package smart

import (
	"code.justin.tv/extensions/smart/services/smart/api"
	"code.justin.tv/extensions/smart/services/smart/clients/pubsub"
	"code.justin.tv/extensions/smart/services/smart/clients/validator"
	"code.justin.tv/extensions/smart/services/smart/rpc"
	"code.justin.tv/extensions/smart/services/smart/store"
	"code.justin.tv/extensions/smart/services/smart/util"

	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/gds/gds/golibs/config"
	hmid "code.justin.tv/gds/gds/golibs/hystrix"
	"code.justin.tv/infosec/cors"
	"github.com/twitchtv/twirp"
	twirpStatsd "github.com/twitchtv/twirp/hooks/statsd"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/cactus/go-statsd-client/statsd"
	goji "goji.io"
	"goji.io/pat"
)

var (
	corsPolicy = cors.Policy{
		AllowedOrigins: cors.Origins("*"),
		AllowMethods:   cors.Methods("GET", "POST", "PUT", "DELETE"),
		AllowHeaders:   cors.Headers("Accept", "Authorization", "Client-ID", "Twitch-Api-Token", "X-Forwarded-Proto", "X-Requested-With", "X-CSRF-Token", "Content-Type"),
	}
)

// BuildServer combines all parts that make up smart to create functioning handlers
func BuildServer(stats statsd.Statter, hystrixTimeout int, cfg config.Config, str store.Store, val validator.Client, pubs pubsub.Client, shouldLog bool) *goji.Mux {
	server := twitchserver.NewServer()
	server.Use(corsPolicy.MustMiddleware)
	server.Use(cors.BlockOnOptions)
	server.Use(hmid.CreateCircuit("unknown"))
	server.Use(config.StoreConfig(cfg))
	server.Use(util.StoreStatsd(stats))

	hystrix.DefaultTimeout = hystrixTimeout
	hystrix.DefaultMaxConcurrent = 2000
	hystrix.DefaultSleepWindow = 1000
	hystrix.DefaultErrorPercentThreshold = 60

	implementation := &api.API{Store: str, Validator: val, PubSub: pubs}
	statsHook := twirpStatsd.NewStatsdServerHooks(stats)
	loggingHook := util.NewLoggingHook(shouldLog)
	handler := smartrpc.NewSmartServer(implementation, twirp.ChainHooks(statsHook, loggingHook)) // Make a twirp http.Handler

	server.Handle(pat.Post(smartrpc.SmartPathPrefix+"*"), handler)

	return server
}
