package memcached

import (
	"log"
	"time"

	"code.justin.tv/extensions/smart/services/smart/data"
	"code.justin.tv/extensions/smart/services/smart/store"
	"github.com/bradfitz/gomemcache/memcache"
)

type memcached struct {
	client Client
}

// NewMemcachedStore returns a new memcahce based store
func NewMemcachedStore(timeout time.Duration, address ...string) store.Store {
	client := memcache.New(address...)
	client.Timeout = timeout
	return &memcached{client}
}

// GetMessage returns the message from the topic with the sequence number
func (m *memcached) GetMessage(topic string, sequenceNumber uint64) (*data.Message, error) {
	item, err := m.client.Get(topic)
	if err != nil {
		return nil, err
	}
	messageStore, err := data.MessageStoreFromBytes(item.Value)
	if err != nil {
		return nil, err
	}
	mes, err := messageStore.GetMessage(sequenceNumber)
	if err != nil {
		return nil, err
	}

	return mes, nil
}

// GetLatestMessage returns the last stored message for a topic
func (m *memcached) GetLatestMessage(topic string) (*data.Message, error) {
	item, err := m.client.Get(topic)
	if err != nil {
		return nil, err
	}
	messageStore, err := data.MessageStoreFromBytes(item.Value)
	if err != nil {
		return nil, err
	}
	message, err := messageStore.GetLastMessage()
	if err != nil {
		return nil, err
	}
	return message, nil
}
func (m *memcached) sequenceMessageForNewTopic(topic string, message *data.Message) (*data.Message, error) {
	messageStore := data.NewMessageStore()
	message = messageStore.AddMessage(message)

	bytes, err := messageStore.ToBytes()
	if err != nil {
		return nil, err
	}

	item := &memcache.Item{Key: topic, Value: bytes, Expiration: 60 * 60}
	err = m.client.Add(item)
	if err != nil {
		return nil, err
	}
	return message, nil
}

func (m *memcached) sequenceMessageForExistingTopic(topic string, item *memcache.Item, message *data.Message) (*data.Message, error) {
	messageStore, err := data.MessageStoreFromBytes(item.Value)
	if err != nil {
		return nil, err
	}
	message = messageStore.AddMessage(message)
	bytes, err := messageStore.ToBytes()
	if err != nil {
		return nil, err
	}
	item.Value = bytes
	item.Expiration = 60 * 60
	err = m.client.CompareAndSwap(item)
	if err != nil {
		return nil, err
	}
	return message, nil
}

// SequenceMessage tries to sequence and store a message
func (m *memcached) SequenceMessage(topic string, message *data.Message) (*data.Message, error) {
	var err error
	var item *memcache.Item
	var returnedMes *data.Message
	for i := 0; i < 10; i++ {
		item, err = m.client.Get(topic)
		if err == memcache.ErrCacheMiss {
			returnedMes, err = m.sequenceMessageForNewTopic(topic, message)
			if err == nil || err != memcache.ErrNotStored {
				return returnedMes, err
			}
		} else if err != nil {
			return nil, err
		} else {
			returnedMes, err = m.sequenceMessageForExistingTopic(topic, item, message)
			if err == nil || err != memcache.ErrCASConflict {
				return returnedMes, err
			} else {
				log.Println("CAS BAD for ", topic)
			}
		}
	}
	return nil, err
}
