package memcached

import (
	"testing"

	"github.com/bradfitz/gomemcache/memcache"
	"github.com/stretchr/testify/assert"

	"code.justin.tv/extensions/smart/services/smart/data"
	"code.justin.tv/extensions/smart/services/smart/store/memcached/mocks"
)

const (
	emptyStore    = `{"messages":[],"sequence":{"number":4,"start":"startTime"}}`
	nonEmptyStore = `{"messages":[{"time_sent":"2006-01-02T15:04:05Z","content_type":"test","content":["test"],"sequence":{"number":4,"start":"startTime"}}],"sequence":{"number":4,"start":"startTime"}}`
)

func TestGetMessage(t *testing.T) {
	mem := &memcached{mocks.AlwaysFail{}}
	t.Run("when there is a cache miss", func(t *testing.T) {
		mes, err := mem.GetMessage("topic", 1)
		assert.Equal(t, memcache.ErrCacheMiss, err)
		assert.Nil(t, mes)
	})
	t.Run("when the topic exists but the requested store does not parse", func(t *testing.T) {
		mem := &memcached{&mocks.Variable{GetReturns: []mocks.GetTuple{mocks.GetTuple{
			&memcache.Item{Value: []byte(``)}, nil}}}}
		mes, err := mem.GetMessage("topic", 3)
		assert.NotNil(t, err)
		assert.Nil(t, mes)
	})
	t.Run("when the topic exists but the requested message does not", func(t *testing.T) {
		mem := &memcached{&mocks.Variable{GetReturns: []mocks.GetTuple{mocks.GetTuple{
			&memcache.Item{Value: []byte(emptyStore)}, nil}}}}
		mes, err := mem.GetMessage("topic", 3)
		assert.NotNil(t, err)
		assert.Nil(t, mes)
	})
	t.Run("when the topic exists and so does the requested message", func(t *testing.T) {
		mem := &memcached{&mocks.Variable{GetReturns: []mocks.GetTuple{mocks.GetTuple{
			&memcache.Item{Value: []byte(nonEmptyStore)}, nil}}}}
		mes, err := mem.GetMessage("topic", 4)
		assert.Nil(t, err)
		assert.NotNil(t, mes)
	})
}

func TestGetLatestMessage(t *testing.T) {
	mem := &memcached{mocks.AlwaysFail{}}
	t.Run("when there is a cache miss", func(t *testing.T) {
		mes, err := mem.GetLatestMessage("topic")
		assert.Equal(t, memcache.ErrCacheMiss, err)
		assert.Nil(t, mes)
	})
	t.Run("when the topic exists but the requested store does not parse", func(t *testing.T) {
		mem := &memcached{&mocks.Variable{GetReturns: []mocks.GetTuple{mocks.GetTuple{
			&memcache.Item{Value: []byte(``)}, nil}}}}
		mes, err := mem.GetLatestMessage("topic")
		assert.NotNil(t, err)
		assert.Nil(t, mes)
	})
	t.Run("when the topic exists but the requested message does not", func(t *testing.T) {
		mem := &memcached{&mocks.Variable{GetReturns: []mocks.GetTuple{mocks.GetTuple{
			&memcache.Item{Value: []byte(emptyStore)}, nil}}}}
		mes, err := mem.GetLatestMessage("topic")
		assert.NotNil(t, err)
		assert.Nil(t, mes)
	})
	t.Run("when the topic exists and so does the requested message", func(t *testing.T) {
		mem := &memcached{&mocks.Variable{GetReturns: []mocks.GetTuple{mocks.GetTuple{
			&memcache.Item{Value: []byte(nonEmptyStore)}, nil}}}}
		mes, err := mem.GetLatestMessage("topic")
		assert.Nil(t, err)
		assert.NotNil(t, mes)
	})
}

func TestSequenceMessage(t *testing.T) {
	t.Run("when there is a cache miss and you cant store anything", func(t *testing.T) {
		mem := &memcached{mocks.AlwaysFail{}}
		mes, err := mem.SequenceMessage("topic", &data.Message{})
		assert.Equal(t, memcache.ErrNotStored, err)
		assert.Nil(t, mes)
	})

	t.Run("when there is a cache miss and you can eventually store", func(t *testing.T) {
		mem := &memcached{&mocks.Variable{GetReturns: []mocks.GetTuple{mocks.GetTuple{nil, memcache.ErrCacheMiss}}, AddReturns: []error{memcache.ErrNotStored, nil}}}
		mes, err := mem.SequenceMessage("topic", &data.Message{})
		assert.Nil(t, err)
		assert.NotNil(t, mes)
	})
	t.Run("when the item exists but cas checks fail", func(t *testing.T) {
		mem := &memcached{&mocks.Variable{GetReturns: []mocks.GetTuple{mocks.GetTuple{
			&memcache.Item{Value: []byte(emptyStore)}, nil}},
			CasReturns: []error{memcache.ErrCASConflict}}}
		mes, err := mem.SequenceMessage("topic", &data.Message{})
		assert.Equal(t, err, memcache.ErrCASConflict)
		assert.Nil(t, mes)
	})
	t.Run("when the item exists and cas eventual works", func(t *testing.T) {
		mem := &memcached{&mocks.Variable{GetReturns: []mocks.GetTuple{
			mocks.GetTuple{&memcache.Item{Value: []byte(emptyStore)}, nil}},
			CasReturns: []error{memcache.ErrCASConflict, nil}}}
		mes, err := mem.SequenceMessage("topic", &data.Message{})
		assert.Nil(t, err)
		assert.NotNil(t, mes)
	})
}
