package util

import (
	"context"
	"log"
	"strings"
	"time"

	"github.com/twitchtv/twirp"
)

var reqStartTimestampKey = new(int)
var reqErrorKey = new(int)

func markReqStart(ctx context.Context) context.Context {
	return context.WithValue(ctx, reqStartTimestampKey, time.Now())
}
func markError(ctx context.Context, err twirp.Error) context.Context {
	return context.WithValue(ctx, reqErrorKey, err)
}

func getReqStart(ctx context.Context) (time.Time, bool) {
	t, ok := ctx.Value(reqStartTimestampKey).(time.Time)
	return t, ok
}

func getError(ctx context.Context) (twirp.Error, bool) {
	t, ok := ctx.Value(reqErrorKey).(twirp.Error)
	return t, ok
}

func NewLoggingHook(shouldlog bool) *twirp.ServerHooks {
	hooks := &twirp.ServerHooks{}
	if shouldlog {
		hooks.RequestReceived = func(ctx context.Context) (context.Context, error) {
			ctx = markReqStart(ctx)
			return ctx, nil
		}
		hooks.Error = func(ctx context.Context, err twirp.Error) context.Context {
			ctx = markError(ctx, err)
			return ctx
		}
		hooks.ResponseSent = func(ctx context.Context) {
			// Three pieces of data to get, none are guaranteed to be present:
			// - time that the request started
			// - method that was called
			// - status code of response
			var (
				start  time.Time
				method string
				status string
				err    twirp.Error

				// haveStart  bool
				// haveMethod bool
				// haveStatus bool
				haveError bool
			)

			start, _ = getReqStart(ctx)
			err, haveError = getError(ctx)
			method, _ = twirp.MethodName(ctx)
			status, _ = twirp.StatusCode(ctx)

			method = sanitize(method)
			status = sanitize(status)
			errorCode := "-"
			if haveError {
				errorCode = err.Meta("error_code")
			}
			dur := time.Now().Sub(start)
			log.Println("-", time.Now(), "POST", method, "-", status, dur, errorCode)

		}
	}
	return hooks
}

func sanitize(s string) string {
	return strings.Map(sanitizeRune, s)
}

func sanitizeRune(r rune) rune {
	switch {
	case 'a' <= r && r <= 'z':
		return r
	case '0' <= r && r <= '9':
		return r
	case 'A' <= r && r <= 'Z':
		return r
	default:
		return '_'
	}
}
