package util

import (
	"context"
	"fmt"
	"log"
	"net/url"
	"time"

	"code.justin.tv/common/config"
	"code.justin.tv/common/spade-client-go/spade"

	"github.com/cactus/go-statsd-client/statsd"
)

// SpadeMetadata is common information for all events sent to spade
type SpadeMetadata struct {
	Environment string `json:"environment"`
}

// ExtensionPubsubSent represents a pubsub event
type ExtensionPubsubSent struct {
	SpadeMetadata
	ChannelID      string `json:"channel_id"`
	ExtensionID    string `json:"extension_id"`
	Target         string `json:"target"`
	Topic          string `json:"topic"`
	SequenceNumber uint64 `json:"sequence_number"`
	SequenceStart  string `json:"sequence_start"`
	PayloadSize    int    `json:"payload_size"`
}

var (
	spadeClient          spade.Client
	defaultSpadeMetadata = SpadeMetadata{config.Environment()}
	spadeTimeout         = 2 * time.Second
)

// SetupSpadeClient sets up default spade client
func SetupSpadeClient(stats statsd.StatSender) {
	spadeBaseURL, err := url.Parse("https://spade.twitch.tv")
	if err != nil {
		log.Fatalf("Error creating spade base URL %v", err)
	}
	if spadeBaseURL == nil {
		log.Fatal("url.Parse returned a nil URL")
	}

	spadeClient, err = spade.NewClient(
		spade.InitBaseURL(*spadeBaseURL),
		spade.InitStatHook(func(name string, httpStatusCode int, d time.Duration) {
			_ = stats.Inc(fmt.Sprintf("spade.%s.%d", name, httpStatusCode), 1, 1)
			_ = stats.TimingDuration(fmt.Sprintf("spade.%s", name), d, 1)
		}))

	if err != nil {
		log.Fatalf("Error creating spade client %v", err)
		spadeClient = nil
	}
}

// SetSpadeEnvironment overrides default environment
// This should be called because the 'real' environment will not be available
// when the default one is set due to timing
func SetSpadeEnvironment(env string) {
	defaultSpadeMetadata.Environment = env
}

// TrackExtensionPubsubSent sends to spade a `extension_pubsub_sent` event
func TrackExtensionPubsubSent(channelID string, extensionID string, target string, topic string, sequenceNumber uint64, sequenceStart string, payloadSize int) {
	if spadeClient != nil {
		ctx, cancel := context.WithTimeout(context.Background(), spadeTimeout)
		defer cancel()
		err := spadeClient.TrackEvent(ctx, "extension_pubsub_sent", ExtensionPubsubSent{
			defaultSpadeMetadata,
			channelID, extensionID, target, topic, sequenceNumber, sequenceStart, payloadSize,
		})
		if err != nil {
			log.Printf("Error sending extension_pubsub_sent event to spade: %v", err)
		}
	}
}
