package twitchclient

import "context"

// ReqOpts contains common request options used to interface with Twitch services.
type ReqOpts struct {
	// AuthorizationToken is the token received from our Authorization service
	// which can be forwarded to other Twitch services. Do not set this if
	// the request leaves the Twitch infrastructure.
	AuthorizationToken string

	// ClientID is the alphanumeric ID of the client making the request.
	ClientID string

	// ClientRowID is the database row ID of the client making the request.
	ClientRowID string

	// Enables request timing stats, which are in the format:
	//    [{StatNamePrefix}.]{StatName}.{RespStatusCode}
	// The StatNamePrefix is part of ClientConf and is optional.
	// The RespStatusCode is the response status, or 0 for errors.
	// The sample rate depends on the StatSampleRate option, being 0.1 by default.
	StatName string

	// Sample rate for both request timing stats and http tracing stats.
	// Defaults to 0.1
	StatSampleRate float32
}

var reqOptsCtxKey = new(int)

// WithReqOpts annotates a context with options for the twitchclient RoundTripper
func WithReqOpts(ctx context.Context, reqOpts ReqOpts) context.Context {
	return context.WithValue(ctx, reqOptsCtxKey, reqOpts)
}

func getReqOpts(ctx context.Context) ReqOpts {
	if reqOpts, ok := ctx.Value(reqOptsCtxKey).(ReqOpts); ok {
		return reqOpts
	}
	return ReqOpts{}
}

// MergeReqOpts returns a new ReqOpts struct using defaults instead of zero-values.
func MergeReqOpts(reqOpts *ReqOpts, defVal ReqOpts) ReqOpts {
	if reqOpts == nil {
		return defVal
	}
	return ReqOpts{
		AuthorizationToken: stringWithDefault(reqOpts.AuthorizationToken, defVal.AuthorizationToken),
		ClientID:           stringWithDefault(reqOpts.ClientID, defVal.ClientID),
		ClientRowID:        stringWithDefault(reqOpts.ClientRowID, defVal.ClientRowID),
		StatName:           stringWithDefault(reqOpts.StatName, defVal.StatName),
		StatSampleRate:     float32WithDefault(reqOpts.StatSampleRate, defVal.StatSampleRate),
	}
}

func stringWithDefault(val, defVal string) string {
	if val != "" {
		return val
	}
	return defVal
}

func float32WithDefault(val, defVal float32) float32 {
	if val != 0 {
		return val
	}
	return defVal
}
