package validator

import (
	"bytes"
	"context"
	"encoding/json"
	"io"
	"net/http"
	"strings"
	"time"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/gds/gds/extensions/validator/model"
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "extensions"
)

// Client expressing actions that can be taken by the validator system
type Client interface {
	ReadSecrets(ctx context.Context, clientID string, reqOpts *twitchclient.ReqOpts) (*model.SecretCollection, error)
	DeleteSecrets(ctx context.Context, clientID string, reqOpts *twitchclient.ReqOpts) error
	UpdateSecrets(ctx context.Context, clientID string, activationDelay time.Duration, reqOpts *twitchclient.ReqOpts) (*model.SecretCollection, error)
	CreateAnonToken(ctx context.Context, clientID string, channelID string, reqOpts *twitchclient.ReqOpts) (*model.PermissionData, error)
	CreateAnonTokens(ctx context.Context, clientIDs []string, channelID string, reqOpts *twitchclient.ReqOpts) (*model.PermissionDataCollection, error)
	CreateUserToken(ctx context.Context, clientID string, channelID string, userID string, reqOpts *twitchclient.ReqOpts) (*model.PermissionData, error)
	CreateUserTokens(ctx context.Context, clientIDs []string, channelID string, userID string, reqOpts *twitchclient.ReqOpts) (*model.PermissionDataCollection, error)
	RefreshAnonToken(ctx context.Context, clientID string, jwt string, reqOpts *twitchclient.ReqOpts) (string, error)
	RefreshUserToken(ctx context.Context, clientID string, userID string, jwt string, reqOpts *twitchclient.ReqOpts) (string, error)
	CheckTokenPermission(ctx context.Context, clientID string, jwt string, channel string, verb string, targets []string, reqOpts *twitchclient.ReqOpts) error
	LinkExtensionForUser(ctx context.Context, clientID string, userID string, jwt string, showUser bool, reqOpts *twitchclient.ReqOpts) (*model.PermissionData, error)
	LinkExtension(ctx context.Context, clientID string, userID string, jwt string, showUser bool, reqOpts *twitchclient.ReqOpts) error
	VerifyExtensionGrants(ctx context.Context, clientID string, jwt string, grants []string, reqOpts *twitchclient.ReqOpts) (*model.VerifiedExtensionGrants, error)
	UpdateExtensionGrants(ctx context.Context, clientID string, userID string, jwt string, grants map[string]bool, reqOpts *twitchclient.ReqOpts) (map[string]bool, error)
	DeleteExtensionGrants(ctx context.Context, clientID string, userID string, reqOpts *twitchclient.ReqOpts) error
	GetLinkedExtensions(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*model.LinkedExtensions, error)
	LookupOpaqueID(ctx context.Context, clientID string, opaqueID string, reqOpts *twitchclient.ReqOpts) (string, error)
	HealthCheck(ctx context.Context, reqOpts *twitchclient.ReqOpts) error
}

type clientImpl struct {
	http twitchclient.Client
}

// NewClient creates an http client to call the validator endpoints
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}

	twitchClient, err := twitchclient.NewClient(conf)
	if err != nil {
		return nil, err
	}

	return &clientImpl{
		http: twitchClient,
	}, nil
}

// ReadSecrets allows an authorized user to retrieve the current JWT signing secrets for an extension
func (cli *clientImpl) ReadSecrets(ctx context.Context, clientID string, reqOpts *twitchclient.ReqOpts) (*model.SecretCollection, error) {
	req, err := cli.http.NewRequest("GET", "/client/"+clientID+"/secrets", nil)
	if err != nil {
		return nil, err
	}
	var out model.SecretCollection
	err = cli.execute(ctx, req, "get_client_secrets", &out, reqOpts)
	return &out, err
}

func (cli *clientImpl) DeleteSecrets(ctx context.Context, clientID string, reqOpts *twitchclient.ReqOpts) error {
	req, err := cli.http.NewRequest("DELETE", "/client/"+clientID+"/secrets", nil)
	if err != nil {
		return err
	}
	err = cli.execute(ctx, req, "delete_client_secrets", nil, reqOpts)
	return err
}

func (cli *clientImpl) createPCSBody(activationDelay time.Duration) io.Reader {
	out := &bytes.Buffer{}
	// Swallowing the error here since this simple a value to
	// encode won't run the risk of returning
	// json.UnsupportedValueError
	_ = json.NewEncoder(out).Encode(map[string]interface{}{"activation_delay": activationDelay})
	return out
}

// UpdateSecrets allows an authorized user to rotate the current JWT signing secrets for an extension
func (cli *clientImpl) UpdateSecrets(ctx context.Context, clientID string, activationDelay time.Duration, reqOpts *twitchclient.ReqOpts) (*model.SecretCollection, error) {
	pcsBody := cli.createPCSBody(activationDelay)

	req, err := cli.http.NewRequest("PUT", "/client/"+clientID+"/secrets", pcsBody)
	if err != nil {
		return nil, err
	}
	var out model.SecretCollection
	err = cli.execute(ctx, req, "put_client_secrets", &out, reqOpts)
	return &out, err
}

// CheckTokenPermission reads the provided token and verify it allows the reuested permissions for the given channel/verb/targets
func (cli *clientImpl) CheckTokenPermission(ctx context.Context, clientID string, jwt string, channel string, verb string, targets []string, reqOpts *twitchclient.ReqOpts) error {
	req, err := cli.http.NewRequest("GET", "/client/"+clientID+"/permission/"+verb+"?channel="+channel+"&targets="+strings.Join(targets, ","), nil)
	if err != nil {
		return err
	}
	cli.setJwt(req, jwt)
	var out struct{}
	return cli.execute(ctx, req, "get_client_permission", &out, reqOpts)
}

// CreateAnonToken generates a new JWT for an anonymous session on the the given client and channel
func (cli *clientImpl) CreateAnonToken(ctx context.Context, clientID string, channelID string, reqOpts *twitchclient.ReqOpts) (*model.PermissionData, error) {
	pctBody := cli.createPCTBody(channelID)

	req, err := cli.http.NewRequest("POST", "/client/"+clientID+"/token", pctBody)
	if err != nil {
		return nil, err
	}
	var out model.PermissionData
	err = cli.execute(ctx, req, "post_client_token_anon", &out, reqOpts)
	return &out, err
}

// CreateAnonTokens generates a new JWT for an anonymous session on the the given client and channel
func (cli *clientImpl) CreateAnonTokens(ctx context.Context, clientIDs []string, channelID string, reqOpts *twitchclient.ReqOpts) (*model.PermissionDataCollection, error) {
	pctsBody := cli.createPCTsBody(clientIDs)

	req, err := cli.http.NewRequest("POST", "/channel/"+channelID+"/token", pctsBody)
	if err != nil {
		return nil, err
	}
	var out model.PermissionDataCollection
	err = cli.execute(ctx, req, "post_client_tokens_anon", &out, reqOpts)
	return &out, err
}

// CreateUserToken generates a new JWT for a logged in user on the given client and channel
func (cli *clientImpl) CreateUserToken(ctx context.Context, clientID string, channelID string, userID string, reqOpts *twitchclient.ReqOpts) (*model.PermissionData, error) {
	pctBody := cli.createPCTBody(channelID)
	req, err := cli.http.NewRequest("POST", "/client/"+clientID+"/token/"+userID, pctBody)
	if err != nil {
		return nil, err
	}
	var out model.PermissionData
	err = cli.execute(ctx, req, "post_client_token_user", &out, reqOpts)
	return &out, err
}

// CreateUserTokens generates a new JWT for a logged in user on the given client and channel
func (cli *clientImpl) CreateUserTokens(ctx context.Context, clientIDs []string, channelID string, userID string, reqOpts *twitchclient.ReqOpts) (*model.PermissionDataCollection, error) {
	pctsBody := cli.createPCTsBody(clientIDs)
	req, err := cli.http.NewRequest("POST", "/channel/"+channelID+"/token/"+userID, pctsBody)
	if err != nil {
		return nil, err
	}
	var out model.PermissionDataCollection
	err = cli.execute(ctx, req, "post_client_tokens_user", &out, reqOpts)
	return &out, err
}

// RefreshAnonToken extends the expiration on a valid anonymous token
func (cli *clientImpl) RefreshAnonToken(ctx context.Context, client string, jwt string, reqOpts *twitchclient.ReqOpts) (string, error) {
	req, err := cli.http.NewRequest("GET", "/client/"+client+"/token", nil)
	if err != nil {
		return "", err
	}
	cli.setJwt(req, jwt)
	var out string
	err = cli.execute(ctx, req, "get_token", &out, reqOpts)
	return out, err
}

// RefreshUserToken extends the expiration on a valid user token
func (cli *clientImpl) RefreshUserToken(ctx context.Context, client string, userID string, jwt string, reqOpts *twitchclient.ReqOpts) (string, error) {
	req, err := cli.http.NewRequest("GET", "/client/"+client+"/token/"+userID, nil)
	if err != nil {
		return "", err
	}
	cli.setJwt(req, jwt)
	var out string
	err = cli.execute(ctx, req, "get_token", &out, reqOpts)
	return out, err
}

// LinkExtensionForUser link/unlinks user with an extension and returns updated JWT token
func (cli *clientImpl) LinkExtensionForUser(ctx context.Context, clientID string, userID string, jwt string, showUser bool, reqOpts *twitchclient.ReqOpts) (*model.PermissionData, error) {
	pctBody := cli.createPCULBodyDeprecated(jwt, showUser)
	req, err := cli.http.NewRequest("POST", "/client/"+clientID+"/user/"+userID+"/link", pctBody)
	if err != nil {
		return nil, err
	}
	var out model.PermissionData
	err = cli.execute(ctx, req, "post_client_token_user", &out, reqOpts)
	return &out, err
}

// LinkExtension link/unlinks user with an extension
func (cli *clientImpl) LinkExtension(ctx context.Context, clientID string, userID string, token string, showUser bool, reqOpts *twitchclient.ReqOpts) error {
	pctBody := cli.createPCULBody(showUser, token)
	req, err := cli.http.NewRequest("PUT", "/client/"+clientID+"/user/"+userID+"/link", pctBody)
	if err != nil {
		return err
	}
	err = cli.execute(ctx, req, "put_client_token_user", nil, reqOpts)
	return err
}

// VerifyExtensionGrants filters a list of requested grants according to what a user has
// explicitly allowed; if the user has not linked their account the function will return
// an unauthoriezd error.
func (cli *clientImpl) VerifyExtensionGrants(ctx context.Context, clientID string, jwt string, grants []string, reqOpts *twitchclient.ReqOpts) (*model.VerifiedExtensionGrants, error) {
	req, err := cli.http.NewRequest("GET", "/client/"+clientID+"/grants?requested"+strings.Join(grants, ","), nil)
	if err != nil {
		return nil, err
	}
	cli.setJwt(req, jwt)
	var out model.VerifiedExtensionGrants
	err = cli.execute(ctx, req, "get_client_grants", &out, reqOpts)
	return &out, err
}

// UpdateExtensionGrants updates grants an extension with specified permissions for a given user
func (cli *clientImpl) UpdateExtensionGrants(ctx context.Context, clientID string, userID string, jwt string, grants map[string]bool, reqOpts *twitchclient.ReqOpts) (map[string]bool, error) {
	pcgBody := cli.createPCGBody(jwt, grants)
	req, err := cli.http.NewRequest("PUT", "/client/"+clientID+"/grants/"+userID, pcgBody)
	if err != nil {
		return nil, err
	}

	out := map[string]bool{}
	err = cli.execute(ctx, req, "put_client_grants", &out, reqOpts)
	return out, err
}

// DeleteExtensionGrants removes all grants by a user for the given extension
func (cli *clientImpl) DeleteExtensionGrants(ctx context.Context, clientID string, userID string, reqOpts *twitchclient.ReqOpts) error {
	req, err := cli.http.NewRequest("DELETE", "/client/"+clientID+"/grants/"+userID, nil)
	if err != nil {
		return err
	}
	err = cli.execute(ctx, req, "delete_client_grants", nil, reqOpts)
	return err
}

// GetLinkedExtensions get all linked extensions for the given user
func (cli *clientImpl) GetLinkedExtensions(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*model.LinkedExtensions, error) {
	req, err := cli.http.NewRequest("GET", "/user/"+userID+"/clients/linked", nil)
	if err != nil {
		return nil, err
	}

	var out model.LinkedExtensions
	err = cli.execute(ctx, req, "get_linked_clients", &out, reqOpts)
	return &out, err
}

// LookupOpaqueID looks up the opaque id and returns the user id
func (cli *clientImpl) LookupOpaqueID(ctx context.Context, clientID string, opaqueID string, reqOpts *twitchclient.ReqOpts) (string, error) {
	req, err := cli.http.NewRequest("GET", "/client/"+clientID+"/opaque/"+opaqueID, nil)
	if err != nil {
		return "", err
	}
	var out string
	err = cli.execute(ctx, req, "get_client_user_by_opaque_id", &out, reqOpts)
	return out, err
}

func (cli *clientImpl) HealthCheck(ctx context.Context, reqOpts *twitchclient.ReqOpts) error {
	req, err := cli.http.NewRequest("GET", "/debug/running", nil)
	if err != nil {
		return err
	}
	_, err = cli.http.Do(ctx, req, twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{}))
	return err
}

func (cli *clientImpl) createPCTBody(channel string) io.Reader {
	out := &bytes.Buffer{}
	// Swallowing the error here since this simple a value to
	// encode won't run the risk of returning
	// json.UnsupportedValueError
	_ = json.NewEncoder(out).Encode(map[string]string{"channel_id": channel})
	return out
}

func (cli *clientImpl) createPCTsBody(clients []string) io.Reader {
	out := &bytes.Buffer{}
	// Swallowing the error here since this simple a value to
	// encode won't run the risk of returning
	// json.UnsupportedValueError
	_ = json.NewEncoder(out).Encode(map[string]interface{}{"client_ids": clients})
	return out
}

func (cli *clientImpl) createPCULBody(showUser bool, token string) io.Reader {
	out := &bytes.Buffer{}
	// Swallowing the error here since this simple a value to
	// encode won't run the risk of returning
	// json.UnsupportedValueError
	_ = json.NewEncoder(out).Encode(map[string]interface{}{"show_user": showUser, "token": token})
	return out
}

func (cli *clientImpl) createPCULBodyDeprecated(jwt string, showUser bool) io.Reader {
	out := &bytes.Buffer{}
	// Swallowing the error here since this simple a value to
	// encode won't run the risk of returning
	// json.UnsupportedValueError
	_ = json.NewEncoder(out).Encode(map[string]interface{}{"token": jwt, "show_user": showUser})
	return out
}

func (cli *clientImpl) createPCGBody(jwt string, grants map[string]bool) io.Reader {
	out := &bytes.Buffer{}
	// Swallowing the error here since this simple a value to
	// encode won't run the risk of returning
	// json.UnsupportedValueError
	_ = json.NewEncoder(out).Encode(map[string]interface{}{"token": jwt, "grants": grants})
	return out
}

func (cli *clientImpl) setJwt(req *http.Request, jwt string) {
	req.Header.Add("Authorization", "Bearer "+jwt)
}

func (cli *clientImpl) execute(ctx context.Context, req *http.Request, statName string, out interface{}, reqOpts *twitchclient.ReqOpts) error {
	opts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions." + statName,
		StatSampleRate: defaultStatSampleRate,
	})

	_, err := cli.http.DoJSON(ctx, out, req, opts)
	return err
}
