package config

import (
	"context"
	"net/http"

	"github.com/jixwanwang/apiutils"
)

const (
	// ConfigOverrideContentHeaderName is the HTTP header name to use when sending JSON config overrides via header
	ConfigOverrideContentHeaderName = "config-overrides"
	// ConfigOverrideSecretHeaderName is the HTTP header that stores the secret to unlock config overrides
	ConfigOverrideSecretHeaderName = "config-secret"
	// ConfigOverrideName is the name of the source constructed by the override logic
	ConfigOverrideName = "http override"
	// ConfigKeyHTTPHeaderAllowed is the configuration key to turn on HTTP header config overrides
	ConfigKeyHTTPHeaderAllowed = "config.http.header.allowed"
	// ConfigKeyHTTPHeaderSecret contains an arbitrary string that must be matched exactly in HTTP override requests
	ConfigKeyHTTPHeaderSecret = "config.http.header.secret"
)

var contextKey = new(int)

// Load pulls the current configuration from a context; it will return an empty
// configuration if none is found.
func Load(ctx context.Context) Config {
	if cfg, ok := ctx.Value(contextKey).(Config); ok {
		return cfg
	}
	return NewConfig()
}

// StoreConfig builds a goji middleware that automatically places the config
// argument into the context of each request; if the argument specifies that
// overrides are allowed this middleware will also check the headers of the
// request and automatically unpack configuration overrides if they are in
// valid JSON format. To manually place a config into a context, see the
// Config.Store function.
func StoreConfig(config Config) func(http.Handler) http.Handler {
	return func(inner http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			cfg := config
			// is the override turned on?
			if r.Header.Get(ConfigOverrideContentHeaderName) != "" && config.GetBool(ConfigKeyHTTPHeaderAllowed, false) {
				// does the secret header match the configured value?
				secret, ok := config.TryGetString(ConfigKeyHTTPHeaderSecret)
				if ok && secret == r.Header.Get(ConfigOverrideSecretHeaderName) {
					// allow the override
					if src, err := FromJSON(ConfigOverrideName, []byte(r.Header.Get(ConfigOverrideContentHeaderName))); err == nil {
						cfg = cfg.WithOverrides(src)
					}
				}
			}
			inner.ServeHTTP(w, r.WithContext(cfg.Store(r.Context())))
		})
	}
}

// Display is a http.ContextCall for config display
func Display(ctx context.Context) (interface{}, *apiutils.ErrorResponse) {
	return Load(ctx).List().BySource(), nil
}
