package sources

import (
	"strings"

	"code.justin.tv/gds/gds/golibs/config/types"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/ec2"
	"github.com/aws/aws-sdk-go/service/ec2/ec2iface"
)

const (
	// ConfigKeyEC2Prefix holds the substring that will be used to find legal configuration in EC2 tags
	ConfigKeyEC2Prefix = "config.ec2.prefix"

	defaultEC2Prefix    = "twitch:ds:"
	ec2RefreshLogicName = "ec2:"
)

type ec2RefreshLogic struct {
	service ec2iface.EC2API
	input   *ec2.DescribeTagsInput
}

// NewEC2RefreshLogic constructs a RefreshLogic that pulls configuration from EC2 tags
func NewEC2RefreshLogic(instanceID string, service ec2iface.EC2API) RefreshLogic {
	filter := &ec2.Filter{
		Name:   aws.String("resource-id"),
		Values: []*string{aws.String(instanceID)},
	}

	return &ec2RefreshLogic{
		service,
		new(ec2.DescribeTagsInput).SetFilters([]*ec2.Filter{filter}),
	}
}

func (a *ec2RefreshLogic) GetName() string { return ec2RefreshLogicName }

func (a *ec2RefreshLogic) GetValues(settings Settings) (Source, error) {
	if a.service == nil || settings == nil {
		return nil, types.ErrArgumentWasNil
	}
	prefix, ok := settings.TryGetString(ConfigKeyEC2Prefix)
	if !ok {
		prefix = defaultEC2Prefix
	}
	prefixLen := len(prefix)
	tags, err := a.service.DescribeTags(a.input)
	if err != nil {
		return nil, err
	}

	data := make(map[string]interface{})
	for _, v := range tags.Tags {
		if v.Key != nil && v.Value != nil && strings.HasPrefix(*v.Key, prefix) {
			key := strings.Replace((*v.Key)[prefixLen:], ":", ".", -1)
			data[key] = *v.Value
		}
	}
	return NewStaticSource(a.GetName()+" "+prefix, data)
}
