package sources

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"sort"
	"strings"

	"code.justin.tv/gds/gds/golibs/awsutil"
	"code.justin.tv/gds/gds/golibs/config/types"
	"code.justin.tv/gds/gds/golibs/discovery"

	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3iface"
)

// ErrMissingS3ConfigurationBucket is returned if S3 refresh logic can't find any bucket
var ErrMissingS3ConfigurationBucket = errors.New("Could not find S3 configuration bucket")

// ErrMultipleS3ConfigurationBuckets is returned if S3 refresh logic finds multiple buckets
var ErrMultipleS3ConfigurationBuckets = errors.New("Found more than one S3 configuration bucket")

const (
	// ConfigKeyS3Prefix is the config key used to prefix buckets if present
	ConfigKeyS3Prefix = "config.s3.prefix"
	// ConfigKeyAppName is the config key that holds the current application's name declaration
	ConfigKeyAppName = "app.name"
	// ConfigKeyAppEnvironment is the config key that holds the current application's environment declaration
	ConfigKeyAppEnvironment = "app.env"
	// ConfigKeyAppComponent is the config key that holds the current application's component declaration
	ConfigKeyAppComponent = "app.component_name"

	defaultTagKey      = "twitch:ds:s3:type"
	defaultTagValue    = "configuration"
	defaultFileSuffix  = ".config.json"
	s3RefreshLogicName = "s3:"
)

type s3RefreshLogic struct {
	service s3iface.S3API
	bucket  string
}

type s3Entry struct {
	name string
	data map[string]interface{}
}

type s3EntrySlice []*s3Entry

func (s s3EntrySlice) Len() int      { return len(s) }
func (s s3EntrySlice) Swap(i, j int) { s[i], s[j] = s[j], s[i] }
func (s s3EntrySlice) Less(i, j int) bool {
	return strings.ToLower(s[i].name) < strings.ToLower(s[j].name)
}

// NewS3RefreshLogic constructs a RefreshLogic that pulls configuration from S3 buckets
func NewS3RefreshLogic(service s3iface.S3API) RefreshLogic {
	return &s3RefreshLogic{service, ""}
}

func (s *s3RefreshLogic) GetName() string { return s3RefreshLogicName }
func (s *s3RefreshLogic) GetValues(settings Settings) (Source, error) {
	if s.service == nil || settings == nil {
		return nil, types.ErrArgumentWasNil
	}

	// construct the path for use inside the bucket
	path, err := s.getPath(settings)
	if err != nil {
		return nil, err
	}

	// find the S3 bucket to use
	bucket, err := s.getBucket()
	if err != nil {
		return nil, err
	}

	// pull out all files under the requested path
	req := new(s3.ListObjectsInput).SetBucket(bucket).SetPrefix(path)
	list, err := s.service.ListObjects(req)
	if err != nil {
		if awsutil.GetCode(err) == s3.ErrCodeNoSuchBucket {
			s.bucket = "" // clear for next refresh
		}
		return nil, err
	}

	// load each file with the correct extension into a map[string]interface{}
	// and sort them for deterministic behavior
	entries := make(s3EntrySlice, 0, len(list.Contents))
	for _, record := range list.Contents {
		if record.Key != nil && strings.HasSuffix(*record.Key, defaultFileSuffix) {
			req := new(s3.GetObjectInput).SetBucket(s.bucket).SetKey(*record.Key)
			obj, err := s.service.GetObject(req)
			if err != nil {
				return nil, err
			}
			content, err := ioutil.ReadAll(obj.Body)
			if err != nil {
				return nil, err
			}
			if err = obj.Body.Close(); err != nil {
				return nil, err
			}
			entry := &s3Entry{*record.Key, make(map[string]interface{})}
			if err := json.Unmarshal(content, &entry.data); err != nil {
				return nil, err
			}
			entries = append(entries, entry)
		}
	}
	sort.Sort(entries)

	// merge the maps and create a source out of them
	values := make([]map[string]interface{}, 0, len(entries))
	for _, entry := range entries {
		values = append(values, entry.data)
	}
	return NewStaticSource(s.GetName()+" "+path, types.Merge(values))
}

func (s *s3RefreshLogic) getBucket() (string, error) {
	if s.bucket != "" {
		return s.bucket, nil
	}

	buckets, err := discovery.BucketsByTag(s.service, defaultTagKey, defaultTagValue)
	if err != nil {
		return "", err
	}

	switch len(buckets) {
	case 0:
		return "", ErrMissingS3ConfigurationBucket
	case 1:
		s.bucket = buckets[0]
	default:
		return "", ErrMultipleS3ConfigurationBuckets
	}
	return s.bucket, nil
}

func (s *s3RefreshLogic) getPath(settings Settings) (string, error) {
	var err error
	if prefix, ok := settings.TryGetString(ConfigKeyS3Prefix); ok {
		return prefix, nil
	}
	var app, env, comp string
	if app, err = loadOrError(settings, ConfigKeyAppName); err != nil {
		return "", err
	}
	if env, err = loadOrError(settings, ConfigKeyAppEnvironment); err != nil {
		return "", err
	}
	if comp, err = loadOrError(settings, ConfigKeyAppComponent); err != nil {
		return "", err
	}
	return fmt.Sprintf("%v/%v/%v/config/", env, app, comp), nil
}

func loadOrError(settings Settings, key string) (string, error) {
	if value, ok := settings.TryGetString(key); ok {
		return value, nil
	}
	return "", types.NewMissingSettingError(key)
}
