package sources

import (
	"code.justin.tv/gds/gds/golibs/config/types"
)

// staticSource contains both a map of raw values for use with GetAll and
// a cache of converted values that maps to the specific calls that have
// been requested, optimizing for frequent queries on a converted type.
type staticSource struct {
	name  string
	raw   map[string]interface{}
	cache *types.ConversionCache
}

// NewStaticSource generates a Source from a map that automatically optimizes
// type conversions if necessary.
func NewStaticSource(name string, values map[string]interface{}) (Source, error) {
	raw, err := types.FlattenAndConvert(values)
	if err != nil {
		return nil, err
	}
	return &staticSource{name, raw, types.NewConversionCache()}, nil
}

func (s *staticSource) GetName() string {
	return s.name
}

func (s *staticSource) GetAll() map[string]interface{} {
	return s.raw
}

func (s *staticSource) GetBool(name string) (bool, bool) {
	if val, ok := s.raw[name]; ok {
		return s.cache.ConvertToBool(name, val)
	}
	return false, false
}

func (s *staticSource) GetFloat(name string) (float64, bool) {
	if val, ok := s.raw[name]; ok {
		return s.cache.ConvertToFloat(name, val)
	}
	return 0, false
}

func (s *staticSource) GetInt(name string) (int64, bool) {
	if val, ok := s.raw[name]; ok {
		return s.cache.ConvertToInt(name, val)
	}
	return 0, false
}

func (s *staticSource) GetString(name string) (string, bool) {
	if val, ok := s.raw[name]; ok {
		return s.cache.ConvertToString(name, val)
	}
	return "", false
}
