package types

import (
	"errors"
	"fmt"
)

// ErrArgumentWasNil is returned by interfaces when a required input is nil
var ErrArgumentWasNil = errors.New("Required argument was nil")

// MissingSettingError is returned when a required value is absent
type MissingSettingError struct {
	key string
}

func (m *MissingSettingError) Error() string {
	return fmt.Sprintf(`Required setting "%v" was not found`, m.key)
}

// NewMissingSettingError generates a MissingSettingError for the indicated key.
func NewMissingSettingError(key string) error {
	return &MissingSettingError{key}
}

// UnsupportedTypeError is returned when attempting to build configuration that
// contains types other than strings, numbers, and booleans
type UnsupportedTypeError struct {
	name      string
	typeValue string
}

func (u *UnsupportedTypeError) Error() string {
	return fmt.Sprintf(`"%v" has unsupported type "%v"`, u.name, u.typeValue)
}

// NewUnsupportedTypeError generates an UnsupportedTypeError for the indicated
// key and value pair.
func NewUnsupportedTypeError(key string, value interface{}) error {
	return &UnsupportedTypeError{key, fmt.Sprintf("%T", value)}
}
