package types

import (
	"fmt"
	"strconv"
)

// https://en.wikipedia.org/wiki/Double-precision_floating-point_format
// https://golang.org/ref/spec#Conversions
const maxConvertableFloat64 = (1 << 53)
const minConvertableFloat64 = -(1 << 53)

type boolValue struct {
	value bool
	found bool
}

func (b *boolValue) Set(raw interface{}) bool {
	switch raw.(type) {
	case bool:
		b.value = raw.(bool)
		b.found = true
	case string:
		if conv, err := strconv.ParseBool(raw.(string)); err == nil {
			b.value = conv
			b.found = true
		} else {
			b.value = false
			b.found = false
		}
	case int64:
		b.value = raw.(int64) != 0
		b.found = true
	case float64:
		b.value = raw.(float64) != 0
		b.found = true
	default:
		b.value = false
		b.found = false
	}
	return b.found
}

type floatValue struct {
	value float64
	found bool
}

func (f *floatValue) Set(raw interface{}) bool {
	switch raw.(type) {
	case float64:
		f.value = raw.(float64)
		f.found = true
	case string:
		if conv, err := strconv.ParseFloat(raw.(string), 64); err == nil {
			f.value = conv
			f.found = true
		} else {
			f.value = 0
			f.found = false
		}
	case bool:
		if raw.(bool) {
			f.value = 1
		} else {
			f.value = 0
		}
		f.found = true
	case int64:
		f.value = float64(raw.(int64))
		f.found = true
	default:
		f.value = 0
		f.found = false
	}
	return f.found
}

type intValue struct {
	value int64
	found bool
}

func (i *intValue) Set(raw interface{}) bool {
	switch raw.(type) {
	case int64:
		i.value = raw.(int64)
		i.found = true
	case string:
		if conv, err := strconv.ParseInt(raw.(string), 10, 64); err == nil {
			i.value = conv
			i.found = true
		} else {
			i.value = 0
			i.found = false
		}
	case bool:
		if raw.(bool) {
			i.value = 1
		} else {
			i.value = 0
		}
		i.found = true
	case float64:
		f := raw.(float64)
		if f >= minConvertableFloat64 && f <= maxConvertableFloat64 {
			i.value = int64(raw.(float64))
			i.found = true
		} else {
			i.value = 0
			i.found = false
		}
	default:
		i.value = 0
		i.found = false
	}
	return i.found
}

type stringValue struct {
	value string
	found bool
}

func (s *stringValue) Set(raw interface{}) bool {
	switch raw.(type) {
	case string:
		s.value = raw.(string)
		s.found = true
	case bool:
		s.value = strconv.FormatBool(raw.(bool))
		s.found = true
	case int64:
		s.value = strconv.FormatInt(raw.(int64), 10)
		s.found = true
	case float64:
		s.value = strconv.FormatFloat(raw.(float64), 'f', -1, 64)
		s.found = true
	default:
		s.value = ""
		s.found = false
	}
	return s.found
}

func convertToSupported(name string, raw interface{}) (interface{}, *UnsupportedTypeError) {
	switch raw.(type) {
	case string:
		return raw, nil
	case int64:
		return raw, nil
	case bool:
		return raw, nil
	case float64:
		return raw, nil
	case int:
		return int64(raw.(int)), nil
	case int8:
		return int64(raw.(int8)), nil
	case int16:
		return int64(raw.(int16)), nil
	case int32:
		return int64(raw.(int32)), nil
	case uint8:
		return int64(raw.(uint8)), nil
	case uint16:
		return int64(raw.(uint16)), nil
	case uint32:
		return int64(raw.(uint32)), nil
	case float32:
		return float64(raw.(float32)), nil
	}
	return nil, &UnsupportedTypeError{name, fmt.Sprintf("%T", raw)}
}

// Merge combines a set of maps into a single one, with each map overwriting the
// content from earlier in the slice
func Merge(in []map[string]interface{}) map[string]interface{} {
	out := make(map[string]interface{})
	for _, m := range in {
		for k, v := range m {
			out[k] = v
		}
	}
	return out
}

// FlattenAndConvert turns a tree of settings into a flat list with dotted names
// and converts each value into its supported equivalent type
func FlattenAndConvert(in map[string]interface{}) (map[string]interface{}, *UnsupportedTypeError) {
	out := make(map[string]interface{})
	if err := flattenAndConvertR("", in, out); err != nil {
		return nil, err
	}
	return out, nil
}

func flattenAndConvertR(prefix string, in, out map[string]interface{}) *UnsupportedTypeError {
	for k, v := range in {
		name := prefix + k
		if section, ok := v.(map[string]interface{}); ok {
			if err := flattenAndConvertR(name+".", section, out); err != nil {
				return err
			}
			continue
		}
		conv, err := convertToSupported(name, v)
		if err != nil {
			return err
		}
		out[name] = conv
	}
	return nil
}
