package discovery

import (
	"fmt"

	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3iface"
)

// BucketsByTag returns the list of S3 bucket names
// that contain the given tag
func BucketsByTag(svc s3iface.S3API, key, value string) ([]string, error) {
	resp, err := svc.ListBuckets(&s3.ListBucketsInput{})
	if err != nil {
		return nil, err
	}

	foundBuckets := []string{}

	for _, bucket := range resp.Buckets {
		tagParams := &s3.GetBucketTaggingInput{
			Bucket: bucket.Name,
		}
		tagResp, err := svc.GetBucketTagging(tagParams)
		if err != nil {
			if awsErr, ok := err.(awserr.Error); ok {
				if awsErr.Code() == "NoSuchTagSet" {
					continue
				}
				if awsErr.Code() == "BucketRegionError" {
					continue
				}
			}
			return nil, fmt.Errorf("Error retreiving tags for s3 bucket %s: %v", *bucket.Name, err)
		}
		for _, tag := range tagResp.TagSet {
			if *tag.Key == key && *tag.Value == value {
				foundBuckets = append(foundBuckets, *bucket.Name)
			}
		}
	}

	return foundBuckets, nil
}
