#!/bin/bash

function root__infra() {
  if [ "${1-}" == "help" ]; then
    echo "Runs infrastructure related commands"
    exit 0
  fi
  if [ "${USE_BUILDER}" == "true" ]; then
    make_exec builder run infra $@
  fi
  process_build infra__ $@
}

function print_valid_infra() {
  find ./cloudformation/infra -type f -name '*.json' | sed "s/.*\///" | sed "s#\..*##"
}

function print_valid_infra_deploy() {
  find ./cloudformation/deploy -type f -name '*.json' | sed "s/.*\///" | sed "s#\..*##"
}

function infra__pipeline() {
  if [ "${1-}" == "help" ]; then
    echo "Update pipeline infrastructure"
    exit 0
  fi
  CFEXECUTE_OPTIONS=""
  while getopts ":a" opt; do
    case ${opt} in
      a)
        CFEXECUTE_OPTIONS="${CFEXECUTE_OPTIONS} -a"
        ;;
      \?)
        echo "Invalid option: -$OPTARG" >&2
        return 1
        ;;
    esac
  done
  shift "$((OPTIND-1))"
  infra_assume_bootstrap_role pipeline

  make_run extra cloudformation ${CFEXECUTE_OPTIONS} ./cloudformation/service/service-pipeline.json
}

function infra__update() {
  # TODO: Add -a support here.
  if [ "${1-}" == "help" ]; then
    echo "Update infrastructure"
    exit 0
  fi
  CFEXECUTE_OPTIONS=""
  while getopts ":a" opt; do
    case ${opt} in
      a)
        CFEXECUTE_OPTIONS="${CFEXECUTE_OPTIONS} -a"
        ;;
      \?)
        echo "Invalid option: -$OPTARG" >&2
        return 1
        ;;
    esac
  done
  shift "$((OPTIND-1))"

  INFRA="${1-}"
  if [ -z ${INFRA} ]; then
    echo "Pick an infra.  Valid infra to update: "
    echo
    print_valid_infra
    exit 1
  fi
  infra_assume_bootstrap_role ${INFRA}
  make_run extra cloudformation ${CFEXECUTE_OPTIONS} ./cloudformation/infra/${INFRA}.json
}

function infra__docker_push() {
  if [ "${1-}" == "help" ]; then
    echo "Push a docker container to an infrastructure"
    exit 0
  fi
  INFRA="${1-}"
  if [ -z ${INFRA} ]; then
    echo "Pick an infra.  Valid infra to docker push: "
    echo
    print_valid_infra
    exit 1
  fi
  # TODO: Show valid infra to update
  infra_assume_infra_role ${INFRA}
  ecr_push ${INFRA} app-
}

function infra__docker_integration_push() {
  if [ "${1-}" == "help" ]; then
    echo "Push an integration docker container to an infrastructure"
    exit 0
  fi
  INFRA="${1-}"
  if [ -z ${INFRA} ]; then
    echo "Pick an infra.  Valid infra to docker push: "
    echo
    print_valid_infra
    exit 1
  fi
  # TODO: Show valid infra to update
  infra_assume_infra_role ${INFRA}
  ecr_push ${INFRA} integration-
}

function infra__output() {
  if [ "${1-}" == "help" ]; then
    echo "Print a CloudFormation output inside the infrastructure"
    exit 0
  fi
  INFRA="${1-}"
  if [ -z ${INFRA} ]; then
    echo "Pick an infra.  Valid infra to print from"
    echo
    print_valid_infra
    exit 1
  fi
  shift
  infra_assume_infra_role ${INFRA}
  VAL="${1-}"
  if [ -z ${VAL} ]; then
    echo "Pick a value to output"
    echo
    aws cloudformation list-exports --no-paginate --query "Exports[*].[Name]" --output text
    exit 1
  fi
  shift
  get_cloudformation_output ${VAL}
}

function infra__deploy() {
  if [ "${1-}" == "help" ]; then
    echo "Deploy a service into an infrastructure"
    exit 0
  fi
  INFRA="${1-}"
  if [ -z ${INFRA} ]; then
    echo "Pick an infra.  Valid infra to deploy: "
    echo
    print_valid_infra_deploy
    exit 1
  fi
  infra_assume_infra_role ${INFRA}
  make_run extra cloudformation -a ./cloudformation/deploy/${INFRA}.json
}

function infra__integration_test_ecs_run() {
  if [ "${1-}" == "help" ]; then
    echo "Run the integration tests inside ECS remotely"
    exit 0
  fi
  INFRA="${1-}"
  if [ -z ${INFRA} ]; then
    echo "Pick an infra.  Valid infra to docker ecs run: "
    echo
    print_valid_infra
    exit 1
  fi
  infra_assume_infra_role ${INFRA}
  populate_app_env
  ecsrun -prefix cmd/${SERVICE}/integration
}

function infra__guess_info() {
  if [ "${1-}" == "help" ]; then
    echo "Guess information about your dev and production account"
    exit 0
  fi
  INFRA="${1-}"
  if [ -z ${INFRA} ]; then
    echo "Pick an infra.  Valid infra to guess about"
    echo
    print_valid_infra
    exit 1
  fi
  infra_assume_bootstrap_role ${INFRA}

  VPC_ID=$(guess_vpc_id)
  echo "AWS region"
  echo "  ${AWS_REGION}"
  echo "Account ID"
  echo "  $(guess_aws_account_id)"
  echo "VPC id"
  echo "  ${VPC_ID}"
  echo "Security group"
  echo "  $(guess_security_group_id)"
  echo "Private Subnets"
  echo "  $(guess_private_subnet_id ${VPC_ID})"

}

# Find the current account ID in AWS_PROFILE
function guess_aws_account_id() {
  aws sts get-caller-identity --query 'Account' --output text
}

function guess_vpc_id() {
	aws ec2 describe-vpcs --filters "Name=isDefault,Values=false" --query 'Vpcs[].VpcId' --output text
}

function guess_security_group_id() {
  aws ec2 describe-security-groups --query "SecurityGroups[?GroupName=='twitch_subnets'].GroupId" --output text
}

function guess_private_subnet_id() {
  VPC_ID=$1
  aws ec2 describe-subnets --query "Subnets[? ! MapPublicIpOnLaunch] | [?VpcId =='$VPC_ID'].SubnetId | join(',', @)" --output text
}