package servicerunner

import (
	"errors"
	"io"
	"os"
	"os/signal"
	"reflect"
	"strings"
	"sync"
	"syscall"
)

// Logger is an interface for log messages, similar to the log package logger
type Logger interface {
	Log(...interface{})
}

// Service is any struct that should be started
type Service interface {
	Start() error
}

// Setupable is an optional Service interface that is called before Start()
type Setupable interface {
	Setup() error
}

// ServiceRunner controls starting and running services until a SIGTERM
type ServiceRunner struct {
	Services     []Service
	Closers      []io.Closer
	Log          Logger
	SigChan      chan os.Signal
	SignalNotify func(c chan<- os.Signal, sig ...os.Signal)
	onClose      chan struct{}
}

// Append a list of services to the service runner
func (r *ServiceRunner) Append(services ...Service) {
	r.Services = append(r.Services, services...)
}

func (r *ServiceRunner) log(vals ...interface{}) {
	if r.Log != nil {
		r.Log.Log(vals...)
	}
}

func (r *ServiceRunner) runServices(wg *sync.WaitGroup) {
	wg.Add(len(r.Services))
	for _, s := range r.Services {
		go func(s Service) {
			defer func() {
				defer wg.Done()
				if rec := recover(); rec != nil {
					// Just absorb panic and log it.  Use errors.New() to get the stack trace
					err := errors.New("service panic")
					r.log("rec", rec, "service", s, "err", err)
				}
			}()
			r.log("service", reflect.TypeOf(s).String(), "Starting service")
			err := s.Start()
			r.log("err", err, "Service finished")
		}(s)
	}
}

// Close would be the same as sending a signal, ending the runner early
func (r *ServiceRunner) Close() error {
	close(r.onClose)
	return nil
}

// Execute starts the service runner.  Runs until SIGTERM or SIGINT
func (r *ServiceRunner) Execute() error {
	return r.exec(false, nil)
}

// ExecuteInBackground starts the services in the background.  The function blocks for Setup steps and returns
// any setup errors.  Then, it schedules the services to all run in the background.  Background running services
// call `closeErrorHandler` on any errors they see during execution and Close.
func (r *ServiceRunner) ExecuteInBackground(closeErrorHandler func(error)) error {
	return r.exec(true, closeErrorHandler)
}

func (r *ServiceRunner) exec(isAsync bool, closeErrorHandler func(error)) error {
	r.onClose = make(chan struct{})
	r.log("service_count", len(r.Services), "Starting service execution")
	defer r.log("Service execution finished")
	if r.SigChan == nil {
		r.SigChan = make(chan os.Signal)
	}
	sigNotify := r.SignalNotify
	if sigNotify == nil {
		sigNotify = signal.Notify
	}
	for _, s := range r.Services {
		if setter, ok := s.(Setupable); ok {
			if err := setter.Setup(); err != nil {
				return err
			}
		}
	}
	var wg sync.WaitGroup
	r.runServices(&wg)

	r.log("Waiting on signal channel")
	executionFunction := func() {
		sigNotify(r.SigChan, syscall.SIGTERM, syscall.SIGINT)
		select {
		case <-r.onClose:
		case sig := <-r.SigChan:
			r.log("sig", sig, "Received signal")
		}
		errs := make([]error, 0, len(r.Services))
		for i := len(r.Services) - 1; i >= 0; i-- {
			if closer, ok := r.Services[i].(io.Closer); ok {
				err := closer.Close()
				if err != nil {
					r.log("err", err, "service", r.Services[i], "Unable to close service")
					errs = append(errs, err)
				}
			}
		}
		for i, c := range r.Closers {
			if err := c.Close(); err != nil {
				r.log("err", err, "index", i, "Unable to close explicit closer")
				errs = append(errs, err)
			}
		}
		r.log("Waiting on all services to finish")
		wg.Wait()
		r.log("Waiting done")
		if closeErrorHandler != nil {
			closeErrorHandler(consolidateErrors(errs))
		}
	}
	if isAsync {
		go executionFunction()
		return nil
	}
	var errorToReturn error
	closeErrorHandler = func(err error) {
		errorToReturn = err
	}
	executionFunction()
	return errorToReturn
}

// MultiErr is a struct that implements the error interface for consolidating multiple errors
type multiError struct {
	errs []error
}

func (e *multiError) Error() string {
	r := make([]string, 0, len(e.errs))
	for _, err := range e.errs {
		r = append(r, err.Error())
	}
	return strings.Join(r, " | ")
}

var _ error = &multiError{}

func consolidateErrors(errs []error) error {
	retErrs := make([]error, 0, len(errs))
	for _, err := range errs {
		if err != nil {
			if multiErr, ok := err.(*multiError); ok {
				retErrs = append(retErrs, multiErr.errs...)
			} else {
				retErrs = append(retErrs, err)
			}
		}
	}
	if len(retErrs) == 0 {
		return nil
	}
	if len(retErrs) == 1 {
		return retErrs[0]
	}
	return &multiError{
		errs: retErrs,
	}
}
