package twitchstatsd

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"os"
	"strings"
	"time"

	"code.justin.tv/hygienic/distconf"
	"code.justin.tv/hygienic/log"
	"github.com/cactus/go-statsd-client/statsd"
)

// Config configures Service
type Config struct {
	FlushInterval *distconf.Duration
	FlushBytes    *distconf.Int
	StatsdHost    *distconf.Str
}

// Load config values from distconf
func (c *Config) Load(d *distconf.Distconf) error {
	c.StatsdHost = d.Str("statsd.region_host", "statsd.internal.justin.tv:8125")
	c.FlushBytes = d.Int("statsd.flush_bytes", 512)
	c.FlushInterval = d.Duration("statsd.flush_interval", time.Second)
	return nil
}

// Service configures a statsd client
type Service struct {
	Team        string
	Service     string
	Environment string
	Config      Config
	Log         log.Logger
	Statsd      statsd.Statter
	onClose     chan struct{}
}

// Setup a statsd service
func (s *Service) Setup() error {
	if s.onClose == nil {
		s.onClose = make(chan struct{})
	}
	if s.Statsd != nil {
		return nil
	}
	if s.Team == "" {
		return errors.New("expected twitchstatsd.Team set to non empty")
	}
	if s.Service == "" {
		return errors.New("expected twitchstatsd.Service set to non empty")
	}
	if s.Environment == "" {
		s.Environment = os.Getenv("ENVIRONMENT")
	}
	if s.Environment == "" {
		return errors.New("unable to detect twitchstatsd.ENVIRONMENT")
	}

	regionStatsdHost := s.Config.StatsdHost.Get()
	if regionStatsdHost == "" {
		s.Log.Log("No statsd hostport setup.  Not sending metrics")
		noopClient, err := statsd.NewNoopClient()
		if err != nil {
			return err
		}
		s.Statsd = noopClient
		return nil
	}

	regionPrefix := fmt.Sprintf("%s.%s.%s.%s", sanitizeStatsd(s.Team+"-region"), sanitizeStatsd(s.Service), sanitizeStatsd(s.Environment), s.region())

	bufferedStats, err := statsd.NewBufferedClient(regionStatsdHost, regionPrefix, s.Config.FlushInterval.Get(), int(s.Config.FlushBytes.Get()))
	if err != nil {
		return err
	}
	s.Statsd = bufferedStats
	return nil
}

// Start waits for close
func (s *Service) Start() error {
	<-s.onClose
	return nil
}

// Close ends the statsd client
func (s *Service) Close() error {
	close(s.onClose)
	return s.Statsd.Close()
}

func (s *Service) region() string {
	region := s.getDocumentInfo("region")
	if region != "" {
		return region
	}
	return "noregion"
}

func (s *Service) log() log.Logger {
	if s.Log == nil {
		return log.Discard
	}
	return s.Log
}

func (s *Service) getDocumentInfo(key string) string {
	req, err := http.NewRequest("GET", "http://169.254.169.254/latest/dynamic/instance-identity/document", nil)
	if err != nil {
		s.log().Log("err", err, "creating request to query instance metadata failed")
		return ""
	}

	ctx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()
	req = req.WithContext(ctx)

	client := http.DefaultClient
	resp, err := client.Do(req)
	if err != nil {
		s.log().Log("err", err, "sending request to query instance metadata failed")
		return ""
	}
	if resp.StatusCode != http.StatusOK {
		s.log().Log("status", resp.StatusCode, "request to query instance metadata failed")
		return ""
	}

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		s.log().Log("err", err, "reading instance metadata query response failed")
		return ""
	}

	err = resp.Body.Close()
	if err != nil {
		s.log().Log("err", err, "closing http body failed")
	}

	var decodeInto map[string]string
	if err := json.Unmarshal(body, &decodeInto); err != nil {
		s.log().Log("err", err, "Decoding document failed")
	}
	return decodeInto[key]
}

func sanitizeStatsd(s string) string {
	if len(s) > 32 {
		s = s[0:32]
	}
	return strings.Replace(s, ".", "_", -1)
}
