package events

import (
	"bytes"
	"errors"
	"io/ioutil"
	"time"

	"code.justin.tv/sse/malachai/pkg/internal/closer"
	"code.justin.tv/sse/malachai/pkg/log"
)

// ErrBufferMaxSizeExceeded is returned if attempting to write more to the
// buffer than is allowed
var ErrBufferMaxSizeExceeded = errors.New("event buffer size limit reached")

// memoryEventBuffer buffers events in memory
type memoryEventBuffer struct {
	bufferLimit int64
	buffer      bytes.Buffer
	closer      *closer.Closer
	logger      log.S2SLogger
}

// Write appends the event data to the buffer
func (meb *memoryEventBuffer) Write(data []byte) (n int, err error) {
	if meb.bufferLimit > 0 && int64(len(data)+meb.buffer.Len()) > meb.bufferLimit {
		err = ErrBufferMaxSizeExceeded
		return
	}

	n, err = meb.buffer.Write(data)
	if err != nil {
		return
	}
	return
}

func (meb *memoryEventBuffer) flush(reportInterval time.Duration, batcher FirehoseBatchWriter) (err error) {
	data, err := ioutil.ReadAll(&meb.buffer)
	if err != nil {
		err = errors.New("memoryEventBuffer: error reading bytes from buffer: " + err.Error())
		return
	}
	err = batcher.WriteToFirehose(meb.closer, data)
	if err != nil {
		err = errors.New("memoryEventBuffer: error writing data to firehose: " + err.Error())
		return
	}
	return
}

// Run starts the event buffer
func (meb *memoryEventBuffer) Run(reportInterval time.Duration, batcher FirehoseBatchWriter) {
	for {
		select {
		case <-meb.closer.Done():
			meb.logger.Debug("memoryEventBuffer: exiting due to stop signal")
			err := meb.flush(reportInterval, batcher)
			if err != nil {
				meb.logger.Error(err.Error())
			}
			meb.buffer.Reset()
			return
		case <-time.NewTimer(reportInterval).C:
			err := meb.flush(reportInterval, batcher)
			if err != nil {
				meb.logger.Error(err.Error())
			}
		}
	}
}
