import * as React from 'react';

import { object } from 'prop-types';

import { ApiClientContext } from 'brickie/core/api-provider';
import { FeedResponse } from 'brickie/models/feed';
import { FeedComponent } from './component';

interface FeedProps {
  userID: string;
}

interface FeedState {
  loading: boolean;
  feed: FeedResponse | null;
}

export class FeedContainer extends React.Component<FeedProps, FeedState> {
  public static contextTypes = {
    feedsEdge: object,
  };

  public context: ApiClientContext;

  constructor(props: FeedProps) {
    super(props);

    this.state = {
      loading: false,
      feed: null,
    };
  }

  public componentWillMount() {
    this.fetchFeed(this.props.userID);
  }

  public componentWillReceiveProps(nextProps: FeedProps) {
    this.fetchFeed(nextProps.userID);
  }

  public render() {
    if (this.state.loading) {
      return (
        <div>Loading...</div>
      );
    }

    return (
      <div>
        <FeedComponent userID={this.props.userID} feed={this.state.feed} feedID={'n:' + this.props.userID} onChange={this.fetchFeed.bind(this, this.props.userID)} />
      </div>
    );
  }

  private fetchFeed(userID: string) {
    if (userID === '') {
      this.setState({
        feed: null,
        loading: false,
      });
      return;
    }
    this.setState({
      loading: true,
    });

    let feedsClient = this.context.feedsEdge;
    return feedsClient.getFeed(`n:${userID}`)
    .then((feed: FeedResponse) => {
      this.setState({
        feed,
        loading: false,
      });
    }).catch((err) => {
      console.error(err);
      this.setState({
        loading: false,
      });
    });
  }
}
