import * as React from 'react';

import { object } from 'prop-types';

import { ApiClientContext } from 'brickie/core/api-provider';
import { User } from 'brickie/models/users';

interface UserSelectorProps {
  onUserID: (userID: string) => void;
}

interface UserSelectorState {
  userID: string;
  login: string;
}

export class UserSelector extends React.Component<UserSelectorProps, UserSelectorState> {
  public static contextTypes = {
    users: object,
  };

  public context: ApiClientContext;
  private value: string;

  constructor(props: UserSelectorProps) {
    super(props);
    this.value = '';
    this.state = {
      userID: '',
      login: '',
    };
  }

  public render() {
    let bottom: {};
    if (this.state.userID !== '') {
      bottom = (
        <div>
          <div>Selected User ID: {this.state.userID}</div>
          <div>User login: {this.state.login}</div>
        </div>
      );
    } else {
      bottom = <div>No user currently selected</div>;
    }
    return (
      <div>
        {bottom}
        <div>
          <span>Enter a user ID or name:</span>
          <input
            onChange={this.onChange}
            defaultValue={this.value}
            onKeyPress={this.onPress}
            type="text"
          />
          <button title="update user id" onClick={this.fetchUserInfo}>
            Update
          </button>
        </div>
      </div>
    );
  }

  public componentWillMount() {
    this.fetchUserInfo();
  }

  private onChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    this.value = e.target.value;
  }

  private onPress = (e: React.KeyboardEvent<HTMLInputElement>) => {
    if (e.key === 'Enter') {
      this.fetchUserInfo();
    }
  }

  private fetchUserFromID(id: string) {
    let usersClient = this.context.users;

    usersClient.getUserFromID(id)
    .then((user: User) => {
      this.setState(
        {
          login: user.name,
          userID: user._id,
        },
      );
      this.props.onUserID(user._id);
    });
  }

  private clearState() {
    this.setState(
      {
        login: '',
        userID: '',
      },
    );
    this.props.onUserID('');
  }

  private fetchUserFromLogin(login: string) {
    let usersClient = this.context.users;

    usersClient.getUserFromLogin(login)
    .then((user) => {
      if (user !== null) {
        this.setState(
          {
            login: user.name,
            userID: user._id,
          },
        );
        this.props.onUserID(user._id);
      } else {
        this.clearState();
      }
    });
  }

  private fetchUserInfo = () => {
    if (this.value === null || this.value === '') {
      this.clearState();
      return;
    }
    if (parseInt(this.value, 10).toString() === this.value) {
      this.fetchUserFromID(this.value);
    } else {
      this.fetchUserFromLogin(this.value);
    }
  }
}
