const path = require('path');

const webpack = require('webpack');
const { CheckerPlugin } = require('awesome-typescript-loader');
const HtmlWebpackPlugin = require('html-webpack-plugin');
const ExtractTextPlugin = require('extract-text-webpack-plugin');

module.exports = function (env = {}) {

  let plugins = [
    new CheckerPlugin(),
    new HtmlWebpackPlugin({
      title: "Brickie",
    }),
  ];

  let moduleRules = [
    {
      test: /\.(ico|png|jpg|gif|svg|eot|ttf|woff|woff2)$/,
      loader: 'file-loader',
      options: {
        name: 'assets/[name].[hash].[ext]'
      }
    },
    {
      test: /\.tsx?$/,
      loader: "awesome-typescript-loader",
      exclude: /node_modules/,
    }
  ];

  let sassLoaders = [
    {
      loader: 'css-loader',
      options: {
        minimize: env.optimized,
        sourceMap: true,
      }
    },
    {
      loader: 'sass-loader',
      options: {
        sourceMap: true,
      }
    }
  ];

  if (env.optimized) {
    // Compressed build
    plugins = plugins.concat([
      new webpack.DefinePlugin({
        'process.env': {
          NODE_ENV: JSON.stringify('production'),
        }
      }),
      new webpack.optimize.UglifyJsPlugin({
        include: /.*\.js/,
        sourceMap: true,
        compress: {
          warnings: false,
        },
        comments: false,
      }),
      new ExtractTextPlugin({
        filename: env.devserver ? 'assets/[name].css' : 'assets/[name].[contenthash].css',
        allChunks: true,
      }),
    ]);

    moduleRules = moduleRules.concat([
      {
        test: /\.sass$/,
        use: ExtractTextPlugin.extract({
          fallback: 'style-loader',
          use: sassLoaders,
        })
      }
    ]);
  } else {
    moduleRules = moduleRules.concat([
      {
        test: /\.sass$/,
        use: [{ loader: 'style-loader' }].concat(sassLoaders),
      }
    ]);
  }

  let environment = env.environment || "development";

  return {
    entry: `./src/entry/${environment}.ts`,
    output: {
      filename: env.devserver ? 'assets/[name].js' : 'assets/[name].[chunkhash].js',
      path: __dirname + "/dist",
      publicPath: "/",
    },
    devServer: {
      host: "localhost.twitch.tv",
    },
    resolve: {
      extensions: [".ts", ".tsx", ".js", ".json"],
      alias: {
        'brickie': path.join(__dirname, 'src'),
      },
    },
    devtool: 'source-map',
    module:  {
      rules: moduleRules
    },
    plugins
  }
}
