package edge

import (
	"fmt"
	"net/http"
	"net/url"
	"strings"
	"time"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/clients/twitchdoer"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/foundation/twitchclient"
	"github.com/google/go-querystring/query"
	"golang.org/x/net/context"
)

const (
	defaultTimingXactName = "feeds-edge"
	defaultStatSampleRate = 1.0
)

// Client is the interface third parties should expect the edge client to implement
type Client interface {
	GetFeed(ctx context.Context, feed entity.Entity, options *GetFeedOptions, reqOpts *twitchclient.ReqOpts) (*Feed, error)

	CreatePost(ctx context.Context, channelID string, body string, userID string, options *CreatePostOptions, reqOpts *twitchclient.ReqOpts) (*Post, error)
	CreateSyndicatedPost(ctx context.Context, channelID string, body string, userID string, postToTwitter bool, options *CreatePostOptions, reqOpts *twitchclient.ReqOpts) (*SyndicatedPost, error)
	GetPost(ctx context.Context, postID string, options *GetPostOptions, reqOpts *twitchclient.ReqOpts) (*Post, error)
	DeletePost(ctx context.Context, postID, userID string, reqOpts *twitchclient.ReqOpts) (*Post, error)
	ReportPost(ctx context.Context, postID, userID, reason, clientID string, reqOpts *twitchclient.ReqOpts) error

	CreateComment(ctx context.Context, parent entity.Entity, body string, userID string, reqOpts *twitchclient.ReqOpts) (*Comment, error)
	GetCommentByID(ctx context.Context, commentID string, options *GetCommentByIDOptions, reqOpts *twitchclient.ReqOpts) (*Comment, error)
	GetCommentsForPost(ctx context.Context, postID string, options *GetCommentsForPostOptions, reqOpts *twitchclient.ReqOpts) (*Comments, error)
	ApproveComment(ctx context.Context, commentID string, userID string, reqOpts *twitchclient.ReqOpts) (*Comment, error)
	DenyComment(ctx context.Context, commentID string, userID string, reqOpts *twitchclient.ReqOpts) error
	DeleteComment(ctx context.Context, commentID, userID string, reqOpts *twitchclient.ReqOpts) (*Comment, error)
	DeleteCommentsByParentAndUser(ctx context.Context, parent entity.Entity, commentUserID string, userID string, reqOpts *twitchclient.ReqOpts) error
	ReportComment(ctx context.Context, commentID, userID, reason string, reqOpts *twitchclient.ReqOpts) error

	GetEmbed(ctx context.Context, embedURL string, options *GetEmbedOptions, reqOpts *twitchclient.ReqOpts) (*Embed, error)

	CreateReaction(ctx context.Context, parent entity.Entity, emoteID, userID string, reqOpts *twitchclient.ReqOpts) (*UserReactions, error)
	GetReactionsForParents(ctx context.Context, parentEntities []entity.Entity, options *GetReactionsForParentsOptions, reqOpts *twitchclient.ReqOpts) (*BatchReactions, error)
	DeleteReaction(ctx context.Context, parent entity.Entity, emoteID, userID string, reqOpts *twitchclient.ReqOpts) (*UserReactions, error)

	GetSettings(ctx context.Context, e entity.Entity, userID string, reqOpts *twitchclient.ReqOpts) (*Settings, error)
	UpdateSettings(ctx context.Context, e entity.Entity, userID string, options *UpdateSettingsOptions, reqOpts *twitchclient.ReqOpts) (*Settings, error)

	CreateShare(ctx context.Context, userID string, targetEntity entity.Entity, reqOpts *twitchclient.ReqOpts) (*Share, error)
	GetShare(ctx context.Context, userID string, shareID string, reqOpts *twitchclient.ReqOpts) (*Share, error)
	GetShares(ctx context.Context, userID string, shareIDs []string, reqOpts *twitchclient.ReqOpts) (*Shares, error)
	DeleteShare(ctx context.Context, userID string, shareID string, reqOpts *twitchclient.ReqOpts) (*Share, error)
	GetSharesSummaries(ctx context.Context, userID string, parentEntities []entity.Entity, authorIDs []string, reqOpts *twitchclient.ReqOpts) (*SharesSummaries, error)
	UnshareByTarget(ctx context.Context, userID string, authorID string, targetEntities []entity.Entity, reqOpts *twitchclient.ReqOpts) (*Shares, error)

	GetPostPermissions(ctx context.Context, userID string, feedID string, reqOpts *twitchclient.ReqOpts) (*PostPermissions, error)
}

// ClientImpl implements the Client interface and uses the twitchclient client to make http requests
type ClientImpl struct {
	Client twitchclient.Client
}

var _ Client = &ClientImpl{}

// NewClient returns a new instance of the Client which uses the given config
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	return &ClientImpl{Client: twitchClient}, err
}

// Feed contains a single page of Posts with a Cursor that points to the next page
type Feed struct {
	Disabled bool      `json:"disabled,omitempty"`
	Cursor   string    `json:"cursor"`
	Posts    []*Post   `json:"posts"`
	Injects  []*Inject `json:"injects,omitempty"`
}

// Post is the user facing representation of post
type Post struct {
	ID           string                     `json:"id"`
	CreatedAt    time.Time                  `json:"created_at"`
	Deleted      bool                       `json:"deleted"`
	UserID       string                     `json:"user_id"`
	Body         string                     `json:"body"`
	Permissions  *PostPermissions           `json:"permissions,omitempty"`
	Emotes       []Emote                    `json:"emotes"`
	Embeds       []Embed                    `json:"embeds"`
	Reactions    map[string]*ReactionToItem `json:"reactions"`
	Comments     *Comments                  `json:"comments,omitempty"`
	ShareSummary ShareSummary               `json:"share_summary"`
	Reasons      []*ReasonSuperStruct       `json:"reasons,omitempty"`
	Tracking     *PostTracking              `json:"tracking,omitempty"`
	Type         string                     `json:"type,omitempty"`
}

// PostTracking contains post-level fields used for tracking
type PostTracking struct {
	RecGenerationID    string `json:"rec_generation_id,omitempty"`
	RecGenerationIndex *int   `json:"rec_generation_index,omitempty"`
	CardImpressionID   string `json:"card_impression_id,omitempty"`
	BatchID            string `json:"batch_id,omitempty"`
}

// ReasonTypeEnum is the set of reason types we expose for feed items
type ReasonTypeEnum string

const (
	// ShareCreatedType is the Post.Reason.Type for created shares
	ShareCreatedType ReasonTypeEnum = "share_created"
	// PostCreatedType is the Post.Reason.Type for created posts
	PostCreatedType ReasonTypeEnum = "post_created"
)

// ReasonSuperStruct should be every reason that can happen.  Honestly, this is the easiest way to do this in Go
type ReasonSuperStruct struct {
	Type      ReasonTypeEnum `json:"type"`
	UserID    string         `json:"user_id,omitempty"`
	CreatedAt *time.Time     `json:"created_at,omitempty"`
}

// ShareSummary allows summary information about a specific item, such as who shared it and the share count
type ShareSummary struct {
	UserIDs    []string `json:"user_ids"`
	ShareCount int      `json:"share_count"`
}

// SharesSummaries is a JSON payload for multiple share summaries
type SharesSummaries struct {
	Items []*ShareSummary
}

// Share is the returned object for a database share
type Share struct {
	ID           string        `json:"id"`
	UserID       string        `json:"user_id"`
	TargetEntity entity.Entity `json:"target_entity"`
	CreatedAt    time.Time     `json:"created_at"`
	DeletedAt    *time.Time    `json:"deleted_at,omitempty"`
}

// Shares is the JSON payload for multiple share objects
type Shares struct {
	Items []*Share
}

// SyndicatedPost is the user facing representation of post that also been posted to Twitter
type SyndicatedPost struct {
	Post        *Post  `json:"post,omitempty"`
	TweetStatus int    `json:"tweet_status"`
	Tweet       string `json:"tweet"`
}

// Comments is a paginated list of comments
type Comments struct {
	Total    int        `json:"total"`
	Cursor   string     `json:"cursor"`
	Comments []*Comment `json:"comments"`
}

// Comment is the user facing representation of a comment
type Comment struct {
	ID            string                     `json:"id"`
	CreatedAt     time.Time                  `json:"created_at"`
	UserID        string                     `json:"user_id"`
	Body          string                     `json:"body"`
	Emotes        []Emote                    `json:"emotes"`
	Permissions   *CommentPermissions        `json:"permissions"`
	Reactions     map[string]*ReactionToItem `json:"reactions"`
	ParentEntity  entity.Entity              `json:"parent_entity"`
	Deleted       bool                       `json:"deleted"`
	NeedsApproval bool                       `json:"needs_approval"`
}

// Entity represents a parent entity
type Entity struct {
	Type string `json:"type"`
	ID   string `json:"id"`
}

func (e *Entity) toString() string {
	return e.Type + ":" + e.ID
}

// CommentPermissions contains a user's permissions to a comment
type CommentPermissions struct {
	CanDelete bool `json:"can_delete"`
}

// BatchReactions contains a slice of reactions and their parent entities.
type BatchReactions struct {
	Reactions []*Reactions `json:"items"`
}

// Reactions contains a parent entity and its corresponding reactions.
type Reactions struct {
	ParentEntity entity.Entity              `json:"parent_entity"`
	Reactions    map[string]*ReactionToItem `json:"reactions"`
}

// ReactionToItem contains info (counts, user IDs) about a single emote reaction to an item
type ReactionToItem struct {
	Emote   string   `json:"emote"`
	Count   int      `json:"count"`
	UserIDs []string `json:"user_ids"`
}

// UserReactions contains the emote IDs that a user has created on a post or comment.
type UserReactions struct {
	UserID   string   `json:"user_id"`
	EmoteIDs []string `json:"emote_ids"`
}

// PostPermissions contains a user's permissions to a post
type PostPermissions struct {
	CanReply       bool           `json:"can_reply"`
	CanDelete      bool           `json:"can_delete"`
	CanModerate    bool           `json:"can_moderate"`
	CanShare       bool           `json:"can_share"`
	CanReplyReason CanReplyReason `json:"can_reply_reason"`
}

// CanReplyReason is the set of reasons a user might not be able to comment on a post
type CanReplyReason string

const (
	// ReasonMustFollow means that a user must follow the author in order to reply to their posts
	ReasonMustFollow CanReplyReason = "MustFollow"
	// ReasonMustFriend means that a user must be friends with the author in order to reply to their posts
	ReasonMustFriend CanReplyReason = "MustFriend"
	// ReasonMustSub means that a user must subscribe to the author in order to reply to their posts
	ReasonMustSub CanReplyReason = "MustSub"
	// ReasonBanned means that a user must be unbanned order to reply to posts
	ReasonBanned CanReplyReason = "Banned"
	// ReasonDisabled means that comments are disabled for an author's posts
	ReasonDisabled CanReplyReason = "Disabled"
	// ReasonDisabledAndHidden means that comments are disabled and hidden for an author's posts
	ReasonDisabledAndHidden CanReplyReason = "DisabledAndHidden"
	// ReasonMustLogin means that a user must login in order to reply to posts
	ReasonMustLogin CanReplyReason = "MustLogin"
)

// Emote is an encoding format for an emote embedded in a string of text
type Emote struct {
	ID    int `json:"id"`
	Start int `json:"start"`
	End   int `json:"end"`
	Set   int `json:"set"`
}

// Inject is the object returned for sponsored and promoted content
type Inject struct {
	Type            string   `json:"type"`
	Position        int      `json:"position"`
	ID              string   `json:"id"`
	Reason          string   `json:"reason,omitempty"`
	Source          string   `json:"source"`
	Description     string   `json:"description,omitempty"`
	AuthorName      string   `json:"author_name,omitempty"`
	AuthorURL       string   `json:"author_url,omitempty"`
	AuthorThumbnail string   `json:"author_thumbnail_url,omitempty"`
	AuthorLogin     string   `json:"author_login,omitempty"`
	Embeds          []*Embed `json:"embeds,omitempty"`
}

// Embed contains information related to the RequestURL
type Embed struct {
	RequestURL      string      `json:"request_url,omitempty"`
	URL             string      `json:"url,omitempty"`
	Type            string      `json:"type,omitempty"`
	Title           string      `json:"title,omitempty"`
	Description     string      `json:"description,omitempty"`
	AuthorName      string      `json:"author_name,omitempty"`
	AuthorURL       string      `json:"author_url,omitempty"`
	AuthorThumbnail string      `json:"author_thumbnail_url,omitempty"`
	AuthorID        string      `json:"author_id,omitempty"`
	AuthorLogin     string      `json:"author_login,omitempty"`
	ThumbnailURL    string      `json:"thumbnail_url,omitempty"`
	Thumbnails      *Thumbnails `json:"thumbnail_urls,omitempty"`
	PlayerHTML      string      `json:"player_html,omitempty"`
	ProviderName    string      `json:"provider_name,omitempty"`

	// Twitch VOD/Clips specific
	CreatedAt       *time.Time `json:"created_at,omitempty"`
	Game            string     `json:"game,omitempty"`
	VideoLength     int        `json:"video_length,omitempty"`
	ViewCount       int        `json:"view_count,omitempty"`
	TwitchType      string     `json:"twitch_type,omitempty"`
	TwitchContentID string     `json:"twitch_content_id,omitempty"`
	StartTime       *time.Time `json:"start_time,omitempty"`
	EndTIme         *time.Time `json:"end_time,omitempty"`

	// Twitch Stream specific
	StreamType string `json:"stream_type,omitempty"`
}

// Thumbnails provides additional thumbnail URLs
type Thumbnails struct {
	Large  string `json:"large,omitempty"`
	Medium string `json:"medium,omitempty"`
	Small  string `json:"small,omitempty"`
	Tiny   string `json:"tiny,omitempty"`
}

// Settings contains user settings
type Settings struct {
	Entity                entity.Entity `json:"entity"` // User ID or feed ID
	CreatedAt             time.Time     `json:"created_at"`
	UpdatedAt             time.Time     `json:"updated_at"`
	SubsCanComment        bool          `json:"subs_can_comment"`
	FriendsCanComment     bool          `json:"friends_can_comment"`
	FollowersCanComment   bool          `json:"followers_can_comment"`
	UserDisabledComments  bool          `json:"user_disabled_comments"`
	AdminDisabledComments bool          `json:"admin_disabled_comments"`
	ChannelFeedEnabled    bool          `json:"channel_feed_enabled"`
}

func (c *ClientImpl) http(ctx context.Context, statName string, method string, path string, queryParams url.Values, body interface{}, reqOpts *twitchclient.ReqOpts, into interface{}) error {
	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       statName,
		StatSampleRate: defaultStatSampleRate,
	})

	doer := &twitchdoer.TwitchHTTPDoer{
		Client: c.Client,
		Reqopt: combinedReqOpts,
	}
	return clients.DoHTTP(ctx, doer, method, path, queryParams, body, into, doer.NewTwitchRequest)
}

// GetFeedOptions specifies the optional parameters of a GetFeed operation
type GetFeedOptions struct {
	UserID       string `url:"user_id"`
	Limit        int    `url:"limit"`
	Cursor       string `url:"cursor"`
	CommentLimit *int   `url:"comment_limit"`
	DeviceID     string `url:"device_id"`
	Language     string `url:"language"`
}

// GetFeed returns the feed for the given consumer
func (c *ClientImpl) GetFeed(ctx context.Context, feed entity.Entity, options *GetFeedOptions, reqOpts *twitchclient.ReqOpts) (*Feed, error) {
	path := "/v1/feeds/" + feed.Encode()

	vq, err := query.Values(options)
	if err != nil {
		return nil, err
	}

	var ret Feed
	if err := c.http(ctx, "service.feeds_edge.get_feed", "GET", path, vq, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetPostOptions specifies the optional parameters of a GetPost operation
type GetPostOptions struct {
	UserID       string `url:"user_id"`
	Limit        int    `url:"limit"`
	Cursor       string `url:"cursor"`
	CommentLimit *int   `url:"comment_limit"`
}

// GetPost returns the post for the post id
func (c *ClientImpl) GetPost(ctx context.Context, postID string, options *GetPostOptions, reqOpts *twitchclient.ReqOpts) (*Post, error) {
	path := "/v1/posts/" + postID

	vq, err := query.Values(options)
	if err != nil {
		return nil, err
	}

	var ret Post
	if err := c.http(ctx, "service.feeds_edge.get_post", "GET", path, vq, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// CreatePostOptions specifies the optional parameters of a CreatePost operation
type CreatePostOptions struct {
	// The time that the post was created.  Used for migration and will be removed post migration
	CreateAt *time.Time
	// The time that the post was deleted.  Used for migration and will be removed post migration
	DeletedAt *time.Time
	// EmbedURLs are an optional list of request URLs for embeds.
	EmbedURLs *[]string
}

type createPostBody struct {
	UserID    string    `json:"user_id"`
	Body      string    `json:"body"`
	EmbedURLs *[]string `json:"embed_urls,omitempty"`
}

// CreatePost creates a new post for the given user
func (c *ClientImpl) CreatePost(ctx context.Context, channelID string, body string, userID string, options *CreatePostOptions, reqOpts *twitchclient.ReqOpts) (*Post, error) {
	path := "/v1/posts?user_id=" + userID
	httpBody := createPostBody{
		UserID: channelID,
		Body:   body,
	}

	if options != nil {
		if options.EmbedURLs != nil {
			httpBody.EmbedURLs = options.EmbedURLs
		}
	}

	var ret Post
	if err := c.http(ctx, "service.feeds_edge.create_post", "POST", path, nil, httpBody, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

type createSyndicatedPostBody struct {
	UserID        string    `json:"user_id"`
	Body          string    `json:"body"`
	PostToTwitter bool      `json:"post_to_twitter"`
	EmbedURLs     *[]string `json:"embed_urls,omitempty"`
}

// CreateSyndicatedPost creates a new post for the given user and if specified, creates a post on Twitter.
func (c *ClientImpl) CreateSyndicatedPost(ctx context.Context, channelID string, body string, userID string, postToTwitter bool, options *CreatePostOptions, reqOpts *twitchclient.ReqOpts) (*SyndicatedPost, error) {
	path := "/v1/posts/syndicate?user_id=" + userID
	httpBody := createSyndicatedPostBody{
		UserID:        channelID,
		Body:          body,
		PostToTwitter: postToTwitter,
	}

	if options != nil {
		if options.EmbedURLs != nil {
			httpBody.EmbedURLs = options.EmbedURLs
		}
	}

	var ret SyndicatedPost
	if err := c.http(ctx, "service.feeds_edge.create_syndicated_post", "POST", path, nil, httpBody, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// DeletePost deletes the given post.  userID specifies the user that is deleting the post.
func (c *ClientImpl) DeletePost(ctx context.Context, postID, userID string, reqOpts *twitchclient.ReqOpts) (*Post, error) {
	path := "/v1/posts/" + postID

	query := url.Values{}
	query.Set("user_id", userID)

	var ret Post
	if err := c.http(ctx, "service.feeds_edge.delete_post", "DELETE", path, query, nil, reqOpts, &ret); err != nil {
		if clients.ErrorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}

	return &ret, nil
}

// reportBody represents the request body expected by the ReportComment and ReportPost endpoints.
type reportBody struct {
	Reason string `json:"reason"`
}

// ReportPost reports the specified post for moderation.
// reason identifies what sort of violation occurred, and gets passed through to Leviathan.  It is expected to be one
// of the values listed under report[reason] in Leviathan's API doc (https://git-aws.internal.justin.tv/chat/leviathan/blob/master/docs/api.md.)
func (c *ClientImpl) ReportPost(ctx context.Context, postID, userID, reason string, clientID string, reqOpts *twitchclient.ReqOpts) error {
	path := fmt.Sprintf("/v1/posts/%s/report", postID)

	query := url.Values{}
	query.Set("user_id", userID)
	query.Set("client_id", clientID)

	requestBody := reportBody{
		Reason: reason,
	}
	return c.http(ctx, "service.feeds_edge.report_post", "POST", path, query, requestBody, reqOpts, nil)
}

type createCommentBody struct {
	UserID       string        `json:"user_id"`
	ParentEntity entity.Entity `json:"parent_entity"`
	Body         string        `json:"body"`
}

// CreateComment creates a new comment for the given parent entity.
func (c *ClientImpl) CreateComment(ctx context.Context, parentEntity entity.Entity, body string, userID string, reqOpts *twitchclient.ReqOpts) (*Comment, error) {
	path := "/v1/comments"
	httpBody := createCommentBody{
		UserID:       userID,
		ParentEntity: parentEntity,
		Body:         body,
	}

	var ret Comment
	if err := c.http(ctx, "service.feeds_edge.create_comment", "POST", path, nil, httpBody, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetCommentByIDOptions specifies the optional parameters of a GetCommentsForPost operation
type GetCommentByIDOptions struct {
	UserID string `url:"user_id,omitempty"`
}

// GetCommentByID returns a comment with the given comment id
func (c *ClientImpl) GetCommentByID(ctx context.Context, commentID string, options *GetCommentByIDOptions, reqOpts *twitchclient.ReqOpts) (*Comment, error) {
	path := "/v2/comments/" + commentID

	vq, err := query.Values(options)
	if err != nil {
		return nil, err
	}

	var ret Comment
	if err := c.http(ctx, "service.feeds_edge.get_comment_by_id", "GET", path, vq, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetCommentsForPostOptions specifies the optional parameters of a GetCommentsForPost operation
type GetCommentsForPostOptions struct {
	UserID string `url:"user_id,omitempty"`
	Limit  int    `url:"limit,omitempity"`
	Cursor string `url:"cursor,omitempty"`
}

// GetCommentsForPost returns a paginated list of comments for a given post
func (c *ClientImpl) GetCommentsForPost(ctx context.Context, postID string, options *GetCommentsForPostOptions, reqOpts *twitchclient.ReqOpts) (*Comments, error) {
	path := "/v1/posts/" + postID + "/comments"

	vq, err := query.Values(options)
	if err != nil {
		return nil, err
	}

	var ret Comments
	if err := c.http(ctx, "service.feeds_edge.get_comments_for_post", "GET", path, vq, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// ApproveComment approves a comment that needs approval
func (c *ClientImpl) ApproveComment(ctx context.Context, commentID string, userID string, reqOpts *twitchclient.ReqOpts) (*Comment, error) {
	path := "/v1/comments/" + commentID + "/approve"

	query := url.Values{}
	query.Set("user_id", userID)

	var ret Comment
	if err := c.http(ctx, "service.feeds_edge.approve_comment", "PUT", path, query, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// DenyComment denies a comment that needs approval and deletes it
func (c *ClientImpl) DenyComment(ctx context.Context, commentID string, userID string, reqOpts *twitchclient.ReqOpts) error {
	path := "/v1/comments/" + commentID + "/deny"

	query := url.Values{}
	query.Set("user_id", userID)

	return c.http(ctx, "service.feeds_edge.deny_comment", "PUT", path, query, nil, reqOpts, nil)
}

// DeleteComment deletes the given comment.  userID specifies the user that is deleting the reaction.
func (c *ClientImpl) DeleteComment(ctx context.Context, commentID, userID string, reqOpts *twitchclient.ReqOpts) (*Comment, error) {
	path := "/v1/comments/" + commentID

	query := url.Values{}
	query.Set("user_id", userID)

	var ret Comment
	if err := c.http(ctx, "service.feeds_edge.delete_comment", "DELETE", path, query, nil, reqOpts, &ret); err != nil {
		if clients.ErrorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}

	return &ret, nil
}

// DeleteCommentsByParentAndUser deletes the comments by the specified commentUserID on the given parent entity.
// the passed userID is the person doing the deletion, and is used to verify permissions
func (c *ClientImpl) DeleteCommentsByParentAndUser(ctx context.Context, parent entity.Entity, commentUserID string, userID string, reqOpts *twitchclient.ReqOpts) error {
	path := "/v1/comments/by_parent_and_user/" + parent.Encode() + "/" + commentUserID

	query := url.Values{}
	query.Set("user_id", userID)

	return c.http(ctx, "service.feeds_edge.delete_comments_by_parent_and_user", "DELETE", path, query, nil, reqOpts, nil)
}

// ReportComment reports the given comment for moderation.
// reason identifies what sort of violation occurred, and gets passed through to Leviathan.  It is expected to be one
// of the values listed under report[reason] in Leviathan's API doc (https://git-aws.internal.justin.tv/chat/leviathan/blob/master/docs/api.md.)
func (c *ClientImpl) ReportComment(ctx context.Context, commentID, userID, reason string, reqOpts *twitchclient.ReqOpts) error {
	path := "/v1/comments/" + commentID + "/report"

	vq := url.Values{}
	vq.Set("user_id", userID)

	requestBody := reportBody{
		Reason: reason,
	}
	return c.http(ctx, "service.feeds_edge.report_comment", "POST", path, vq, requestBody, reqOpts, nil)
}

// GetEmbedOptions specifies the optional parameters of the GetEmbed function
type GetEmbedOptions struct {
	Autoplay *bool `url:"autoplay"`
}

// GetEmbed returns embed data for a provided URL
func (c *ClientImpl) GetEmbed(ctx context.Context, embedURL string, options *GetEmbedOptions, reqOpts *twitchclient.ReqOpts) (*Embed, error) {
	path := "v1/embed"

	vq, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	vq.Add("url", embedURL)

	var ret Embed
	if err := c.http(ctx, "service.feeds_edge.get_embed", "GET", path, vq, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// CreateReaction creates a reaction on the given entity (post, comment), and returns a struct that contains all the
// reactions that the user currently has created on the entity.  userID specifies the user that is creating the
// reaction.
func (c *ClientImpl) CreateReaction(ctx context.Context, parent entity.Entity, emoteID, userID string, reqOpts *twitchclient.ReqOpts) (*UserReactions, error) {
	path := "/v1/reactions/" + parent.Encode() + "/" + emoteID

	query := url.Values{}
	query.Set("user_id", userID)

	var ret UserReactions
	if err := c.http(ctx, "service.feeds_edge.create_reaction", "PUT", path, query, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetReactionsForParentsOptions specifies the optional parameters of a GetReactionsForParents operation.
type GetReactionsForParentsOptions struct {
	UserID string `url:"user_id,omitempty"`
}

// GetReactionsForParents returns all reactions for each entity (post, comment) passed in as
// part of the parentEntities slice, formatted as the ReactionsWithParentEntities struct.
func (c *ClientImpl) GetReactionsForParents(ctx context.Context, parentEntities []entity.Entity, options *GetReactionsForParentsOptions, reqOpts *twitchclient.ReqOpts) (*BatchReactions, error) {
	path := "/v1/reactions"

	vq, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	vq.Set("parent_entity", strings.Join(entity.EncodeAll(parentEntities), ","))

	var ret BatchReactions
	if err := c.http(ctx, "service.feeds_edge.get_reactions_by_parents", "GET", path, vq, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// DeleteReaction removes the specified emote, on the given entity (post, comment), and returns a struct that contains
// the remaining emotes that the user created on the entity.  userID specifies the user that is deleting the emote.
func (c *ClientImpl) DeleteReaction(ctx context.Context, parent entity.Entity, emoteID, userID string, reqOpts *twitchclient.ReqOpts) (*UserReactions, error) {
	path := "/v1/reactions/" + parent.Encode() + "/" + emoteID

	query := url.Values{}
	query.Set("user_id", userID)

	var ret UserReactions
	if err := c.http(ctx, "service.feeds_edge.delete_reaction", "DELETE", path, query, nil, reqOpts, &ret); err != nil {
		if clients.ErrorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}

	return &ret, nil
}

// GetSettings gets settings for the given user
func (c *ClientImpl) GetSettings(ctx context.Context, e entity.Entity, userID string, reqOpts *twitchclient.ReqOpts) (*Settings, error) {
	path := "/v1/settings/" + e.Encode()

	query := url.Values{}
	query.Set("user_id", userID)

	var ret Settings
	if err := c.http(ctx, "service.feeds_edge.get_settings", "GET", path, query, nil, reqOpts, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}

// UpdateSettingsOptions specifies the optional parameters of an UpdateSettings operation
type UpdateSettingsOptions struct {
	SubsCanComment        *bool `json:"subs_can_comment,omitempty"`
	FriendsCanComment     *bool `json:"friends_can_comment,omitempty"`
	FollowersCanComment   *bool `json:"followers_can_comment,omitempty"`
	UserDisabledComments  *bool `json:"user_disabled_comments,omitempty"`
	AdminDisabledComments *bool `json:"admin_disabled_comments,omitempty"`
	ChannelFeedEnabled    *bool `json:"channel_feed_enabled,omitempty"`
}

// UpdateSettings updates settings for the given user
func (c *ClientImpl) UpdateSettings(ctx context.Context, e entity.Entity, userID string, options *UpdateSettingsOptions, reqOpts *twitchclient.ReqOpts) (*Settings, error) {
	path := "/v1/settings/" + e.Encode()

	query := url.Values{}
	query.Set("user_id", userID)

	var ret Settings
	if err := c.http(ctx, "service.feeds_edge.update_settings", "POST", path, query, options, reqOpts, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}

// CreateShare will create a share for a user
func (c *ClientImpl) CreateShare(ctx context.Context, userID string, targetEntity entity.Entity, reqOpts *twitchclient.ReqOpts) (*Share, error) {
	path := "/v1/shares"
	httpBody := struct {
		UserID       string        `json:"user_id"`
		TargetEntity entity.Entity `json:"target_entity"`
	}{
		UserID:       userID,
		TargetEntity: targetEntity,
	}

	query := make(url.Values)
	query.Add("user_id", userID)

	var ret Share
	if err := c.http(ctx, "service.feeds_edge.create_share", "POST", path, query, httpBody, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetShare returns a share by the share's ID
func (c *ClientImpl) GetShare(ctx context.Context, userID string, shareID string, reqOpts *twitchclient.ReqOpts) (*Share, error) {
	path := "/v1/shares/" + shareID
	query := make(url.Values)
	query.Add("user_id", userID)

	var ret Share
	if err := c.http(ctx, "service.feeds_edge.get_share", "GET", path, query, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// UnshareByTarget will unshare a list of target entities
func (c *ClientImpl) UnshareByTarget(ctx context.Context, userID string, authorID string, targetEntities []entity.Entity, reqOpts *twitchclient.ReqOpts) (*Shares, error) {
	path := "/v1/unshare_by_target"
	query := make(url.Values)
	query.Add("user_id", userID)
	query.Add("author_id", authorID)
	query.Add("target_entities", strings.Join(entity.EncodeAll(targetEntities), ","))

	var ret Shares
	if err := c.http(ctx, "service.feeds_edge.unshare_by_target", "DELETE", path, query, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetPostPermissions returns permissions about posting as a user to a feed
func (c *ClientImpl) GetPostPermissions(ctx context.Context, userID string, feedID string, reqOpts *twitchclient.ReqOpts) (*PostPermissions, error) {
	path := "/v1/permissions/" + feedID
	query := make(url.Values)
	query.Add("user_id", userID)

	var ret PostPermissions
	if err := c.http(ctx, "service.feeds_edge.get_post_permissions", "GET", path, query, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetShares returns a list of shares by IDs
func (c *ClientImpl) GetShares(ctx context.Context, userID string, shareIDs []string, reqOpts *twitchclient.ReqOpts) (*Shares, error) {
	path := "/v1/shares"
	query := make(url.Values)
	query.Add("user_id", userID)
	query.Add("ids", strings.Join(shareIDs, ","))

	var ret Shares
	if err := c.http(ctx, "service.feeds_edge.get_shares", "GET", path, query, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetSharesSummaries returns multiple GetSharesSummaries for a list of parent entities.  If the entity does not exist, then it assumes
// zero shares.
func (c *ClientImpl) GetSharesSummaries(ctx context.Context, userID string, parentEntities []entity.Entity, authorIDs []string, reqOpts *twitchclient.ReqOpts) (*SharesSummaries, error) {
	path := "/v1/shares_summaries"
	query := make(url.Values)
	query.Add("user_id", userID)
	query.Add("author_ids", strings.Join(authorIDs, ","))
	query.Add("parent_entities", strings.Join(entity.EncodeAll(parentEntities), ","))

	var ret SharesSummaries
	if err := c.http(ctx, "service.feeds_edge.get_shares_summaries", "GET", path, query, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// DeleteShare removes a share by the share ID (not the target of the share)
func (c *ClientImpl) DeleteShare(ctx context.Context, userID string, shareID string, reqOpts *twitchclient.ReqOpts) (*Share, error) {
	path := "/v1/shares/" + shareID
	query := make(url.Values)
	query.Add("user_id", userID)

	var ret Share
	if err := c.http(ctx, "service.feeds_edge.delete_share", "DELETE", path, query, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}
