package feeddataflow

import (
	"time"

	"code.justin.tv/feeds/feeds-common/entity"
)

type timeValidAt struct {
	T       time.Time
	ValidAt time.Time
}

type boolValidAt struct {
	B       bool
	ValidAt time.Time
}

type stringSetValidAt struct {
	S       map[string]bool
	ValidAt time.Time
}

// Metadata allows users to send information across calls between feed services so we don't have to duplicate traffic.
// Use each trait's ValidAt field to see if it's still information you should consider valid.
type Metadata struct {
	EntityCreationTimes map[entity.Entity]timeValidAt `json:",omitempty"`
	EntityIsDeleted     map[entity.Entity]boolValidAt `json:",omitempty"`
	Follows             map[string]stringSetValidAt   `json:",omitempty"`
	Version             int
}

func (m *Metadata) mergeEntityCreationTimes(from *Metadata) {
	if len(from.EntityCreationTimes) == 0 {
		return
	}
	if m.EntityCreationTimes == nil {
		m.EntityCreationTimes = make(map[entity.Entity]timeValidAt, len(from.EntityCreationTimes))
	}
	for entity, tv := range from.EntityCreationTimes {
		creationTime, validAt, exists := m.GetEntityCreationTime(entity)
		if !exists {
			m.SetEntityCreationTime(entity, tv.T, tv.ValidAt)
			continue
		}
		if tv.ValidAt.After(validAt) {
			m.SetEntityCreationTime(entity, creationTime, validAt)
			continue
		}
	}
}

func (m *Metadata) mergeEntityIsDeleted(from *Metadata) {
	if len(from.EntityIsDeleted) == 0 {
		return
	}
	if m.EntityIsDeleted == nil {
		m.EntityIsDeleted = make(map[entity.Entity]boolValidAt, len(from.EntityIsDeleted))
	}
	for entity, tv := range from.EntityIsDeleted {
		isDeleted, validAt, exists := m.GetIsDeleted(entity)
		if !exists {
			m.SetIsDeleted(entity, tv.B, tv.ValidAt)
			continue
		}
		if tv.ValidAt.After(validAt) {
			m.SetIsDeleted(entity, isDeleted, validAt)
			continue
		}
	}
}

func (m *Metadata) mergeFollows(from *Metadata) {
	if len(from.Follows) == 0 {
		return
	}
	if m.Follows == nil {
		m.Follows = make(map[string]stringSetValidAt, len(from.Follows))
	}
	for from, sv := range from.Follows {
		for to, doesFollow := range sv.S {
			m.SetFollow(from, to, doesFollow, sv.ValidAt)
		}
	}
}

// MergeFrom will merge from into this structure
func (m *Metadata) MergeFrom(from *Metadata) {
	if from == nil {
		return
	}
	m.mergeEntityCreationTimes(from)
	m.mergeEntityIsDeleted(from)
	m.mergeFollows(from)
}

// SetFollow adds that from follows to
func (m *Metadata) SetFollow(from string, to string, follows bool, validAt time.Time) {
	if m.Follows == nil {
		m.Follows = map[string]stringSetValidAt{
			from: {
				S:       map[string]bool{to: follows},
				ValidAt: validAt,
			},
		}
		return
	}
	current, exists := m.Follows[from]
	if exists {
		current.S[to] = follows
		if validAt.Before(current.ValidAt) {
			current.ValidAt = validAt
		}
		return
	}
	m.Follows[from] = stringSetValidAt{
		S:       map[string]bool{to: follows},
		ValidAt: validAt,
	}
}

// GetFollow tells if from follows -> to
func (m *Metadata) GetFollow(from string, to string) (bool, time.Time, bool) {
	if m == nil || len(m.Follows) == 0 {
		return false, time.Time{}, false
	}
	currentSet, exists := m.Follows[from]
	if !exists {
		return false, time.Time{}, false
	}
	follows, knowsAbout := currentSet.S[to]
	if !knowsAbout {
		return false, time.Time{}, false
	}
	return follows, currentSet.ValidAt, true
}

// SetEntityCreationTime stores that entity was created at a time and it's valid as of validAt
func (m *Metadata) SetEntityCreationTime(ent entity.Entity, createdAt time.Time, validAt time.Time) {
	if m.EntityCreationTimes == nil {
		m.EntityCreationTimes = map[entity.Entity]timeValidAt{
			ent: {
				T:       createdAt,
				ValidAt: validAt,
			},
		}
		return
	}
	m.EntityCreationTimes[ent] = timeValidAt{
		T:       createdAt,
		ValidAt: validAt,
	}
}

// GetEntityCreationTime will return the entity's creation time and when it was valid
func (m *Metadata) GetEntityCreationTime(ent entity.Entity) (time.Time, time.Time, bool) {
	if m == nil || len(m.EntityCreationTimes) == 0 {
		return time.Time{}, time.Time{}, false
	}
	item, exists := m.EntityCreationTimes[ent]
	if !exists {
		return time.Time{}, time.Time{}, false
	}
	return item.T, item.ValidAt, true
}

// SetIsDeleted set that an entity is already deleted
func (m *Metadata) SetIsDeleted(ent entity.Entity, isDeleted bool, validAt time.Time) {
	if m.EntityIsDeleted == nil {
		m.EntityIsDeleted = map[entity.Entity]boolValidAt{
			ent: {
				B:       isDeleted,
				ValidAt: validAt,
			},
		}
		return
	}
	m.EntityIsDeleted[ent] = boolValidAt{
		B:       isDeleted,
		ValidAt: validAt,
	}
}

// GetIsDeleted returns if metadata knows an item is deleted
func (m *Metadata) GetIsDeleted(ent entity.Entity) (bool, time.Time, bool) {
	if m == nil || len(m.EntityIsDeleted) == 0 {
		return false, time.Time{}, false
	}
	item, exists := m.EntityIsDeleted[ent]
	if !exists {
		return false, time.Time{}, false
	}
	return item.B, item.ValidAt, true
}
