// +build integration

package feedsedge_test

import (
	"strconv"
	"testing"
	"time"

	"net/http"

	"code.justin.tv/feeds/clients/edge"
	"code.justin.tv/feeds/clients/feedsedge"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

func newSetup() *testSetup {
	return &testSetup{
		Common: service_common.ConfigCommon{
			Team:          "feeds",
			Service:       "client_int_tests",
			BaseDirectory: "../",
		},
	}
}

const (
	testUserID     = "27184041"
	testChannelID  = testUserID
	testUserID2    = "153061467"
	testChannelID2 = testUserID2

	// These are used to spread out tests because of post creation rate limiting
	// TODO We should use integration environments with low post creation timeouts
	testUserID3 = "140837659"
	testUserID4 = "46491334"
	testUserID5 = "140601908"
	testUserID6 = "140601818"
	testUserID7 = "129660415"
	testUserID8 = "152973670"
)

// Config configures edge http client
type Config struct {
	Host func() string
}

// Load configuration information
func (c *Config) Load(dconf *distconf.Distconf) error {
	c.Host = dconf.Str("edge.http_endpoint", "").Get
	return nil
}

func timestamp() string {
	return strconv.Itoa(int(time.Now().UnixNano()))
}

func TestClientIntegration_SuggestedFeeds(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func(c C) {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		c.Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		cl := setup.Client
		t.Log("using client addr ", setup.Config.Host())

		Convey("Should be able to get suggested feeds", func() {
			feeds, err := cl.GetSuggestedFeeds(ctx, testUserID, nil)
			So(err, ShouldBeNil)
			So(len(feeds.FeedIDs), ShouldBeGreaterThan, 0)
		})
		Convey("Should require user ID", func() {
			_, err := cl.GetSuggestedFeeds(ctx, "", nil)
			So(err, ShouldNotBeNil)
		})
	})
}

func TestClientIntegration_Feed(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func(c C) {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		c.Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		cl := setup.Client
		t.Log("using client addr ", setup.Config.Host())

		Convey("Should be able to get an empty feed", func() {
			feedEntity := entity.New("c", timestamp())
			feed, err := cl.GetFeed(ctx, feedEntity, nil, nil)
			So(err, ShouldBeNil)
			So(feed, ShouldNotBeNil)
		})

		Convey("Should be able to get a feed", func() {
			feedEntity := entity.New("r", testUserID)
			feed, err := cl.GetFeed(ctx, feedEntity, &feedsedge.GetFeedOptions{
				UserID: testUserID,
			}, nil)
			So(err, ShouldBeNil)
			So(feed, ShouldNotBeNil)
		})
		Convey("Feeds should check permissions", func() {
			feedEntity := entity.New("r", testUserID)
			feed, err := cl.GetFeed(ctx, feedEntity, &feedsedge.GetFeedOptions{
				UserID: testUserID + "_",
			}, nil)
			So(errorCode(err), ShouldEqual, http.StatusUnauthorized)
			So(feed, ShouldBeNil)
		})
		Convey("Should be able to get settings", func() {
			settingsEntity := entity.New(entity.NamespaceUser, newUserID())

			settings, err := cl.GetSettings(ctx, settingsEntity, settingsEntity.ID(), nil)
			So(err, ShouldBeNil)
			So(settings, ShouldNotBeNil)
			So(settings.Entity, ShouldResemble, settingsEntity)
		})

		Convey("Should be able to update settings", func() {
			settingsEntity := entity.New(entity.NamespaceUser, newUserID())

			settings, err := cl.GetSettings(ctx, settingsEntity, settingsEntity.ID(), nil)
			So(err, ShouldBeNil)
			So(settings.Entity, ShouldResemble, settingsEntity)
			So(settings.ChannelFeedEnabled, ShouldBeTrue) // default value

			newChannelFeedEnabled := false
			updated, err := cl.UpdateSettings(ctx, settingsEntity, settingsEntity.ID(), &feedsedge.UpdateSettingsOptions{
				ChannelFeedEnabled: &newChannelFeedEnabled,
			}, nil)
			So(err, ShouldBeNil)
			So(updated.Entity, ShouldResemble, settingsEntity)
			So(updated.ChannelFeedEnabled, ShouldEqual, newChannelFeedEnabled)
		})
	})
}

func TestClientIntegration_Post(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func(c C) {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		c.Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		cl := setup.Client
		t.Log("using client addr ", setup.Config.Host())

		Convey("Should be able to create posts", func() {
			post1Body := "hello https://clips.twitch.tv/InexpensiveDeliciousDiamondFrankerZ"
			postResponse1, err := cl.CreatePost(ctx, testChannelID, post1Body, testUserID, false, nil, nil)
			So(err, ShouldBeNil)
			So(postResponse1, ShouldNotBeNil)
			post1 := postResponse1.Post
			So(post1, ShouldNotBeNil)
		})

		Convey("Should be able to get multiple posts", func() {
			post1Body := "hello https://clips.twitch.tv/InexpensiveDeliciousDiamondFrankerZ"
			postResponse1, err := cl.CreatePost(ctx, testChannelID, post1Body, testUserID, false, nil, nil)
			So(err, ShouldBeNil)
			So(postResponse1, ShouldNotBeNil)
			post1 := postResponse1.Post
			So(post1, ShouldNotBeNil)

			testBody := "TestClientIntegration_Post " + timestamp()
			postResponse2, err := cl.CreatePost(ctx, testChannelID2, testBody, testUserID2, false, nil, nil)
			So(err, ShouldBeNil)
			So(postResponse2, ShouldNotBeNil)
			post2 := postResponse2.Post
			So(post2, ShouldNotBeNil)
			So(post2.EmbedEntities, ShouldHaveLength, 0)

			posts, err := cl.GetPostsByIDs(ctx, []string{post1.ID, post2.ID}, nil)
			So(err, ShouldBeNil)
			So(posts, ShouldNotBeNil)
			So(posts.Items, ShouldHaveLength, 2)

			postMap := make(map[string]*feedsedge.Post)
			for _, post := range posts.Items {
				postMap[post.ID] = post
			}

			So(postMap[post1.ID], ShouldNotBeNil)
			So(postMap[post1.ID].UserID, ShouldEqual, testUserID)
			So(postMap[post1.ID].Body, ShouldEqual, post1Body)
			So(postMap[post1.ID].EmbedEntities, ShouldNotBeNil)
			So(len(*postMap[post1.ID].EmbedEntities), ShouldEqual, 1)
			So((*postMap[post1.ID].EmbedEntities)[0].ID(), ShouldEqual, "InexpensiveDeliciousDiamondFrankerZ")
			So((*postMap[post1.ID].EmbedEntities)[0].Namespace(), ShouldEqual, entity.NamespaceClip)

			So(postMap[post2.ID], ShouldNotBeNil)
			So(postMap[post2.ID].UserID, ShouldEqual, testUserID2)
			So(postMap[post2.ID].Body, ShouldEqual, testBody)
			So(postMap[post2.ID].EmbedEntities, ShouldNotBeNil)
			So(len(*postMap[post2.ID].EmbedEntities), ShouldEqual, 0)
		})

		Convey("Should be able to get multiple posts' permissions", func() {
			post1Body := "hello https://clips.twitch.tv/InexpensiveDeliciousDiamondFrankerZ"
			postResponse1, err := cl.CreatePost(ctx, testChannelID, post1Body, testUserID, false, nil, nil)
			So(err, ShouldBeNil)
			So(postResponse1, ShouldNotBeNil)
			post1 := postResponse1.Post
			So(post1, ShouldNotBeNil)

			testBody := "TestClientIntegration_Post " + timestamp()
			postResponse2, err := cl.CreatePost(ctx, testChannelID2, testBody, testUserID2, false, nil, nil)
			So(err, ShouldBeNil)
			So(postResponse2, ShouldNotBeNil)
			post2 := postResponse2.Post
			So(post2, ShouldNotBeNil)

			permissions, err := cl.GetPostsPermissionsByIDs(ctx, []string{post1.ID, post2.ID}, testUserID, nil)
			So(err, ShouldBeNil)
			So(permissions, ShouldNotBeNil)
			So(permissions.Items, ShouldHaveLength, 2)
			if permissions.Items[0].PostID != post1.ID {
				tmp := permissions.Items[0]
				permissions.Items[0] = permissions.Items[1]
				permissions.Items[1] = tmp
			}
			So(permissions.Items[0].PostID, ShouldEqual, post1.ID)
			So(permissions.Items[0].CanDelete, ShouldBeTrue)
			So(permissions.Items[1].PostID, ShouldEqual, post2.ID)
			So(permissions.Items[1].CanDelete, ShouldBeFalse)
		})

		Convey("Should be able to delete posts", func() {
			post1Body := "hello https://clips.twitch.tv/InexpensiveDeliciousDiamondFrankerZ"
			postResponse1, err := cl.CreatePost(ctx, testChannelID, post1Body, testUserID, false, nil, nil)
			So(err, ShouldBeNil)
			So(postResponse1, ShouldNotBeNil)
			post1 := postResponse1.Post
			So(post1, ShouldNotBeNil)

			postID1 := post1.ID
			post3, err := cl.DeletePost(ctx, postID1, testUserID, nil)
			So(err, ShouldBeNil)
			So(post3.ID, ShouldEqual, postID1)
			posts, err := cl.GetPostsByIDs(ctx, []string{postID1}, nil)
			So(err, ShouldBeNil)
			So(posts, ShouldNotBeNil)
			So(posts.Items, ShouldHaveLength, 0)
		})
	})
}

func TestClientIntegration_Share(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func(c C) {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		c.Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		cl := setup.Client
		oldEdgeClient := setup.OldEdgeClient
		t.Log("using client addr ", setup.Config.Host())

		Convey("Should be able to create a share", func() {
			userID := testUserID3
			testBody := "TestClientIntegration_Post_Create" + timestamp()
			postResponse, err := cl.CreatePost(ctx, userID, testBody, userID, false, nil, nil)
			So(err, ShouldBeNil)
			So(postResponse, ShouldNotBeNil)
			So(postResponse.Post, ShouldNotBeNil)
			postEntity := entity.New("post", postResponse.Post.ID)

			share, err := cl.CreateShare(ctx, postEntity, testUserID, nil)
			So(err, ShouldBeNil)
			So(share, ShouldNotBeNil)
			So(share.UserID, ShouldEqual, testUserID)
			So(share.TargetEntity.Encode(), ShouldEqual, postEntity.Encode())
		})

		Convey("Should be able to get multiple shares", func() {
			userID := testUserID3
			testBody := "TestClientIntegration_Post_Get" + timestamp()
			post, err := oldEdgeClient.CreatePost(ctx, userID, testBody, userID, &edge.CreatePostOptions{}, nil)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			postEntity := entity.New("post", post.ID)

			share1, err := oldEdgeClient.CreateShare(ctx, testUserID, postEntity, nil)
			So(err, ShouldBeNil)
			So(share1, ShouldNotBeNil)

			share2, err := oldEdgeClient.CreateShare(ctx, testUserID2, postEntity, nil)
			So(err, ShouldBeNil)
			So(share1, ShouldNotBeNil)

			shares, err := cl.GetSharesByIDs(ctx, []string{share1.ID, share2.ID}, nil)
			So(err, ShouldBeNil)
			So(shares, ShouldNotBeNil)
			So(shares.Items, ShouldHaveLength, 2)

			shareMap := make(map[string]*feedsedge.Share)
			for _, share := range shares.Items {
				shareMap[share.ID] = share
			}

			So(shareMap[share1.ID], ShouldNotBeNil)
			So(shareMap[share1.ID].UserID, ShouldEqual, share1.UserID)
			So(shareMap[share1.ID].TargetEntity, ShouldResemble, share1.TargetEntity)
			So(shareMap[share1.ID].CreatedAt, ShouldResemble, share1.CreatedAt)

			So(shareMap[share2.ID], ShouldNotBeNil)
			So(shareMap[share2.ID].UserID, ShouldEqual, share2.UserID)
			So(shareMap[share2.ID].TargetEntity, ShouldResemble, share2.TargetEntity)
			So(shareMap[share2.ID].CreatedAt, ShouldResemble, share2.CreatedAt)
		})

		Convey("Should be able to delete a share", func() {
			userID := testUserID3
			testBody := "TestClientIntegration_Post_Create" + timestamp()
			postResponse, err := cl.CreatePost(ctx, userID, testBody, userID, false, nil, nil)
			So(err, ShouldBeNil)
			So(postResponse, ShouldNotBeNil)
			So(postResponse.Post, ShouldNotBeNil)
			postEntity := entity.New("post", postResponse.Post.ID)

			share1, err := cl.CreateShare(ctx, postEntity, testUserID, nil)
			So(err, ShouldBeNil)
			So(share1, ShouldNotBeNil)

			share2, err := cl.DeleteShare(ctx, share1.ID, testUserID, nil)
			So(err, ShouldBeNil)
			So(share2, ShouldNotBeNil)
			So(share2.ID, ShouldEqual, share1.ID)

			shares, err := cl.GetSharesByIDs(ctx, []string{share1.ID}, nil)
			So(err, ShouldBeNil)
			So(shares, ShouldNotBeNil)
			So(shares.Items, ShouldHaveLength, 0)
		})
	})
}

func TestClientIntegration_Reaction(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func(c C) {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		c.Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		cl := setup.Client
		oldEdgeClient := setup.OldEdgeClient
		t.Log("using client addr ", setup.Config.Host())

		Convey("Should be able to get reactions for multiple entities", func() {
			userID1 := testUserID4
			userID2 := testUserID5
			testBody := "TestClientIntegration_Post " + timestamp()
			post1, err := oldEdgeClient.CreatePost(ctx, userID1, testBody, userID1, &edge.CreatePostOptions{}, nil)
			So(err, ShouldBeNil)
			So(post1, ShouldNotBeNil)
			postEntity1 := entity.New("post", post1.ID)

			post2, err := oldEdgeClient.CreatePost(ctx, userID2, testBody, userID2, &edge.CreatePostOptions{}, nil)
			So(err, ShouldBeNil)
			So(post2, ShouldNotBeNil)
			postEntity2 := entity.New("post", post2.ID)

			err = cl.CreateReaction(ctx, postEntity1, "25", userID1, nil)
			So(err, ShouldBeNil)

			So(cl.CreateReaction(ctx, postEntity2, "25", userID1, nil), ShouldBeNil)

			reactions, err := cl.GetReactionsSummariesByEntities(ctx, []string{postEntity1.String(), postEntity2.String()}, &feedsedge.GetReactionsSummariesOptions{UserID: userID1}, nil)
			So(err, ShouldBeNil)
			So(reactions, ShouldNotBeNil)
			So(reactions.Items, ShouldHaveLength, 2)

			reactionMap := make(map[string]*feedsedge.ReactionSummaries)

			for _, reaction := range reactions.Items {
				reactionMap[reaction.ParentEntity.String()] = reaction
			}

			expectedReaction := &feedsedge.ReactionSummary{
				EmoteID:     "25",
				EmoteName:   "Kappa",
				Count:       1,
				UserReacted: true,
			}

			So(reactionMap[postEntity1.String()], ShouldNotBeNil)
			So(reactionMap[postEntity1.String()].Summaries, ShouldHaveLength, 1)
			So(reactionMap[postEntity1.String()].Summaries[0], ShouldResemble, expectedReaction)

			So(reactionMap[postEntity2.String()], ShouldNotBeNil)
			So(reactionMap[postEntity2.String()].Summaries, ShouldHaveLength, 1)
			So(reactionMap[postEntity2.String()].Summaries[0], ShouldResemble, expectedReaction)

			Convey("And remove them", func() {
				So(cl.DeleteReaction(ctx, postEntity1, "25", userID1, nil), ShouldBeNil)

				reactions, err := cl.GetReactionsSummariesByEntities(ctx, []string{postEntity1.String(), postEntity2.String()}, &feedsedge.GetReactionsSummariesOptions{UserID: userID1}, nil)
				So(err, ShouldBeNil)
				So(reactions, ShouldNotBeNil)
				So(reactions.Items, ShouldHaveLength, 2)

				reactionMap := make(map[string]*feedsedge.ReactionSummaries)

				for _, reaction := range reactions.Items {
					reactionMap[reaction.ParentEntity.String()] = reaction
				}

				expectedReaction := &feedsedge.ReactionSummary{
					EmoteID:     "25",
					EmoteName:   "Kappa",
					Count:       1,
					UserReacted: true,
				}

				So(reactionMap[postEntity1.String()], ShouldNotBeNil)
				So(reactionMap[postEntity1.String()].Summaries, ShouldHaveLength, 0)

				So(reactionMap[postEntity2.String()], ShouldNotBeNil)
				So(reactionMap[postEntity2.String()].Summaries, ShouldHaveLength, 1)
				So(reactionMap[postEntity2.String()].Summaries[0], ShouldResemble, expectedReaction)
			})

		})
	})
}

type testSetup struct {
	context.Context
	feedsedge.Client
	OldEdgeClient edge.Client // We should probably remove this v1 client once all endpoints have transitioned over to v2. (@bernmar - 8/3/17)
	cancelFunc    func()
	Common        service_common.ConfigCommon
	Config        *Config
}

func (t *testSetup) Setup() error {
	t.Context, t.cancelFunc = context.WithTimeout(context.Background(), time.Second*10)
	if err := t.Common.Setup(); err != nil {
		return err
	}
	t.Config = &Config{}
	if err := t.Config.Load(t.Common.Config); err != nil {
		return err
	}

	var err error
	t.Client, err = feedsedge.NewClient(twitchclient.ClientConf{
		Host: t.Config.Host(),
	})
	if err != nil {
		return err
	}

	t.OldEdgeClient, err = edge.NewClient(twitchclient.ClientConf{
		Host: t.Config.Host(),
	})

	return err
}

func (c *testSetup) Close() error {
	c.cancelFunc()
	return c.Common.Close()
}

func errorCode(err error) int {
	if coded, ok := errors.Cause(err).(service_common.ErrorWithCode); ok {
		return coded.HTTPCode()
	}
	return 0
}

func newUserID() string {
	return timestamp()
}
