package shine

import (
	"net/http"
	"net/url"
	"time"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/clients/twitchdoer"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/foundation/twitchclient"
	"github.com/google/go-querystring/query"
	"golang.org/x/net/context"
)

const (
	defaultTimingXactName = "feeds-shine"
	defaultStatPrefix     = "service.feeds_shine."
	defaultStatSampleRate = 1.0
)

// Client satisfies visage's interface requirements for clients
type Client interface {
	GetEntitiesForURLs(ctx context.Context, urls []string, reqOpts *twitchclient.ReqOpts) (*EntitiesForURLs, error)
	GetEmbedForEntity(ctx context.Context, ent entity.Entity, options *GetEmbedOptions, reqOpts *twitchclient.ReqOpts) (*Embed, error)
	GetEmbed(ctx context.Context, embedURL string, options *GetEmbedOptions, reqOpts *twitchclient.ReqOpts) (*Embed, error)
	GetOembedsForURLs(ctx context.Context, urls []string, opts *GetOembedsForURLsOptions, reqOpts *twitchclient.ReqOpts) (*OembedsForURLs, error)
}

// ClientImpl implements the Client interface and uses the twitchhttp client to make http requests
type ClientImpl struct {
	Client twitchclient.Client
}

var _ Client = &ClientImpl{}

// NewClient returns a new instance of the Client which uses the given config
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	if err != nil {
		return nil, err
	}
	return &ClientImpl{Client: twitchClient}, nil
}

// Embed contains information related to the RequestURL
type Embed struct {
	RequestURL      string      `json:"request_url"`
	Type            string      `json:"type"`
	Title           string      `json:"title"`
	Description     string      `json:"description"`
	AuthorName      string      `json:"author_name"`
	AuthorURL       string      `json:"author_url,omitempty"`
	AuthorThumbnail string      `json:"author_thumbnail_url,omitempty"`
	AuthorID        string      `json:"author_id,omitempty"`
	AuthorLogin     string      `json:"author_login,omitempty"`
	ThumbnailURL    string      `json:"thumbnail_url"`
	ThumbnailWidth  int         `json:"thumbnail_width,omitempty"`
	ThumbnailHeight int         `json:"thumbnail_height,omitempty"`
	Thumbnails      *Thumbnails `json:"thumbnail_urls,omitempty"`
	PlayerHTML      string      `json:"html"`
	Width           int         `json:"width,omitempty"` // required if HTML or URL are specified
	Height          int         `json:"height,omitempty"`
	ProviderName    string      `json:"provider_name"`
	ProviderURL     string      `json:"provider_url,omitempty"`
	URL             string      `json:"url,omitempty"`

	// Twitch VOD/Clips specific
	CreatedAt       *time.Time `json:"created_at"`
	Game            string     `json:"game"`
	VideoLength     int        `json:"video_length"`
	ViewCount       int        `json:"view_count,omitempty"`
	TwitchType      string     `json:"twitch_type"`
	TwitchURLType   string     `json:"twitch_url_type,omitempty"`
	TwitchContentID string     `json:"twitch_content_id,omitempty"`
	TwitchEntity    string     `json:"twitch_entity"`
	StartTime       *time.Time `json:"start_time"`
	EndTime         *time.Time `json:"end_time"`

	// Twitch Clips specific
	CuratorName string `json:"curator_name,omitempty"`
	CuratorURL  string `json:"curator_url,omitempty"`

	// Twitch Stream specific
	StreamType string `json:"stream_type,omitempty"`
}

// Thumbnails provides additional thumbnail URLs
type Thumbnails struct {
	Large  string `json:"large,omitempty"`
	Medium string `json:"medium,omitempty"`
	Small  string `json:"small,omitempty"`
	Tiny   string `json:"tiny,omitempty"`
}

// Oembed  contains embed information for non-twitch content eg: YouTube, Vimeo, Reddit
//
// Note: The Oembed spec can be found at: http://oembed.com
type Oembed struct {
	Type         string        `json:"type"`
	Version      string        `json:"version"` // Must always be 1.0
	Title        string        `json:"title"`
	AuthorName   string        `json:"author_name"`
	AuthorURL    string        `json:"author_url"`
	ProviderName string        `json:"provider_name"`
	ProviderURL  string        `json:"provider_url"`
	CacheAge     time.Duration `json:"cache_age"`

	// optional fields
	HTML   string `json:"html"`
	URL    string `json:"url"`
	Width  int    `json:"width"`
	Height int    `json:"height"`

	// thumbnail fields
	ThumbnailURL    string `json:"thumbnail_url"`
	ThumbnailWidth  int    `json:"thumbnail_width"`
	ThumbnailHeight int    `json:"thumbnail_height"`
}

// Config configures shine http client
type Config struct {
	Host func() string
}

func (c *ClientImpl) http(ctx context.Context, statName string, method string, path string, queryParams url.Values, body interface{}, reqOpts *twitchclient.ReqOpts, into interface{}) error {
	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       defaultStatPrefix + statName,
		StatSampleRate: defaultStatSampleRate,
	})

	doer := &twitchdoer.TwitchHTTPDoer{
		Client: c.Client,
		Reqopt: combinedReqOpts,
	}
	return clients.DoHTTP(ctx, doer, method, path, queryParams, body, into, doer.NewTwitchRequest)
}

// GetEmbedOptions specifies the optional parameters of the GetEmbed function
type GetEmbedOptions struct {
	Autoplay  *bool   `url:"autoplay"`
	Providers *string `url:"providers"`
}

// GetEmbed grabs embed information for a provided URL
func (c *ClientImpl) GetEmbed(ctx context.Context, embedURL string, options *GetEmbedOptions, reqOpts *twitchclient.ReqOpts) (*Embed, error) {
	var response Embed
	path := "/v1/embed_for_url"
	query, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	query.Add("url", embedURL)

	if err := c.http(ctx, "get_embed", "GET", path, query, nil, reqOpts, &response); err != nil {
		if clients.ErrorCode(err) == http.StatusUnprocessableEntity {
			return nil, nil
		}
		return nil, err
	}
	return &response, nil
}

// GetEmbedForEntity returns embed information for the given entity
func (c *ClientImpl) GetEmbedForEntity(ctx context.Context, ent entity.Entity, options *GetEmbedOptions, reqOpts *twitchclient.ReqOpts) (*Embed, error) {
	var response Embed
	path := "/v1/embed_for_entity"
	vq, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	vq.Add("entity", ent.Encode())

	if err := c.http(ctx, "get_embed_for_entity", "GET", path, vq, nil, reqOpts, &response); err != nil {
		if clients.ErrorCode(err) == http.StatusUnprocessableEntity {
			return nil, nil
		}
		return nil, err
	}
	return &response, nil
}

// URLAndEntity contains the entity associated with URL
type URLAndEntity struct {
	URL    string        `json:"url"`
	Entity entity.Entity `json:"entity"`
}

// EntitiesForURLs captures data returned by GetEntitiesForURLs
type EntitiesForURLs struct {
	Entities []URLAndEntity `json:"entities"`
}

// getEntitiesForURLsParams contains the urls to create the request body for GetEntitiesForURLs
type getEntitiesForURLsParams struct {
	URLs []string `json:"urls"`
}

// GetEntitiesForURLs returns entites for given urls
// GetEntitiesForURLs returns an error if urls does not contain atleast one url
//
// NOTE: Any url which can't be converted to an entity is silently dropped in the response
func (c *ClientImpl) GetEntitiesForURLs(ctx context.Context, urls []string, reqOpts *twitchclient.ReqOpts) (*EntitiesForURLs, error) {
	var response EntitiesForURLs
	path := "/v1/entities_for_urls"
	body := getEntitiesForURLsParams{URLs: urls}
	if err := c.http(ctx, "get_entities_for_urls", "POST", path, nil, body, reqOpts, &response); err != nil {
		return nil, err
	}
	return &response, nil
}

// URLAndOembed contains the embed associated with an URL
type URLAndOembed struct {
	URL    string  `json:"url"`
	Oembed *Oembed `json:"oembed"`
}

// OembedsForURLs contains a data returned by getEmbedsForURLs
type OembedsForURLs struct {
	Oembeds []URLAndOembed `json:"oembeds"`
}

// GetOembedsForURLsOptions contains optional parameters that are passed in to GetOembedsForURLs
type GetOembedsForURLsOptions struct {
	Providers *string `url:"providers"`
}

// getEmbedsForURLsParams contains the urls to create the request body for GetOembedsForURLs
type getOembedsForURLsParams struct {
	URLs []string `json:"urls"`
}

// GetOembedsForURLs returns oembeds for given urls
// GetOembedsForURLs can handle a maximum of 100 URLs per function call
// GetOembedsForURLs returns an error if the above limit is violated.
// GetOembedsForURLs returns an error if urls does not contain atleast one url
//
// NOTE: Any url which can't be converted to an oembed is silently dropped in the response
func (c *ClientImpl) GetOembedsForURLs(ctx context.Context, urls []string, opts *GetOembedsForURLsOptions, reqOpts *twitchclient.ReqOpts) (*OembedsForURLs, error) {
	var response OembedsForURLs
	path := "/v1/oembeds_for_urls"
	vq, err := query.Values(opts)
	if err != nil {
		return nil, err
	}
	body := getOembedsForURLsParams{URLs: urls}
	if err := c.http(ctx, "get_oembeds_for_urls", "POST", path, vq, body, reqOpts, &response); err != nil {
		return nil, err
	}
	return &response, nil
}
