const _ = require('lodash');
const moment = require('moment');
const AWS = require('aws-sdk');
const pgp = require('pg-promise')();
const rateLimit = require('rate-limit-promise');

const TIMESTAMPTZ_OID = 1184; pgp.pg.types.setTypeParser(TIMESTAMPTZ_OID, _.identity); // pass string through directly
const TIMESTAMP_OID = 1114;   pgp.pg.types.setTypeParser(TIMESTAMP_OID, _.identity); // pass string through directly

class DuploDataValidation {
    constructor(opts) {
        this.opts = opts;
        this.duploDynamoDB = new AWS.DynamoDB(opts.duplo.dynamoDB);
        this.audreyRDS = pgp(opts.audrey.rds);
        this.audreyGUIDCache = {};

        this.validatePosts = this.makeValidator(this.validatePostsBatch, this.getAudreyPostsCount);
        this.validateComments = this.makeValidator(this.validateCommentsBatch, this.getAudreyCommentsCount);
        this.validateReactions = this.makeValidator(this.validateReactionsBatch, this.getAudreyReactionsCount);

        this.duploRateLimit = _.reduce(opts.duplo.rateLimit, (obj, {count, durationMs}, table) => {
            obj[table] = rateLimit(count, durationMs);
            return obj;
        }, {});
    }
    aggregate(audreyItems, duploItems, audreyForDuploKey='duplo_guid', duploKey='id.S') {
        let duploItemsByID = _.keyBy(duploItems, duploKey);
        return audreyItems.map(audreyItem => ({
            audrey: audreyItem,
            duplo: duploItemsByID[_.get(audreyItem, audreyForDuploKey)]
        }));
    }

    validate() {
        // Must be serial as comments depend on cached posts
        return this.validatePosts().then(invalidPosts => (
            this.validateComments().then(invalidComments => (
                this.validateReactions().then(invalidReactions => [
                    {type: 'posts', invalid: invalidPosts},
                    {type: 'comments', invalid: invalidComments},
                    {type: 'reactions', invalid: invalidReactions},
                ])
            ))
        ));
    }

    validatePostsBatch(cursor) {
        return this.getAudreyPostsBatch(cursor)
            .then(audreyPosts => (
                Promise
                    .all(_(audreyPosts).map('duplo_guid').chunk(this.opts.duplo.maxBatchSize).map(this.getDuploPostsByID.bind(this)).value())
                    .then(duploPostsBatches => this.aggregate(audreyPosts, _.flatten(duploPostsBatches)))
            ))
            .then(posts => ({
                cursor: _.get(_.last(posts), 'audrey.id'),
                checkedCount: posts.length,
                invalid: posts.filter(({audrey, duplo}) => {
                    this.audreyGUIDCache[`post:${audrey.id}`] = audrey.duplo_guid;
                    if (!(duplo
                        && moment.utc(audrey.created_at).isSame(parseInt(duplo.created_at.N)/1000000, 'second')
                        && ((!audrey.deleted_at && !duplo.deleted_at) || moment.utc(audrey.deleted_at).isSame(parseInt(duplo.deleted_at.N)/1000000, 'second'))
                        && audrey.id === duplo.audrey_id.S
                        && audrey.duplo_guid === duplo.id.S
                        && audrey.content === duplo.body.S
                        && audrey.channel_id.toString() === duplo.user_id.S
                    )) {
                        return true;
                    }
                })
            }));
    }

    getAudreyPostsCount() {
        let {ignoreAfter} = this.opts;
        return this.audreyRDS.one('SELECT COUNT(*) FROM posts WHERE duplo_guid IS NOT NULL AND created_at < $(ignoreAfter)', {ignoreAfter}).then(({count}) => parseInt(count));
    }
    getAudreyPostsBatch(cursor, count=this.opts.audrey.batchSize) {
        /*
            { id: '7441321461010513',
              channel_id: 744132,
              created_at: 2016-04-19T03:15:13.419Z,
              deleted_at: null,
              content: 'Testy testy testicle',
              reply_permission: 'public',
              reactions3: null,
              reactions: { endorse: 1 },
              duplo_guid: 'fe378b80110ea3f164160ed83d95d440' }
        */
        if (this.opts.debug) console.log(`Getting ${count} Audrey posts${cursor ? ` after ${cursor}` : ""}`);
        let {ignoreAfter} = this.opts;
        return this.audreyRDS.any(`SELECT * FROM posts WHERE duplo_guid IS NOT NULL AND created_at < $(ignoreAfter) ${cursor ? 'AND id > $(cursor)' : ''} ORDER BY id LIMIT $(count)`, {cursor, count, ignoreAfter});
    }
    getDuploPostsByID(ids) {
        /*
            { created_at: { N: '1454631994089350000' },
              id: { S: '3428f972-b305-48d7-a11a-dbeab70e99be' },
              deleted_at: { N: '1454701812561391000' },
              body: { S: 'fdsafdsa' },
              user_id: { S: '104447238' },
              audrey_id: { S: '93' } }
        */
        return this.getDuploTableItemsByKeys(this.opts.duplo.tableNames.posts, ids.map(id => ({id: {S: id}})));
    }

    validateCommentsBatch(cursor) {
        return this.getAudreyCommentsBatch(cursor)
            .then(audreyComments => (
                Promise
                    .all(_(audreyComments).map('duplo_guid').chunk(this.opts.duplo.maxBatchSize).map(this.getDuploCommentsByID.bind(this)).value())
                    .then(duploCommentsBatches => this.aggregate(audreyComments, _.flatten(duploCommentsBatches)))
            ))
            .then(comments => ({
                cursor: _.get(_.last(comments), 'audrey.id'),
                checkedCount: comments.length,
                invalid: comments.filter(({audrey, duplo}) => {
                    this.audreyGUIDCache[`comment:${audrey.id}`] = audrey.duplo_guid;
                    if (!(duplo
                        && moment.utc(audrey.created_at).isSame(parseInt(duplo.created_at.N)/1000000, 'second')
                        && ((!audrey.deleted_at && !duplo.deleted_at) || moment.utc(audrey.deleted_at).isSame(parseInt(duplo.deleted_at.N)/1000000, 'second'))
                        && audrey.id === duplo.audrey_id.S
                        && audrey.duplo_guid === duplo.id.S
                        && audrey.content === duplo.body.S
                        && audrey.user_id.toString() === duplo.user_id.S
                        && `post:${this.audreyGUIDCache[`post:${audrey.post_id}`]}` === duplo.parent_entity.S
                    )) {
                        return true;
                    }
                })
            }));
    }

    getAudreyCommentsCount() {
        let {ignoreAfter} = this.opts;
        return this.audreyRDS.one('SELECT COUNT(*) FROM comments WHERE duplo_guid IS NOT NULL AND created_at < $(ignoreAfter)', {ignoreAfter}).then(({count}) => parseInt(count));
    }
    getAudreyCommentsBatch(cursor, count=this.opts.audrey.batchSize) {
        /*
            { id: '65',
              post_id: '953091221463432891',
              parent_comment_id: null,
              channel_id: 95309122,
              user_id: 95309122,
              created_at: 2016-05-17T04:08:21.596Z,
              deleted_at: null,
              content: 'fdsfdsfdsa',
              reactions: {},
              duplo_guid: 'c384ec148bcd748579a703584e2eb4ab' }
        */
        if (this.opts.debug) console.log(`Getting ${count} Audrey comments${cursor ? ` after ${cursor}` : ""}`);
        let {ignoreAfter} = this.opts;
        return this.audreyRDS.any(`SELECT * FROM comments WHERE duplo_guid IS NOT NULL AND created_at < $(ignoreAfter) ${cursor ? 'AND id > $(cursor)' : ''} ORDER BY id LIMIT $(count)`, {cursor, count, ignoreAfter});
    }
    getDuploCommentsByID(ids) {
        /*
            { id: { S: '545fa70e-e8e5-4098-bc77-1ae8f6aa83ab' },
              created_at: { N: '1464381009236986000' },
              deleted_at: { N: '1464381024846355000' },
              body: { S: 'fdsafdsa' },
              user_id: { S: '95309122' },
              parent_entity: { S: 'post:224831bf-9fdb-41e1-a1b4-5e51de92b807' },
              audrey_id: { S: '159' } }
        */
        return this.getDuploTableItemsByKeys(this.opts.duplo.tableNames.comments, ids.map(id => ({id: {S: id}})));
    }

    validateReactionsBatch(cursor) {
        return this.getAudreyReactionsBatch(cursor).then(audreyReactions => {
            if (!audreyReactions.length) return {cursor: null, checkedCount: 0, invalid: []};
            let currentDuploGUID = _.first(audreyReactions).duplo_guid;
            let invalid = audreyReactions.filter(({id, post_id, parent_comment_id, duplo_guid}) => {
                this.audreyGUIDCache[`reaction:${id}`] = duplo_guid; // unused
                return duplo_guid !== (parent_comment_id
                    ? `comment:${this.audreyGUIDCache[`comment:${parent_comment_id}`]}`
                    : `post:${this.audreyGUIDCache[`post:${post_id}`]}`
                );
            }).map(audrey => ({audrey, duplo: null}));
            return Promise.all([
                this.getDuploReactionsByKeys(_(audreyReactions).map('user_id').uniq().map(user_id => ({
                    parent_entity: {S: currentDuploGUID},
                    user_id: {S: user_id.toString()},
                })).value()).then(duploReactions => {
                    let audreyReactionsGroupedByDuploReactionsKey = _.groupBy(audreyReactions, r => `${r.user_id}:${r.duplo_guid}`);
                    duploReactions.forEach(dr => {
                        let key = `${dr.user_id.S}:${dr.parent_entity.S}`;
                        let audreyReactionsForDuploReaction = audreyReactionsGroupedByDuploReactionsKey[key];
                        if (!_(audreyReactionsForDuploReaction).map('emote_id').xor(dr.emote_ids.SS).isEmpty()) {
                            invalid.push({duplo: dr, audrey: audreyReactionsForDuploReaction});
                        }
                    });
                }),
                this.getDuploReactionSummariesByKeys([{parent_entity: {S: currentDuploGUID}}]).then(([drs]) => {
                    if (!_(drs)
                        .omit('parent_entity')
                        .mapKeys((v, k) => k.replace(/^emote_/, ''))
                        .mapValues(({N}) => parseInt(N))
                        .isEqual(_(audreyReactions).groupBy('emote_id').mapValues('length').value())
                    ) {
                        invalid.push({duplo: drs, audrey: audreyReactions});
                    }
                }),
            ]).then(() => ({
                cursor: currentDuploGUID,
                checkedCount: audreyReactions.length,
                invalid,
            }))
        });
    }

    getAudreyReactionsCount() {
        let {ignoreAfter} = this.opts;
        return this.audreyRDS.one('SELECT COUNT(*) FROM reactions WHERE duplo_guid IS NOT NULL AND created_at < $(ignoreAfter)', {ignoreAfter}).then(({count}) => parseInt(count));
    }
    getAudreyReactionsBatch(cursor) {
        /*
            { id: 2189,
              post_id: '1233331071464284000',
              parent_comment_id: null,
              channel_id: 123333107,
              user_id: 49709777,
              created_at: '2016-06-02 20:47:50.820563',
              emote_id: '89411',
              duplo_guid: '932f4f69-ae3f-4c15-83c6-1ac9c1c5fb2a' }
        */
        if (this.opts.debug) console.log(`Getting Audrey reactions${cursor ? ` after ${cursor}` : ""}`);
        let {ignoreAfter} = this.opts;
        return this.audreyRDS.any(`SELECT * FROM reactions r WHERE r.duplo_guid IS NOT NULL AND r.duplo_guid = (SELECT MIN(r2.duplo_guid) FROM reactions r2 WHERE created_at < $(ignoreAfter) ${cursor ? 'AND r2.duplo_guid > $(cursor)' : ''})`, {cursor, ignoreAfter});
    }
    getDuploReactionsByKeys(keys) {
        /*
            { user_id: { S: '49709777' },
              emote_ids: { SS: [ '89411' ] },
              parent_entity: { S: '932f4f69-ae3f-4c15-83c6-1ac9c1c5fb2a' } }
        */
        return this.getDuploTableItemsByKeys(this.opts.duplo.tableNames.reactions, keys);
    }
    getDuploReactionSummariesByKeys(keys) {
        /*
            { emote_endorse: { N: '1' },
              parent_entity: { S: '008d88bd-992c-4cb2-9953-b4e5b4eb4042' } }
        */
        return this.getDuploTableItemsByKeys(this.opts.duplo.tableNames.reactionSummaries, keys);
    }

    getDuploTableItemsByKeys(table, keys) {
        if (this.opts.debug) {
            let c = this.opts.displayIDCount;
            console.log(`Getting Duplo items from ${table}: ${keys.slice(0, c).map(JSON.stringify).join(", ")}${keys.length > c ? ` and ${keys.length - c} other${keys.length - c === 1 ? "" : "s"}` : ""}`);
        }
        if (!keys.length) return Promise.resolve([]);
        return Promise
            .resolve(table in this.duploRateLimit && Promise.all(_.times(keys.length, this.duploRateLimit[table])))
            .then(() => this.duploDynamoDB.batchGetItem({
                RequestItems: {
                    [table]: {Keys: keys}
                }
            }).promise())
            .then(({Responses}) => Responses[table]);
    }

    makeValidator (validateBatch, getCount) {
        validateBatch = this.getFn(validateBatch);
        getCount = this.getFn(getCount);
        return function validate (curs, total=0) {
            return validateBatch(curs).then(({checkedCount, invalid, cursor}) => {
                total += checkedCount;
                if (this.opts.debug) {
                    if (invalid.length) throw _.merge(new Error("debug mode, breaking on first invalid data"), {invalid});
                    console.log(`Checked ${total} pairs${curs ? ` up to ${curs}` : ""}${cursor ? "" : " (final batch)"}`);
                }
                return cursor
                    ? validate.call(this, cursor, total).then(newInvalid => [...invalid, ...newInvalid])
                    : getCount().then(count => (
                        total === count
                            ? invalid
                            : [...invalid, {total, count}]
                    ))
            });
        }.bind(this);
    }

    getFn (fn) {
        if (typeof fn === 'string') fn = this[fn];
        return fn.bind(this);
    }
}

module.exports = DuploDataValidation;
