package activity

import (
	"bytes"
	"encoding/json"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sns"
	"golang.org/x/net/context"
)

// Publisher publishes out an activity
type Publisher struct {
	SNS    *sns.SNS
	Config *Config
	Log    *log.ElevatedLog
}

// Config is the configuration for a publisher
type Config struct {
	TopicARN *distconf.Str
}

// Activity is the json message for an activity
type Activity struct {
	Entity entity.Entity `json:"entity"`
	Verb   verb.Verb     `json:"verb"`
	Actor  entity.Entity `json:"actor"`
}

// Verify verifies the configuration
func (c *Config) Verify(prefix string, dconf *distconf.Distconf) error {
	c.TopicARN = dconf.Str(prefix+"_topic_arn", "")
	if c.TopicARN.Get() == "" {
		return errors.Errorf("unable to find %v_topic_arn", prefix)
	}
	return nil
}

// Publish pushes out an activity
func (a *Publisher) Publish(ctx context.Context, entity entity.Entity, verb verb.Verb, actor entity.Entity) error {
	activity := &Activity{
		Entity: entity,
		Verb:   verb,
		Actor:  actor,
	}

	buf := &bytes.Buffer{}
	if err := json.NewEncoder(buf).Encode(activity); err != nil {
		return errors.Wrap(err, "could not marshall activity")
	}
	message := buf.String()

	input := &sns.PublishInput{
		Message: &message,
		MessageAttributes: map[string]*sns.MessageAttributeValue{
			"event": {
				DataType:    aws.String("String"),
				StringValue: aws.String("feed-comment-activity"),
			},
		},
		TopicArn: aws.String(a.Config.TopicARN.Get()),
	}

	req, _ := a.SNS.PublishRequest(input)
	if err := service_common.ContextSend(ctx, req, a.Log); err != nil {
		return errors.Wrap(err, "could not publish activity")
	}
	return nil
}
