package api

import (
	"encoding/json"
	"net/http"
	"strconv"
	"strings"
	"sync"
	"time"

	"code.justin.tv/feeds/clients/fanout"
	"code.justin.tv/feeds/clients/feeddataflow"
	"code.justin.tv/feeds/duplo/cmd/duplo/internal/db"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"code.justin.tv/feeds/service-common"
	"goji.io"
	"goji.io/pat"
	"golang.org/x/net/context"
	"golang.org/x/sync/errgroup"
)

// DB is the backing data store for where API stores duplo objects.  Dupdynamo.DB{} is one implementation of this API.
type DB interface {

	// Post CRUD
	CreatePost(ctx context.Context, userID string, body string, createdAt *time.Time, deletedAt *time.Time, audreyID string, emotes []db.Emote, embedURLs *[]string, embedEntities *[]entity.Entity) (*db.Post, error)
	GetPost(ctx context.Context, id string) (*db.Post, error)
	DeletePost(ctx context.Context, id string, deletedAt *time.Time) (*db.Post, error)
	UpdatePost(ctx context.Context, id string, emotes []db.Emote, embedURLs *[]string, embedEntities *[]entity.Entity) error

	// Post finder methods
	GetPostIDByAudreyID(ctx context.Context, audreyID string) (*string, error)
	GetPostsByIDs(ctx context.Context, ids []string) ([]*db.Post, error)
	GetPostIDsByUser(ctx context.Context, userID string, limit int64, cursor string) (*db.PostIDsResponse, error)

	// Share CRUD
	CreateShare(ctx context.Context, userID string, entity entity.Entity) (*db.Share, error)
	GetShare(ctx context.Context, shareID string) (*db.Share, error)
	DeleteShare(ctx context.Context, id string) (*db.Share, error)
	GetSharesByIDs(ctx context.Context, ids []string) ([]*db.Share, error)
	GetShareSummary(ctx context.Context, parentEntity entity.Entity) (*db.ShareSummary, error)
	GetSharesByTargetEntity(ctx context.Context, parentEntity entity.Entity, userID string) ([]*db.Share, error)

	// Comment CRUD
	CreateComment(ctx context.Context, userID string, parentEntity entity.Entity, body string, createdAt *time.Time, deletedAt *time.Time, audreyID string, needsApproval bool) (*db.Comment, error)
	GetComment(ctx context.Context, id string) (*db.Comment, error)
	UpdateComment(ctx context.Context, id string, needsApproval *bool, emotes []db.Emote) (*db.Comment, error)
	DeleteComment(ctx context.Context, id string, deletedAt *time.Time) (*db.Comment, error)

	// Comment batch update methods
	DeleteCommentsByParentAndUser(ctx context.Context, parentEntity entity.Entity, userID string) error

	// Comment finder methods
	BatchGetComments(ctx context.Context, ids []string) ([]*db.Comment, error)
	GetCommentIDByAudreyID(ctx context.Context, audreyID string) (*string, error)
	GetCommentIDsByParent(ctx context.Context, parentEntity entity.Entity, limit int64, cursor string) (*db.CommentIDsResponse, error)

	// Comment Summary Read
	BatchGetCommentsSummary(ctx context.Context, parentEntities []entity.Entity) ([]*db.CommentsSummary, error)

	// Reactions CRUD
	CreateReaction(ctx context.Context, parentEntity entity.Entity, userID string, emoteID string) (*db.Reactions, error)
	GetReactions(ctx context.Context, parentEntity entity.Entity, userID string) (*db.Reactions, error)
	DeleteReaction(ctx context.Context, parentEntity entity.Entity, userID string, emoteID string) (*db.Reactions, error)

	// Reactions finder methods
	BatchGetReactions(ctx context.Context, parentEntities []entity.Entity, userID string) ([]*db.Reactions, error)

	// Reaction Summary Read
	GetReactionsSummary(ctx context.Context, parentEntity entity.Entity) (*db.ReactionsSummary, error)

	// Reaction Summary finder methods
	BatchGetReactionsSummary(ctx context.Context, parentEntities []entity.Entity, stronglyConsistent bool) ([]*db.ReactionsSummary, error)
}

// ActivityPublisher sends out activities
type ActivityPublisher interface {
	// Publish an activity
	Publish(ctx context.Context, entity entity.Entity, verb verb.Verb, actor entity.Entity) error
}

// HTTPServer handles responding to API requests via HTTP
type HTTPServer struct {
	service_common.BaseHTTPServer
	DB              DB
	Fanout          *fanout.AsyncClient
	CommentActivity ActivityPublisher
}

// DuploRoutes are all the duplo specific goji routes
func (s *HTTPServer) DuploRoutes(mux *goji.Mux) {
	// Post CRUD
	mux.Handle(pat.Post("/v1/posts"), s.CreateHandler("create_post", s.createPost))
	mux.Handle(pat.Get("/v1/posts/:post_id"), s.CreateHandler("get_post", s.getPost))
	mux.Handle(pat.Delete("/v1/posts/:post_id"), s.CreateHandler("delete_post", s.deletePost))
	mux.Handle(pat.Put("/v1/posts/:post_id"), s.CreateHandler("update_post", s.updatePost))

	// Share CRUD
	// return model.Share
	// /v1/shares is REST
	mux.Handle(pat.Post("/v1/shares"), s.CreateHandler("create_share", s.createShare))
	mux.Handle(pat.Get("/v1/shares"), s.CreateHandler("get_shares", s.getSharesByIDs))
	mux.Handle(pat.Get("/v1/shares/:share_id"), s.CreateHandler("get_share", s.getShare))
	mux.Handle(pat.Delete("/v1/shares/:share_id"), s.CreateHandler("delete_share", s.deleteShare))

	// /v1/shares_by_author is RPC
	mux.Handle(pat.Get("/v1/shares_by_author"), s.CreateHandler("get_shares_by_author", s.getSharesByAuthor))

	// /v1/shares_summary is RPC
	mux.Handle(pat.Get("/v1/shares_summaries"), s.CreateHandler("get_shares_summaries", s.getSharesSummaries))

	// Post finder methods
	mux.Handle(pat.Get("/v1/posts"), s.CreateHandler("get_posts", s.getPostsByIDs))
	mux.Handle(pat.Get("/v1/posts/legacy/:audrey_id/id"), s.CreateHandler("get_post", s.getPostIDByAudreyID))
	mux.Handle(pat.Get("/v1/posts/ids_by_user/:user_id"), s.CreateHandler("posts_by_user", s.getPostIDsByUser))

	// Comments CRUD
	mux.Handle(pat.Post("/v1/comments"), s.CreateHandler("create_comment", s.createComment))
	mux.Handle(pat.Get("/v1/comments/summary"), s.CreateHandler("get_comments_summaries", s.getCommentsSummaries))
	mux.Handle(pat.Get("/v1/comments/:comment_id"), s.CreateHandler("get_comment", s.getComment))
	mux.Handle(pat.Put("/v1/comments/:comment_id"), s.CreateHandler("update_comment", s.updateComment))
	mux.Handle(pat.Delete("/v1/comments/:comment_id"), s.CreateHandler("delete_comment", s.deleteComment))

	// Comments batch update methods
	mux.Handle(pat.Delete("/v1/comments/by_parent_and_user/:parent_entity/:user_id"), s.CreateHandler("delete_comments_by_parent_and_user", s.deleteCommentsByParentAndUser))

	// Comments finder methods
	mux.Handle(pat.Get("/v1/comments"), s.CreateHandler("get_comments_by_parent", s.getCommentsByParent))
	mux.Handle(pat.Get("/v1/comments/legacy/:audrey_id/id"), s.CreateHandler("get_post", s.getCommentIDByAudreyID))

	// Reaction summaries
	mux.Handle(pat.Get("/v1/reactions/summary/:parent_entity"), s.CreateHandler("get_reactions_summary", s.getReactionsSummary))

	// Reaction finder methods
	mux.Handle(pat.Get("/v1/reactions/summary"), s.CreateHandler("get_reactions_summaries", s.getReactionsSummaries))
	mux.Handle(pat.Get("/v1/reactions/user_reactions/:user_id"), s.CreateHandler("reactions_by_user_id", s.getUserReactions))

	// Reactions CRUD
	mux.Handle(pat.Get("/v1/reactions/:parent_entity/:user_id"), s.CreateHandler("get_reactions", s.getReactions))
	mux.Handle(pat.Put("/v1/reactions/:parent_entity/:user_id/:emote_id"), s.CreateHandler("create_reaction", s.createReaction))
	mux.Handle(pat.Delete("/v1/reactions/:parent_entity/:user_id/:emote_id"), s.CreateHandler("delete_reaction", s.deleteReaction))
}

func (s *HTTPServer) getPostIDByAudreyID(r *http.Request) (interface{}, error) {
	audreyID := pat.Param(r, "audrey_id")

	id, err := s.DB.GetPostIDByAudreyID(r.Context(), audreyID)
	if err != nil {
		return nil, err
	} else if id == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("audrey_id not found"),
		}
	}

	return newIDFromString(*id), nil
}

// CheckAWS makes sure we have table permissions for the DB
func (s *HTTPServer) CheckAWS(r *http.Request) error {
	// Should return nil,nil if we can connect to AWS
	_, err := s.DB.GetPost(r.Context(), "__invalid_id__")
	return err
}

func (s *HTTPServer) getPost(r *http.Request) (interface{}, error) {
	id := pat.Param(r, "post_id")

	dbPost, err := s.DB.GetPost(r.Context(), id)
	if err != nil {
		return nil, err
	} else if dbPost == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("post not found"),
		}
	}

	post := newPostFromDBPost(dbPost)
	return post, nil
}

// Emote is the emote object for the API
type Emote struct {
	ID    int `json:"id"`
	Start int `json:"start"`
	End   int `json:"end"`
	Set   int `json:"set"`
}

type createPostParams struct {
	UserID        string           `json:"user_id"`
	Body          string           `json:"body"`
	CreatedAt     *time.Time       `json:"created_at"`
	DeletedAt     *time.Time       `json:"deleted_at"`
	AudreyID      string           `json:"audrey_id"`
	Emotes        *[]Emote         `json:"emotes,omitempty"`
	EmbedURLs     *[]string        `json:"embed_urls,omitempty"`
	EmbedEntities *[]entity.Entity `json:"embed_entities,omitempty"`
}

type updatePostParams struct {
	Emotes        *[]Emote         `json:"emotes,omitempty"`
	EmbedURLs     *[]string        `json:"embed_urls,omitempty"`
	EmbedEntities *[]entity.Entity `json:"embed_entities,omitempty"`
}

func dbEmotesFromEmotes(emotes *[]Emote) []db.Emote {
	if emotes == nil || *emotes == nil {
		return nil
	}
	ret := make([]db.Emote, 0, len(*emotes))
	for _, e := range *emotes {
		ret = append(ret, db.Emote(e))
	}
	return ret
}

func verifyCreatePostParams(params createPostParams) error {
	if params.Body == "" && (params.EmbedURLs == nil || len(*params.EmbedURLs) == 0) {
		return &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("post creation requires either a body or embed urls"),
		}
	}
	if params.UserID == "" {
		return &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("post creation requires a user ID"),
		}
	}
	return nil
}

func (s *HTTPServer) createPost(r *http.Request) (interface{}, error) {
	var params createPostParams
	decoder := json.NewDecoder(r.Body)
	if err := decoder.Decode(&params); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "cannot parse request body"),
		}
	}
	if err := verifyCreatePostParams(params); err != nil {
		return nil, err
	}

	dbEmotes := dbEmotesFromEmotes(params.Emotes)

	dbPost, err := s.DB.CreatePost(r.Context(), params.UserID, params.Body, params.CreatedAt, params.DeletedAt, params.AudreyID, dbEmotes, params.EmbedURLs, params.EmbedEntities)
	if err != nil {
		return nil, err
	}

	post := newPostFromDBPost(dbPost)

	postEntity := entity.New(entity.NamespacePost, post.ID)
	metadata := feeddataflow.Metadata{}
	metadata.SetEntityCreationTime(postEntity, dbPost.CreatedAt, time.Now())
	metadata.SetIsDeleted(postEntity, false, time.Now())

	if err = s.Fanout.AddActivity(r.Context(), &fanout.Activity{
		Entity:   postEntity,
		Verb:     verb.Create,
		Actor:    entity.New(entity.NamespaceUser, params.UserID),
		Metadata: &metadata,
	}); err != nil {
		s.Log.Log("err", err, "post_id", post.ID, "user_id", params.UserID, "could not send post:create activity to fanout")
	}
	return post, nil
}

func getEmbedURL(embedURLs *[]string, index int) string {
	if embedURLs == nil || len(*embedURLs) <= index {
		return ""
	}
	return (*embedURLs)[index]
}

type createShareParams struct {
	UserID       string        `json:"user_id"`
	TargetEntity entity.Entity `json:"target_entity"`
}

func (s *HTTPServer) createShare(r *http.Request) (interface{}, error) {
	var params createShareParams
	decoder := json.NewDecoder(r.Body)
	err := decoder.Decode(&params)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "cannot parse request body"),
		}
	}

	dbShare, err := s.DB.CreateShare(r.Context(), params.UserID, params.TargetEntity)
	if err != nil {
		return nil, err
	}

	share := newShareFromDBShare(dbShare)

	shareEntity := entity.New(entity.NamespaceShare, share.ID)
	metadata := feeddataflow.Metadata{}
	metadata.SetEntityCreationTime(shareEntity, dbShare.CreatedAt, time.Now())
	metadata.SetIsDeleted(shareEntity, false, time.Now())

	if err = s.Fanout.AddActivity(r.Context(), &fanout.Activity{
		Entity:   shareEntity,
		Verb:     verb.Create,
		Actor:    entity.New(entity.NamespaceUser, params.UserID),
		Metadata: &metadata,
	}); err != nil {
		s.Log.Log("err", err, "share_id", share.ID, "user_id", params.UserID, "could not send share:create activity to fanout")
	}

	return share, nil
}

func (s *HTTPServer) getShare(r *http.Request) (interface{}, error) {
	shareID := pat.Param(r, "share_id")

	dbShare, err := s.DB.GetShare(r.Context(), shareID)
	if err != nil {
		return nil, err
	} else if dbShare == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("share not found"),
		}
	}

	share := newShareFromDBShare(dbShare)
	return share, nil
}

func entitiesFromURL(r *http.Request, paramName string) ([]entity.Entity, error) {
	targetEntityIDs := itemsFromQueryParam(r, paramName)
	targetEntities := make([]entity.Entity, len(targetEntityIDs))
	for idx, id := range targetEntityIDs {
		targetEntity, err := entity.Decode(id)
		if err != nil {
			return nil, &service_common.CodedError{
				Code: http.StatusBadRequest,
				Err:  errors.Wrapf(err, "malformed entity query parameter %s", paramName),
			}
		}
		targetEntities[idx] = targetEntity
	}
	return targetEntities, nil
}

func (s *HTTPServer) getSharesByAuthor(r *http.Request) (interface{}, error) {
	authorID := r.URL.Query().Get("author_id")
	if authorID == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("missing author_id"),
		}
	}
	targetEntities, err := entitiesFromURL(r, "target_entities")
	if err != nil {
		return nil, err
	}

	var retItemsMu sync.Mutex
	ret := Shares{
		Items: make([]*Share, 0, len(targetEntities)),
	}
	eg, egCtx := errgroup.WithContext(r.Context())
	for _, targetEntity := range targetEntities {
		targetEntity := targetEntity
		eg.Go(func() error {
			dbShares, err := s.DB.GetSharesByTargetEntity(egCtx, targetEntity, authorID)
			if err != nil {
				return err
			}
			retItemsMu.Lock()
			for _, dbShare := range dbShares {
				ret.Items = append(ret.Items, newShareFromDBShare(dbShare))
			}
			retItemsMu.Unlock()
			return nil
		})
	}

	if err := eg.Wait(); err != nil {
		return nil, err
	}

	return ret, nil
}

func (s *HTTPServer) getSharesSummaries(r *http.Request) (interface{}, error) {
	parentEntities, err := entitiesFromURL(r, "parent_entities")
	if err != nil {
		return nil, err
	}

	ret := SharesSummaries{
		Items: make([]*SharesSummary, len(parentEntities)),
	}
	eg, egCtx := errgroup.WithContext(r.Context())
	for idx, targetEntity := range parentEntities {
		idx := idx
		targetEntity := targetEntity
		eg.Go(func() error {
			dbSummary, err := s.DB.GetShareSummary(egCtx, targetEntity)
			if err != nil {
				return err
			}
			if dbSummary == nil {
				dbSummary = &db.ShareSummary{
					ParentEntity: targetEntity,
					Total:        0,
				}
			}
			ret.Items[idx] = newSharesSummaryFromDBSharesSummary(dbSummary)
			return nil
		})
	}
	if err := eg.Wait(); err != nil {
		return nil, err
	}

	return ret, nil
}

func (s *HTTPServer) getSharesByIDs(r *http.Request) (interface{}, error) {
	ids := itemsFromQueryParam(r, "ids")
	if len(ids) == 0 {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("must provide at least one ids"),
		}
	}

	dbShares, err := s.DB.GetSharesByIDs(r.Context(), ids)
	if err != nil {
		return nil, err
	} else if dbShares == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("posts not found"),
		}
	}

	shares := make([]*Share, 0, len(dbShares))
	for _, dbShare := range dbShares {
		if dbShare != nil {
			shares = append(shares, newShareFromDBShare(dbShare))
		}
	}

	ret := Shares{
		Items: shares,
	}

	return ret, nil
}

func (s *HTTPServer) updatePost(r *http.Request) (interface{}, error) {
	postID := pat.Param(r, "post_id")
	var params updatePostParams
	decoder := json.NewDecoder(r.Body)
	err := decoder.Decode(&params)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "cannot parse request body"),
		}
	}

	if err := s.DB.UpdatePost(r.Context(), postID, dbEmotesFromEmotes(params.Emotes), params.EmbedURLs, params.EmbedEntities); err != nil {
		if strings.Contains(err.Error(), "ConditionalCheckFailedException") {
			return nil, &service_common.CodedError{
				Code: http.StatusNotFound,
				Err:  errors.Wrap(err, "post ID does not exist"),
			}
		}
		return nil, err
	}
	return nil, nil
}

func (s *HTTPServer) deleteShare(r *http.Request) (interface{}, error) {
	id := pat.Param(r, "share_id")

	dbShare, err := s.DB.DeleteShare(r.Context(), id)
	if err != nil {
		return nil, err
	} else if dbShare == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("share not found"),
		}
	}

	share := newShareFromDBShare(dbShare)

	shareEntity := entity.New(entity.NamespaceShare, share.ID)
	metadata := feeddataflow.Metadata{}
	metadata.SetEntityCreationTime(shareEntity, dbShare.CreatedAt, time.Now())
	metadata.SetIsDeleted(shareEntity, true, time.Now())

	if err = s.Fanout.AddActivity(r.Context(), &fanout.Activity{
		Entity:   shareEntity,
		Verb:     verb.Delete,
		Actor:    entity.New(entity.NamespaceUser, dbShare.UserID),
		Metadata: &metadata,
	}); err != nil {
		s.Log.Log("err", err, "share_id", share.ID, "user_id", dbShare.UserID, "could not send share:create activity to fanout")
	}

	return share, nil
}

func (s *HTTPServer) deletePost(r *http.Request) (interface{}, error) {
	id := pat.Param(r, "post_id")
	deletedAtParam := r.URL.Query().Get("deleted_at")
	var deletedAt *time.Time
	if len(deletedAtParam) > 0 {
		t, err := time.Parse(time.RFC3339, deletedAtParam)
		if err != nil {
			return nil, &service_common.CodedError{
				Code: http.StatusBadRequest,
				Err:  errors.Wrap(err, "deleted_at malformed"),
			}
		}
		deletedAt = &t
	}

	dbPost, err := s.DB.DeletePost(r.Context(), id, deletedAt)
	if err != nil {
		return nil, err
	} else if dbPost == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("post not found"),
		}
	}

	postEntity := entity.New(entity.NamespacePost, id)
	metadata := feeddataflow.Metadata{}
	metadata.SetEntityCreationTime(postEntity, dbPost.CreatedAt, time.Now())
	metadata.SetIsDeleted(postEntity, true, time.Now())

	if err = s.Fanout.AddActivity(r.Context(), &fanout.Activity{
		Entity:   postEntity,
		Verb:     verb.Delete,
		Actor:    entity.New(entity.NamespaceUser, dbPost.UserID),
		Metadata: &metadata,
	}); err != nil {
		s.Log.Log("err", err, "post_id", id, "user_id", dbPost.UserID, "could not send post:delete activity to fanout")
	}

	post := newPostFromDBPost(dbPost)

	return post, nil
}

func (s *HTTPServer) getPostIDsByUser(r *http.Request) (interface{}, error) {
	userID := pat.Param(r, "user_id")
	cursor := r.URL.Query().Get("cursor")
	limit := parseInt(r.URL.Query().Get("limit"), 100)

	resp, err := s.DB.GetPostIDsByUser(r.Context(), userID, limit, cursor)
	if err != nil {
		return nil, err
	}
	return PaginatedPostIDs{
		PostIDs: resp.PostIDs,
		Cursor:  resp.Cursor,
	}, nil
}

func (s *HTTPServer) getPostsByIDs(r *http.Request) (interface{}, error) {
	ids := itemsFromQueryParam(r, "ids")
	if len(ids) == 0 {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("must provide at least one ids"),
		}
	}

	dbPosts, err := s.DB.GetPostsByIDs(r.Context(), ids)
	if err != nil {
		return nil, err
	} else if dbPosts == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("posts not found"),
		}
	}

	posts := make([]*Post, 0, len(dbPosts))
	for i := 0; i < len(dbPosts); i++ {
		posts = append(posts, newPostFromDBPost(dbPosts[i]))
	}

	ret := Posts{
		Items: posts,
	}
	return ret, nil
}

func (s *HTTPServer) getCommentIDByAudreyID(r *http.Request) (interface{}, error) {
	audreyID := pat.Param(r, "audrey_id")

	id, err := s.DB.GetCommentIDByAudreyID(r.Context(), audreyID)
	if err != nil {
		return nil, err
	} else if id == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("audrey_id not found"),
		}
	}

	return newIDFromString(*id), nil
}

func (s *HTTPServer) getComment(r *http.Request) (interface{}, error) {
	id := pat.Param(r, "comment_id")

	dbComment, err := s.DB.GetComment(r.Context(), id)
	if err != nil {
		return err, nil
	} else if dbComment == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("post not found"),
		}
	}

	comment := newCommentFromDBComment(dbComment)
	return comment, nil
}

func parseInt(num string, defaultValue int64) int64 {
	if num == "" {
		return defaultValue
	}

	conv, err := strconv.ParseInt(num, 10, 64)

	if err != nil {
		return defaultValue
	}

	return conv
}

func (s *HTTPServer) getCommentsByParent(r *http.Request) (interface{}, error) {
	cursor := r.URL.Query().Get("cursor")
	limit := parseInt(r.URL.Query().Get("limit"), 5)

	parentEntity, err := entity.Decode(r.URL.Query().Get("parent_entity"))
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "malformed parent_entity"),
		}
	}

	dbCommentIds, err := s.DB.GetCommentIDsByParent(r.Context(), parentEntity, limit, cursor)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusInternalServerError,
			Err:  err,
		}
	} else if dbCommentIds == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("comments not found"),
		}
	}

	dbComments, err := s.DB.BatchGetComments(r.Context(), dbCommentIds.CommentIds)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusInternalServerError,
			Err:  err,
		}
	} else if dbComments == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("comments not found"),
		}
	}

	comments := make([]*Comment, 0, len(dbComments))
	for i := 0; i < len(dbComments); i++ {
		comments = append(comments, newCommentFromDBComment(dbComments[i]))
	}

	resp := PaginatedComments{
		Items:  comments,
		Cursor: dbCommentIds.Cursor,
	}

	return &resp, nil
}

func (s *HTTPServer) getCommentsSummaries(r *http.Request) (interface{}, error) {
	parentEntities, err := entitiesFromQueryParam(r, "parent_entity")
	if err != nil {
		return nil, err
	}

	dSummaries, err := s.DB.BatchGetCommentsSummary(r.Context(), parentEntities)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusInternalServerError,
			Err:  err,
		}
	}

	summaries := make([]*CommentsSummary, 0, len(dSummaries))
	for _, summary := range dSummaries {
		summaries = append(summaries, newCommentsSummaryFromDBCommentsSummary(summary))
	}

	resp := CommentsSummariesResponse{
		Items: summaries,
	}

	return &resp, nil
}

type createCommentParams struct {
	UserID        string        `json:"user_id"`
	Body          string        `json:"body"`
	ParentEntity  entity.Entity `json:"parent_entity"`
	CreatedAt     *time.Time    `json:"created_at"`
	DeletedAt     *time.Time    `json:"deleted_at"`
	AudreyID      string        `json:"audrey_id"`
	NeedsApproval bool          `json:"needs_approval"`
}

func verifyCreateCommentParams(params createCommentParams) error {
	if params.Body == "" {
		return &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("comment creation requires a body"),
		}
	}
	if params.UserID == "" {
		return &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("comment creation requires a user ID"),
		}
	}
	if params.ParentEntity.Encode() == "" {
		return &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("comment creation requires a parent entity"),
		}
	}
	return nil
}

func (s *HTTPServer) createComment(r *http.Request) (interface{}, error) {
	var params createCommentParams
	decoder := json.NewDecoder(r.Body)
	if err := decoder.Decode(&params); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "could not parse body"),
		}
	}
	if err := verifyCreateCommentParams(params); err != nil {
		return nil, err
	}

	dbComment, err := s.DB.CreateComment(r.Context(), params.UserID, params.ParentEntity, params.Body, params.CreatedAt, params.DeletedAt, params.AudreyID, params.NeedsApproval)
	if err != nil {
		return nil, err
	}

	comment := newCommentFromDBComment(dbComment)

	if publish, err := s.shouldPublishCommentActivity(r.Context(), params.ParentEntity); err != nil {
		s.Log.Log("err", err, "comment_id", comment.ID, "user_id", comment.UserID, "should public check error")
	} else if publish {
		commentEntity := entity.New(entity.NamespaceComment, comment.ID)
		userEntity := entity.New(entity.NamespaceUser, comment.UserID)
		if err := s.CommentActivity.Publish(r.Context(), commentEntity, verb.Create, userEntity); err != nil {
			s.Log.Log("err", err, "comment_id", comment.ID, "user_id", comment.UserID, "could not publish comment:create activity")
		}
	}

	return comment, nil
}

type updateCommentParams struct {
	NeedsApproval *bool    `json:"needs_approval,omitempty"`
	Emotes        *[]Emote `json:"emotes,omitempty"`
}

func (s *HTTPServer) updateComment(r *http.Request) (interface{}, error) {
	id := pat.Param(r, "comment_id")
	var params updateCommentParams
	decoder := json.NewDecoder(r.Body)
	if err := decoder.Decode(&params); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "could not parse body"),
		}
	}

	dbComment, err := s.DB.UpdateComment(r.Context(), id, params.NeedsApproval, dbEmotesFromEmotes(params.Emotes))
	if err != nil {
		if strings.Contains(err.Error(), "comment not found") {
			return nil, &service_common.CodedError{
				Code: http.StatusNotFound,
				Err:  errors.New("comment ID does not exist"),
			}
		}
		return nil, err
	}

	comment := newCommentFromDBComment(dbComment)
	return comment, nil
}

func (s *HTTPServer) shouldPublishCommentActivity(ctx context.Context, parentEntity entity.Entity) (bool, error) {
	if parentEntity.Namespace() == entity.NamespacePost {
		post, err := s.DB.GetPost(ctx, parentEntity.ID())
		if err != nil {
			return false, errors.Wrap(err, "could not get post")
		}
		if post == nil {
			return false, errors.Errorf("post %v not found", parentEntity.ID())
		}
		return true, nil
	}
	return false, nil
}

func (s *HTTPServer) deleteComment(r *http.Request) (interface{}, error) {
	id := pat.Param(r, "comment_id")
	deletedAtParam := r.URL.Query().Get("deleted_at")
	var deletedAt *time.Time
	if len(deletedAtParam) > 0 {
		t, err := time.Parse(time.RFC3339, deletedAtParam)
		if err != nil {
			return nil, &service_common.CodedError{
				Code: http.StatusBadRequest,
				Err:  errors.Wrap(err, "malformed deletedAtParam"),
			}
		}
		deletedAt = &t
	}

	dbComment, err := s.DB.DeleteComment(r.Context(), id, deletedAt)
	if err != nil {
		return nil, err
	} else if dbComment == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("comment not found"),
		}
	}

	comment := newCommentFromDBComment(dbComment)
	return comment, nil
}

func (s *HTTPServer) deleteCommentsByParentAndUser(r *http.Request) (interface{}, error) {
	parentEntity, err := entity.Decode(pat.Param(r, "parent_entity"))
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "malformed parent_entity"),
		}
	}
	userID := pat.Param(r, "user_id")
	if userID == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("must provide a user_id"),
		}
	}
	err = s.DB.DeleteCommentsByParentAndUser(r.Context(), parentEntity, userID)
	return nil, err
}

func (s *HTTPServer) getUserReactions(r *http.Request) (interface{}, error) {
	userID := pat.Param(r, "user_id")
	parentEntities, err := entitiesFromQueryParam(r, "parent_entity")
	if err != nil {
		return nil, err
	}

	dbReaction, err := s.DB.BatchGetReactions(r.Context(), parentEntities, userID)
	if err != nil {
		return nil, err
	} else if dbReaction == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("reactions not found"),
		}
	}

	ret := ReactionList{
		Reactions: make([]*Reactions, 0, len(dbReaction)),
	}
	for _, r := range dbReaction {
		ret.Reactions = append(ret.Reactions, newReactionsFromDBReactions(r))
	}

	return ret, nil
}

func (s *HTTPServer) getReactions(r *http.Request) (interface{}, error) {
	parentEntity, err := entity.Decode(pat.Param(r, "parent_entity"))
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "malformed parent_entity"),
		}
	}

	userID := pat.Param(r, "user_id")

	dbReaction, err := s.DB.GetReactions(r.Context(), parentEntity, userID)
	if err != nil {
		return nil, err
	} else if dbReaction == nil || len(dbReaction.EmoteIDs) == 0 {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("reactions not found"),
		}
	}

	reaction := newReactionsFromDBReactions(dbReaction)
	return reaction, nil
}

func (s *HTTPServer) createReaction(r *http.Request) (interface{}, error) {
	parentEntity, err := entity.Decode(pat.Param(r, "parent_entity"))
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "malformed parent_entity"),
		}
	}

	userID := pat.Param(r, "user_id")
	emoteID := pat.Param(r, "emote_id")
	if len(emoteID) == 0 {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("emote not valid"),
		}
	}

	dbReaction, err := s.DB.CreateReaction(r.Context(), parentEntity, userID, emoteID)
	if err != nil {
		return nil, err
	}

	reaction := newReactionsFromDBReactions(dbReaction)
	return reaction, nil
}

func (s *HTTPServer) deleteReaction(r *http.Request) (interface{}, error) {
	parentEntity, err := entity.Decode(pat.Param(r, "parent_entity"))
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "malformed parent_entity"),
		}
	}

	userID := pat.Param(r, "user_id")
	emoteID := pat.Param(r, "emote_id")

	dbReaction, err := s.DB.DeleteReaction(r.Context(), parentEntity, userID, emoteID)
	if err != nil {
		return nil, err
	} else if dbReaction == nil {
		return nil, &service_common.CodedError{
			Code: http.StatusNotFound,
			Err:  errors.New("reaction not found"),
		}
	}

	reaction := newReactionsFromDBReactions(dbReaction)
	return reaction, nil
}

func (s *HTTPServer) getReactionsSummary(r *http.Request) (interface{}, error) {
	parentEntity, err := entity.Decode(pat.Param(r, "parent_entity"))
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "malformed parent_entity"),
		}
	}

	dbReaction, err := s.DB.GetReactionsSummary(r.Context(), parentEntity)
	if err != nil {
		return nil, err
	}

	var reaction *ReactionsSummary
	if dbReaction == nil {
		reaction = &ReactionsSummary{
			ParentEntity:     parentEntity,
			EmoteSummaries:   map[string]*EmoteSummary{},
			DeprecatedEmotes: map[string]int{},
		}
	} else {
		reaction = newReactionsSummaryFromDBReactionsSummary(dbReaction)
	}

	return reaction, nil
}

func entitiesFromQueryParam(r *http.Request, paramName string) ([]entity.Entity, error) {
	values := itemsFromQueryParam(r, paramName)
	if len(values) == 0 {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("must provide at least one " + paramName),
		}
	}
	entities := make([]entity.Entity, len(values))
	for i, value := range values {
		ent, err := entity.Decode(value)
		if err != nil {
			return nil, &service_common.CodedError{
				Code: http.StatusBadRequest,
				Err:  errors.Wrap(err, "malformed "+paramName),
			}
		}
		entities[i] = ent
	}
	return entities, nil
}

func itemsFromQueryParam(r *http.Request, paramName string) []string {
	items := strings.Split(r.URL.Query().Get(paramName), ",")
	ret := make([]string, 0, len(items))
	for _, val := range items {
		entity := strings.TrimSpace(val)
		if entity != "" {
			ret = append(ret, entity)
		}
	}
	return ret
}

// populateUserReactions is a helper function used by getReactionsSummaries inside a errgroup
func populateUserReactions(ctx context.Context, db DB, parentEntities []entity.Entity, userID string, parentEntityToUserIDToEmotes map[entity.Entity]map[string][]string, mapMu sync.Locker, newMapItemSize int) func() error {
	return func() error {
		var err error
		dbReaction, err := db.BatchGetReactions(ctx, parentEntities, userID)
		if err != nil {
			return err
		}
		// Important: Do map lock after the DB query
		mapMu.Lock()
		defer mapMu.Unlock()
		for _, reaction := range dbReaction {
			if _, exists := parentEntityToUserIDToEmotes[reaction.ParentEntity]; !exists {
				parentEntityToUserIDToEmotes[reaction.ParentEntity] = make(map[string][]string, newMapItemSize)
			}
			parentEntityToUserIDToEmotes[reaction.ParentEntity][userID] = reaction.EmoteIDs
		}
		return nil
	}
}

func (s *HTTPServer) getReactionsSummaries(r *http.Request) (interface{}, error) {
	userIDs := itemsFromQueryParam(r, "user_ids")
	entities, err := entitiesFromQueryParam(r, "parent_entity")
	if err != nil {
		return nil, err
	}

	var stronglyConsistent bool
	if sc := r.Header.Get(StronglyConsistentReadHeader); strings.ToLower(sc) == "true" {
		stronglyConsistent = true
	}
	// NOTE: do not write to stronglyConsistent after this point to avoid race conditions

	errGroup, groupCtx := errgroup.WithContext(r.Context())
	dbReactionsSummaries := []*db.ReactionsSummary{}
	var parentEntityToUserIDToEmotesMu sync.Mutex
	parentEntityToUserIDToEmotes := make(map[entity.Entity]map[string][]string, len(entities))
	errGroup.Go(func() error {
		var err error
		dbReactionsSummaries, err = s.DB.BatchGetReactionsSummary(groupCtx, entities, stronglyConsistent)
		if err != nil {
			return &service_common.CodedError{
				Code: http.StatusInternalServerError,
				Err:  err,
			}
		}
		return nil
	})

	for _, userID := range userIDs {
		errGroup.Go(populateUserReactions(groupCtx, s.DB, entities, userID, parentEntityToUserIDToEmotes, &parentEntityToUserIDToEmotesMu, len(userIDs)))
	}
	if err := errGroup.Wait(); err != nil {
		return nil, err
	}

	resp := &ReactionsSummariesResponse{
		Items: make([]*ReactionsSummary, 0, len(dbReactionsSummaries)),
	}
	for _, dbReactionSummary := range dbReactionsSummaries {
		item := newReactionsSummaryFromDBReactionsSummary(dbReactionSummary)
		for userID, emotes := range parentEntityToUserIDToEmotes[dbReactionSummary.ParentEntity] {
			for _, emote := range emotes {
				if summary, exists := item.EmoteSummaries[emote]; exists {
					summary.UserIDs = append(item.EmoteSummaries[emote].UserIDs, userID)
				}
			}
		}
		resp.Items = append(resp.Items, item)
	}

	return resp, nil
}
