package api

import (
	"time"

	"code.justin.tv/feeds/duplo/cmd/duplo/internal/db"
	"code.justin.tv/feeds/feeds-common/entity"
)

const (
	StronglyConsistentReadHeader = "StronglyConsistentRead"
)

// ID is an ID
type ID struct {
	ID string `json:"id"`
}

// Post is the post object for the API
type Post struct {
	ID        string     `json:"id"`
	UserID    string     `json:"user_id"`
	Body      string     `json:"body"`
	CreatedAt time.Time  `json:"created_at"`
	DeletedAt *time.Time `json:"deleted_at,omitempty"`
	AudreyID  string     `json:"audrey_id,omitempty"`
	Emotes    *[]Emote   `json:"emotes,omitempty"`
	// deprecated.  Use EmbedEntities
	EmbedURLs     *[]string        `json:"embed_urls,omitempty"`
	EmbedEntities *[]entity.Entity `json:"embed_entities,omitempty"`
}

// Share is the returned object for a database share
type Share struct {
	ID           string        `json:"id"`
	UserID       string        `json:"user_id"`
	TargetEntity entity.Entity `json:"target_entity"`
	CreatedAt    time.Time     `json:"created_at"`
	DeletedAt    *time.Time    `json:"deleted_at,omitempty"`
}

// Shares is a wrapper around a list of share results
type Shares struct {
	Items []*Share `json:"items"`
}

// SharesSummaries is a wrapper around a list of ShareSummary results
type SharesSummaries struct {
	Items []*SharesSummary `json:"items"`
}

// Posts is a wrapper around a list of post results
type Posts struct {
	Items []*Post `json:"items"`
}

// Comment is the comment object for the API
type Comment struct {
	ID            string        `json:"id"`
	ParentEntity  entity.Entity `json:"parent_entity"`
	UserID        string        `json:"user_id"`
	Body          string        `json:"body"`
	CreatedAt     time.Time     `json:"created_at"`
	DeletedAt     *time.Time    `json:"deleted_at,omitempty"`
	AudreyID      string        `json:"audrey_id,omitempty"`
	NeedsApproval bool          `json:"needs_approval,omitempty"`
	Emotes        *[]Emote      `json:"emotes,omitempty"`
}

// PaginatedComments is the comment object that is returned for paginated comments
type PaginatedComments struct {
	Items  []*Comment `json:"items"`
	Cursor string     `json:"cursor"`
}

// PaginatedPostIDs is the list of post IDs that can be cursor'd
type PaginatedPostIDs struct {
	PostIDs []string `json:"post_ids"`
	Cursor  string   `json:"cursor"`
}

// PaginatedCommentIDs is the list of comment IDs that can be cursor'd
type PaginatedCommentIDs struct {
	CommentIDs []string `json:"comment_ids"`
	Cursor     string   `json:"cursor"`
}

// SharesSummary contains summary information about an object that may be shared
type SharesSummary struct {
	ParentEntity entity.Entity `json:"parent_entity"`
	Total        int           `json:"total"`
}

// CommentsSummary is the comment object that contains additional meta data for comments
// CommentsSummary is the comment object that contains additional meta data for comments
type CommentsSummary struct {
	ParentEntity entity.Entity `json:"parent_entity"`
	Total        int           `json:"total"`
}

// CommentsSummariesResponse is a wrapper around a list of Comments Summary responses
type CommentsSummariesResponse struct {
	Items []*CommentsSummary `json:"items"`
}

// Reactions is the reaction object for the API
type Reactions struct {
	ParentEntity entity.Entity `json:"parent_entity"`
	UserID       string        `json:"user_id"`
	EmoteIDs     []string      `json:"emote_ids"`
}

// ReactionList is a response envelope for a list of reactions
type ReactionList struct {
	Reactions []*Reactions `json:"reactions"`
}

// ReactionsSummary is the reaction object for the API
type ReactionsSummary struct {
	ParentEntity   entity.Entity            `json:"parent_entity"`
	EmoteSummaries map[string]*EmoteSummary `json:"emote_summaries"`

	// deprecated.  Use EmoteSummaries instead
	DeprecatedEmotes map[string]int `json:"emotes"`
}

// EmoteSummary contains a true count of reactions of an emote type and a subset of the userids that have actually
// reacted
type EmoteSummary struct {
	// The real total number of reactions of this emote type
	Count int `json:"count"`
	// Some of the user IDs that reacted to this emote
	UserIDs []string `json:"user_ids,omitempty"`
}

// ReactionsSummariesResponse is a wrapper around a list of Reactions Summary responses
type ReactionsSummariesResponse struct {
	Items []*ReactionsSummary `json:"items"`
}

func newIDFromString(id string) *ID {
	return &ID{ID: id}
}

func newCommentFromDBComment(dbComment *db.Comment) *Comment {
	return &Comment{
		ID:            dbComment.ID,
		ParentEntity:  dbComment.ParentEntity,
		UserID:        dbComment.UserID,
		Body:          dbComment.Body,
		CreatedAt:     dbComment.CreatedAt,
		DeletedAt:     dbComment.DeletedAt,
		AudreyID:      dbComment.AudreyID,
		NeedsApproval: dbComment.NeedsApproval,
		Emotes:        emotesFromDbEmotes(dbComment.Emotes),
	}
}

func newCommentsSummaryFromDBCommentsSummary(dbCommentsSummary *db.CommentsSummary) *CommentsSummary {
	return &CommentsSummary{
		ParentEntity: dbCommentsSummary.ParentEntity,
		Total:        dbCommentsSummary.Total,
	}
}

func newShareFromDBShare(dbShare *db.Share) *Share {
	if dbShare == nil {
		return nil
	}
	return &Share{
		ID:           dbShare.ID,
		UserID:       dbShare.UserID,
		CreatedAt:    dbShare.CreatedAt,
		DeletedAt:    dbShare.DeletedAt,
		TargetEntity: dbShare.TargetEntity,
	}
}

func newPostFromDBPost(dbPost *db.Post) *Post {
	return &Post{
		ID:            dbPost.ID,
		UserID:        dbPost.UserID,
		Body:          dbPost.Body,
		CreatedAt:     dbPost.CreatedAt,
		DeletedAt:     dbPost.DeletedAt,
		AudreyID:      dbPost.AudreyID,
		Emotes:        emotesFromDbEmotes(dbPost.Emotes),
		EmbedURLs:     dbPost.EmbedURLs,
		EmbedEntities: dbPost.EmbedEntities,
	}
}

func emotesFromDbEmotes(dbEmotes []db.Emote) *[]Emote {
	if dbEmotes == nil {
		return nil
	}
	ret := make([]Emote, 0, len(dbEmotes))
	for _, e := range dbEmotes {
		ret = append(ret, Emote(e))
	}
	return &ret
}

func newReactionsFromDBReactions(dbReactions *db.Reactions) *Reactions {
	return &Reactions{
		ParentEntity: dbReactions.ParentEntity,
		UserID:       dbReactions.UserID,
		EmoteIDs:     dbReactions.EmoteIDs,
	}
}

func newReactionsSummaryFromDBReactionsSummary(dbReactionsSummary *db.ReactionsSummary) *ReactionsSummary {
	summary := &ReactionsSummary{
		ParentEntity:     dbReactionsSummary.ParentEntity,
		DeprecatedEmotes: dbReactionsSummary.Emotes,
	}
	summary.EmoteSummaries = make(map[string]*EmoteSummary, len(dbReactionsSummary.Emotes))
	for emoteName, count := range dbReactionsSummary.Emotes {
		summary.EmoteSummaries[emoteName] = &EmoteSummary{
			Count: count,
		}
	}
	return summary
}

func newSharesSummaryFromDBSharesSummary(dbSummary *db.ShareSummary) *SharesSummary {
	return &SharesSummary{
		ParentEntity: dbSummary.ParentEntity,
		Total:        dbSummary.Total,
	}
}
