package dupdynamo

import (
	"bytes"
	"encoding/base64"
	"encoding/json"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/duplo/cmd/duplo/internal/db"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/service-common/feedsdynamo"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/satori/go.uuid"
	"golang.org/x/net/context"
)

// DB holds the dynamo DB connection
type DB struct {
	Dynamo                 *dynamodb.DynamoDB
	Config                 *Config
	Log                    *log.ElevatedLog
	RequireConsistentReads bool
}

// Config will configure the database
type Config struct {
	PostTableName             *distconf.Str
	ShareTableName            *distconf.Str
	CommentTableName          *distconf.Str
	CommentsSummaryTableName  *distconf.Str
	ReactionTableName         *distconf.Str
	ReactionsSummaryTableName *distconf.Str
	SharesSummaryTableName    *distconf.Str
	ReadOperationTimeout      *distconf.Duration
}

// Load the config from a distconf source
func (c *Config) Load(dconf *distconf.Distconf) error {
	c.ReadOperationTimeout = dconf.Duration("duplo.read_operation_timeout", time.Second)

	c.PostTableName = dconf.Str("duplo.post_table", "")
	if c.PostTableName.Get() == "" {
		return errors.New("unable to find duplo post table")
	}

	c.ShareTableName = dconf.Str("duplo.share_table", "")
	if c.ShareTableName.Get() == "" {
		return errors.New("unable to find duplo share table")
	}

	c.CommentTableName = dconf.Str("duplo.comment_table", "")
	if c.CommentTableName.Get() == "" {
		return errors.New("unable to find duplo comment table")
	}

	c.CommentsSummaryTableName = dconf.Str("duplo.comment_summary_table", "")
	if c.CommentsSummaryTableName.Get() == "" {
		return errors.New("unable to find duplo comment summary table")
	}

	c.ReactionTableName = dconf.Str("duplo.reaction_table", "")
	if c.ReactionTableName.Get() == "" {
		return errors.New("unable to find duplo reaction table")
	}

	c.ReactionsSummaryTableName = dconf.Str("duplo.reaction_summary_table", "")
	if c.ReactionsSummaryTableName.Get() == "" {
		return errors.New("unable to find duplo reaction summary table")
	}

	c.SharesSummaryTableName = dconf.Str("duplo.shares_summary_table", "")
	if c.SharesSummaryTableName.Get() == "" {
		return errors.New("unable to find duplo shares summary table")
	}
	return nil
}

func timeToAttributeValue(t time.Time) *dynamodb.AttributeValue {
	nano := strconv.FormatInt(t.UTC().UnixNano(), 10)
	return &dynamodb.AttributeValue{N: aws.String(nano)}
}

func attributeValueToTime(val *dynamodb.AttributeValue) (*time.Time, error) {
	nano, err := strconv.ParseInt(*val.N, 10, 64)
	if err != nil {
		return nil, err
	}
	t := time.Unix(0, nano).UTC()
	return &t, nil
}

func idToAttributeValueMap(id string) map[string]*dynamodb.AttributeValue {
	key := map[string]*dynamodb.AttributeValue{
		"id": {S: aws.String(id)},
	}
	return key
}

func now() *time.Time {
	t := time.Now().UTC().Truncate(time.Microsecond)
	return &t
}

func attributeValueMapToShare(item map[string]*dynamodb.AttributeValue) (*db.Share, error) {
	createdAt, err := attributeValueToTime(item["created_at"])
	if err != nil {
		return nil, err
	}

	entityObj, err := entity.Decode(*item["target_entity"].S)
	if err != nil {
		return nil, err
	}

	share := &db.Share{
		ID:           *item["id"].S,
		UserID:       *item["user_id"].S,
		TargetEntity: entityObj,
		CreatedAt:    *createdAt,
	}
	if val, ok := item["deleted_at"]; ok {
		deletedAt, err := attributeValueToTime(val)
		if err != nil {
			return nil, err
		}
		share.DeletedAt = deletedAt
	}
	return share, nil
}

func attributeValueMapToPost(item map[string]*dynamodb.AttributeValue) (*db.Post, error) {
	createdAt, err := attributeValueToTime(item["created_at"])
	if err != nil {
		return nil, err
	}

	post := &db.Post{
		ID:        *item["id"].S,
		UserID:    *item["user_id"].S,
		CreatedAt: *createdAt,
	}
	if val, ok := item["body"]; ok {
		post.Body = *val.S
	}
	if val, ok := item["deleted_at"]; ok {
		deletedAt, err := attributeValueToTime(val)
		if err != nil {
			return nil, err
		}
		post.DeletedAt = deletedAt
	}
	if val, ok := item["audrey_id"]; ok {
		post.AudreyID = *val.S
	}

	if val, ok := item["emotes"]; ok {
		post.Emotes = emotesFromDynamo(val)
	}

	if val, ok := item["embed_urls"]; ok {
		post.EmbedURLs = embedURLsFromDynamo(val)
	}

	if val, ok := item["embed_entities"]; ok {
		post.EmbedEntities = embedEntitiesFromDynamo(val)
	}

	return post, nil
}

func (d *DB) setGetItemInputConsistency(ctx context.Context, in *dynamodb.GetItemInput, stronglyConsistent bool) {
	if d.RequireConsistentReads || stronglyConsistent {
		in.ConsistentRead = aws.Bool(true)
	}
}

func (d *DB) setBatchGetItemConsistency(ctx context.Context, in *dynamodb.BatchGetItemInput, stronglyConsistent bool) {
	if d.RequireConsistentReads || stronglyConsistent {
		for _, v := range in.RequestItems {
			v.ConsistentRead = aws.Bool(true)
		}
	}
}

func emotesFromDynamo(val *dynamodb.AttributeValue) []db.Emote {
	if val.M == nil {
		return nil
	}
	emoteAttr, exists := val.M["emotes"]
	if !exists {
		return nil
	}
	emoteList := emoteAttr.L
	if emoteList == nil {
		return nil
	}
	ret := make([]db.Emote, 0, len(emoteList))
	for _, emote := range emoteList {
		emoteMap := emote.M
		if emoteMap == nil {
			return nil
		}
		intVals, err := feedsdynamo.AwsInts(emoteMap, []string{"id", "start", "end", "set"})
		if err != nil {
			return nil
		}
		ret = append(ret, db.Emote{
			ID:    int(intVals["id"]),
			Start: int(intVals["start"]),
			End:   int(intVals["end"]),
			Set:   int(intVals["set"]),
		})
	}
	return ret
}

func embedURLsFromDynamo(val *dynamodb.AttributeValue) *[]string {
	if val.M == nil {
		return nil
	}
	embedAttr, exists := val.M["embed_urls"]
	if !exists {
		return nil
	}
	embedList := embedAttr.L
	if embedList == nil {
		return nil
	}
	ret := make([]string, 0, len(embedList))
	for _, embed := range embedList {
		if embed.S != nil {
			ret = append(ret, *embed.S)
		}
	}
	return &ret
}

func embedEntitiesFromDynamo(val *dynamodb.AttributeValue) *[]entity.Entity {
	if val.M == nil {
		return nil
	}
	embedAttr, exists := val.M["embed_entities"]
	if !exists {
		return nil
	}
	embedList := embedAttr.L
	if embedList == nil {
		return nil
	}
	ret := make([]entity.Entity, 0, len(embedList))
	for _, embed := range embedList {
		if embed.S != nil {
			ent, err := entity.Decode(*embed.S)
			if err == nil {
				ret = append(ret, ent)
			}
		}
	}
	return &ret
}

func emotesToDynamo(emotes []db.Emote, currentTime time.Time) *dynamodb.AttributeValue {
	emoteAttributeValues := make([]*dynamodb.AttributeValue, 0, len(emotes))
	for _, e := range emotes {
		emoteAttributeValues = append(emoteAttributeValues, &dynamodb.AttributeValue{
			M: map[string]*dynamodb.AttributeValue{
				"id": {
					N: aws.String(strconv.Itoa(e.ID)),
				},
				"start": {
					N: aws.String(strconv.Itoa(e.Start)),
				},
				"end": {
					N: aws.String(strconv.Itoa(e.End)),
				},
				"set": {
					N: aws.String(strconv.Itoa(e.Set)),
				},
			},
		})
	}

	return &dynamodb.AttributeValue{
		M: map[string]*dynamodb.AttributeValue{
			"updated_at": timeToAttributeValue(currentTime),
			"emotes": {
				L: emoteAttributeValues,
			},
		},
	}
}

func embedURLsToDynamo(embedURLs *[]string, currentTime time.Time) *dynamodb.AttributeValue {
	urlSlice := *embedURLs
	urlList := make([]*dynamodb.AttributeValue, 0, len(urlSlice))
	for _, url := range urlSlice {
		urlList = append(urlList, &dynamodb.AttributeValue{S: aws.String(url)})
	}

	return &dynamodb.AttributeValue{
		M: map[string]*dynamodb.AttributeValue{
			"updated_at": timeToAttributeValue(currentTime),
			"embed_urls": {
				L: urlList,
			},
		},
	}
}

func embedEntitiesToDynamo(embedEntities *[]entity.Entity, currentTime time.Time) *dynamodb.AttributeValue {
	entitySlice := *embedEntities
	entityList := make([]*dynamodb.AttributeValue, 0, len(entitySlice))
	for _, entity := range entitySlice {
		entityList = append(entityList, &dynamodb.AttributeValue{S: aws.String(entity.Encode())})
	}

	return &dynamodb.AttributeValue{
		M: map[string]*dynamodb.AttributeValue{
			"updated_at": timeToAttributeValue(currentTime),
			"embed_entities": {
				L: entityList,
			},
		},
	}
}

// UpdatePost updates a post with a new set of emotes
func (d *DB) UpdatePost(ctx context.Context, id string, emotes []db.Emote, embedURLs *[]string, embedEntities *[]entity.Entity) error {
	var expressions []string
	expressionAttrValues := map[string]*dynamodb.AttributeValue{}

	if emotes != nil {
		expressions = append(expressions, "emotes = :emotes")
		expressionAttrValues[":emotes"] = emotesToDynamo(emotes, *now())
	}
	if embedURLs != nil {
		expressions = append(expressions, "embed_urls = :embed_urls")
		expressionAttrValues[":embed_urls"] = embedURLsToDynamo(embedURLs, *now())
	}
	if embedEntities != nil {
		expressions = append(expressions, "embed_entities = :embed_entities")
		expressionAttrValues[":embed_entities"] = embedEntitiesToDynamo(embedEntities, *now())
	}
	if len(expressionAttrValues) == 0 {
		return errors.New("no attributes to update")
	}
	updateExpression := "set " + strings.Join(expressions, ", ")
	key := idToAttributeValueMap(id)

	updateParams := &dynamodb.UpdateItemInput{
		TableName:                 aws.String(d.Config.PostTableName.Get()),
		Key:                       key,
		UpdateExpression:          aws.String(updateExpression),
		ExpressionAttributeValues: expressionAttrValues,
		ConditionExpression:       aws.String("attribute_exists(id) AND attribute_not_exists(deleted_at)"),
	}
	req, _ := d.Dynamo.UpdateItemRequest(updateParams)
	return service_common.ContextSend(ctx, req, d.Log)
}

// UpdateComment sets a comment's needs_approval flag
func (d *DB) UpdateComment(ctx context.Context, id string, needsApproval *bool, emotes []db.Emote) (*db.Comment, error) {
	updateParams, err := d.getUpdateCommentExpression(ctx, id, needsApproval, emotes)
	if err != nil {
		return nil, err
	}

	req, resp := d.Dynamo.UpdateItemRequest(updateParams)
	if err = service_common.ContextSend(ctx, req, d.Log); err != nil {
		if strings.Contains(err.Error(), "ConditionalCheckFailedException") {
			return nil, errors.New("comment not found")
		}
		return nil, err
	}

	oldComment, err := attributeValueMapToComment(resp.Attributes)
	if err != nil {
		return nil, err
	}

	newComment := *oldComment
	newComment.Emotes = emotes

	if needsApproval != nil {
		if err := d.updateCommentSummaryForNeedsApproval(ctx, oldComment.ParentEntity, oldComment.NeedsApproval, *needsApproval); err != nil {
			return nil, err
		}
		newComment.NeedsApproval = *needsApproval
	}

	return &newComment, nil
}

func (d *DB) getUpdateCommentExpression(ctx context.Context, id string, needsApproval *bool, emotes []db.Emote) (*dynamodb.UpdateItemInput, error) {
	expressionAttrValues := map[string]*dynamodb.AttributeValue{}
	updateExpressionClauses := make([]string, 0, 2)

	if needsApproval != nil {
		expressionAttrValues[":needsApproval"] = &dynamodb.AttributeValue{BOOL: needsApproval}
		updateExpressionClauses = append(updateExpressionClauses, "needs_approval = :needsApproval")
	}

	if emotes != nil {
		expressionAttrValues[":emotes"] = emotesToDynamo(emotes, *now())
		updateExpressionClauses = append(updateExpressionClauses, "emotes = :emotes")
	}

	if len(expressionAttrValues) == 0 {
		return nil, errors.New("no attributes to update")
	}

	updateExpression := "set " + strings.Join(updateExpressionClauses, ", ")

	// update comment attributes
	return &dynamodb.UpdateItemInput{
		TableName:                 aws.String(d.Config.CommentTableName.Get()),
		Key:                       idToAttributeValueMap(id),
		UpdateExpression:          aws.String(updateExpression),
		ExpressionAttributeValues: expressionAttrValues,
		ReturnValues:              aws.String("ALL_OLD"),
		ConditionExpression:       aws.String("attribute_exists(id)"),
	}, nil
}

func (d *DB) updateCommentSummaryForNeedsApproval(ctx context.Context, parentEntity entity.Entity, oldNeedsApproval bool, newNeedsApproval bool) error {
	if oldNeedsApproval == newNeedsApproval {
		return nil
	}

	incAmount := "1"
	if newNeedsApproval {
		incAmount = "-1"
	}
	summaryParams := &dynamodb.UpdateItemInput{
		TableName: aws.String(d.Config.CommentsSummaryTableName.Get()),
		Key: map[string]*dynamodb.AttributeValue{
			"parent_entity": {S: aws.String(parentEntity.Encode())},
		},
		UpdateExpression: aws.String("ADD #count :inc_amnt"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":inc_amnt": {N: aws.String(incAmount)},
		},
		ExpressionAttributeNames: map[string]*string{
			"#count": aws.String("count"),
		},
	}
	req, _ := d.Dynamo.UpdateItemRequest(summaryParams)
	return service_common.ContextSend(ctx, req, d.Log)
}

// GetShare returns the share for the given ID
func (d *DB) GetShare(ctx context.Context, id string) (*db.Share, error) {
	item, err := d.keyedGet(ctx, id, d.Config.ShareTableName.Get())
	if err != nil {
		return nil, err
	} else if item == nil {
		return nil, nil
	}

	share, err := attributeValueMapToShare(item)
	if err != nil {
		return nil, err
	}
	if share != nil && share.DeletedAt != nil {
		return nil, nil
	}

	return share, nil
}

func (d *DB) modifySharedByCount(ctx context.Context, parentEntity entity.Entity, delta int) error {
	summaryParams := &dynamodb.UpdateItemInput{
		TableName: aws.String(d.Config.SharesSummaryTableName.Get()),
		Key: map[string]*dynamodb.AttributeValue{
			"parent_entity": {S: aws.String(parentEntity.Encode())},
		},
		UpdateExpression: aws.String("ADD share_count :incr_amnt"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":incr_amnt": {N: aws.String(strconv.Itoa(delta))},
		},
	}

	req, _ := d.Dynamo.UpdateItemRequest(summaryParams)
	return service_common.ContextSend(ctx, req, d.Log)
}

// DeleteShare deletes the share for the given ID
func (d *DB) DeleteShare(ctx context.Context, id string) (*db.Share, error) {

	item, err := d.keyedGet(ctx, id, d.Config.ShareTableName.Get())
	if err != nil {
		return nil, err
	} else if item == nil {
		return nil, nil
	}

	share, err := attributeValueMapToShare(item)
	if err != nil {
		return nil, err
	}
	if share != nil && share.DeletedAt != nil {
		return nil, nil
	}

	share.DeletedAt = now()

	updateParams := &dynamodb.UpdateItemInput{
		TableName:        aws.String(d.Config.ShareTableName.Get()),
		Key:              idToAttributeValueMap(id),
		UpdateExpression: aws.String("set deleted_at = :deleted_at"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":deleted_at": timeToAttributeValue(*share.DeletedAt),
		},
	}

	req, _ := d.Dynamo.UpdateItemRequest(updateParams)
	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return nil, err
	}

	if err := d.modifySharedByCount(ctx, share.TargetEntity, -1); err != nil {
		d.Log.LogCtx(ctx, "err", err, "unable to modify share by count")
	}

	return share, nil
}

// GetSharesByTargetEntity returns all shares that a person has done on an entity, or nil, nil if no such
// share exists.  Usually, this will return a single item.
func (d *DB) GetSharesByTargetEntity(ctx context.Context, targetEntity entity.Entity, userID string) ([]*db.Share, error) {
	// Filters happen after the query.  It's possible everything got filtered but there is more to fetch.  We need
	// to keep looping and evaluating ExclusiveStartKey till it's nil.  This is 100% true for every dynamo query
	// that also uses FilterExpression
	var ExclusiveStartKey map[string]*dynamodb.AttributeValue
	ret := make([]*db.Share, 0, 16)

	// We want to get them all, but just in case we only loop 100 times so we don't get stuck forever
	for i := 0; i < 100; i++ {
		queryParams := &dynamodb.QueryInput{
			TableName:              aws.String(d.Config.ShareTableName.Get()),
			KeyConditionExpression: aws.String("target_entity = :target_entity AND user_id = :user_id"),
			FilterExpression:       aws.String("attribute_not_exists(deleted_at)"),
			IndexName:              aws.String("target_entity-index"),
			ExclusiveStartKey:      ExclusiveStartKey,
			ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
				":target_entity": {S: aws.String(targetEntity.Encode())},
				":user_id":       {S: &userID},
			},
		}

		req, queryOutput := d.Dynamo.QueryRequest(queryParams)
		if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
			return nil, err
		}
		for _, item := range queryOutput.Items {
			asDB, err := attributeValueMapToShare(item)
			if err != nil {
				return nil, err
			}
			ret = append(ret, asDB)
		}
		ExclusiveStartKey = queryOutput.LastEvaluatedKey
		if ExclusiveStartKey == nil {
			break
		}
	}
	return ret, nil
}

// GetShareSummary returns the share summary for the parent entity ID (just count)
func (d *DB) GetShareSummary(ctx context.Context, parentEntity entity.Entity) (*db.ShareSummary, error) {
	getParams := &dynamodb.GetItemInput{
		TableName: aws.String(d.Config.SharesSummaryTableName.Get()),
		Key: map[string]*dynamodb.AttributeValue{
			"parent_entity": {S: aws.String(parentEntity.Encode())},
		},
	}
	d.setGetItemInputConsistency(ctx, getParams, false)

	req, reactionsSummaryItem := d.Dynamo.GetItemRequest(getParams)
	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return nil, err
	} else if reactionsSummaryItem.Item == nil {
		return &db.ShareSummary{
			ParentEntity: parentEntity,
			Total:        0,
		}, nil
	}

	dbReactionsSummary, err := newDBShareSummaryFromDynamoItem(reactionsSummaryItem.Item)

	if err != nil {
		return nil, err
	}

	return dbReactionsSummary, nil
}

// CreateShare creates a share
func (d *DB) CreateShare(ctx context.Context, userID string, entityID entity.Entity) (*db.Share, error) {
	id := uuid.NewV4().String()
	createdAt := now()
	share := &db.Share{
		ID:           id,
		UserID:       userID,
		TargetEntity: entityID,
		CreatedAt:    *createdAt,
	}
	item := map[string]*dynamodb.AttributeValue{
		"id":            {S: aws.String(share.ID)},
		"user_id":       {S: aws.String(share.UserID)},
		"created_at":    timeToAttributeValue(share.CreatedAt),
		"target_entity": {S: aws.String(share.TargetEntity.Encode())},
	}
	params := &dynamodb.PutItemInput{
		TableName: aws.String(d.Config.ShareTableName.Get()),
		Item:      item,
	}

	req, _ := d.Dynamo.PutItemRequest(params)
	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return nil, err
	}

	if err := d.modifySharedByCount(ctx, share.TargetEntity, 1); err != nil {
		d.Log.LogCtx(ctx, "err", err, "unable to modify share by count")
	}

	return share, nil
}

// CreatePost creates a post
func (d *DB) CreatePost(ctx context.Context, userID string, body string, createdAt *time.Time, deletedAt *time.Time, audreyID string, emotes []db.Emote, embedURLs *[]string, embedEntities *[]entity.Entity) (*db.Post, error) {
	id := uuid.NewV4().String()
	if createdAt == nil {
		createdAt = now()
	}

	post := &db.Post{
		ID:            id,
		UserID:        userID,
		Body:          body,
		CreatedAt:     *createdAt,
		DeletedAt:     deletedAt,
		AudreyID:      audreyID,
		Emotes:        emotes,
		EmbedURLs:     embedURLs,
		EmbedEntities: embedEntities,
	}

	item := map[string]*dynamodb.AttributeValue{
		"id":         {S: aws.String(post.ID)},
		"user_id":    {S: aws.String(post.UserID)},
		"created_at": timeToAttributeValue(post.CreatedAt),
	}
	if post.Body != "" {
		item["body"] = &dynamodb.AttributeValue{S: aws.String(post.Body)}
	}
	if post.DeletedAt != nil {
		item["deleted_at"] = timeToAttributeValue(*post.DeletedAt)
	}
	if len(post.AudreyID) > 0 {
		item["audrey_id"] = &dynamodb.AttributeValue{S: aws.String(post.AudreyID)}
	}
	if emotes != nil {
		item["emotes"] = emotesToDynamo(emotes, *now())
	}
	if embedURLs != nil {
		item["embed_urls"] = embedURLsToDynamo(embedURLs, *now())
	}
	if embedEntities != nil {
		item["embed_entities"] = embedEntitiesToDynamo(embedEntities, *now())
	}

	params := &dynamodb.PutItemInput{
		TableName: aws.String(d.Config.PostTableName.Get()),
		Item:      item,
	}

	req, _ := d.Dynamo.PutItemRequest(params)
	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return nil, err
	}

	return post, nil
}

// GetPostIDByAudreyID gets the post ID given an Audrey ID
func (d *DB) GetPostIDByAudreyID(ctx context.Context, audreyID string) (*string, error) {
	return d.audreyIDSearch(ctx, d.Config.PostTableName.Get(), audreyID)
}

// GetPost returns the post for the given ID
func (d *DB) GetPost(ctx context.Context, id string) (*db.Post, error) {

	item, err := d.keyedGet(ctx, id, d.Config.PostTableName.Get())
	if err != nil {
		return nil, err
	} else if item == nil {
		return nil, nil
	}

	post, err := attributeValueMapToPost(item)
	if post != nil && post.DeletedAt != nil {
		return nil, err
	}
	return post, err
}

// DeletePost deletes the post for the given ID
func (d *DB) DeletePost(ctx context.Context, id string, deletedAt *time.Time) (*db.Post, error) {
	if deletedAt == nil {
		deletedAt = now()
	}
	key := idToAttributeValueMap(id)
	getParams := &dynamodb.GetItemInput{
		TableName: aws.String(d.Config.PostTableName.Get()),
		Key:       key,
	}
	d.setGetItemInputConsistency(ctx, getParams, false)

	req, item := d.Dynamo.GetItemRequest(getParams)

	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return nil, err
	} else if item.Item == nil {
		return nil, nil
	}

	post, err := attributeValueMapToPost(item.Item)
	if err != nil {
		return nil, err
	} else if post.DeletedAt != nil {
		return nil, err
	}

	post.DeletedAt = deletedAt

	updateParams := &dynamodb.UpdateItemInput{
		TableName:        aws.String(d.Config.PostTableName.Get()),
		Key:              key,
		UpdateExpression: aws.String("set deleted_at = :deleted_at"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":deleted_at": timeToAttributeValue(*post.DeletedAt),
		},
	}

	req, _ = d.Dynamo.UpdateItemRequest(updateParams)
	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return nil, err
	}
	return post, nil
}

// GetSharesByIDs returns the shares for the given ID
func (d *DB) GetSharesByIDs(ctx context.Context, ids []string) ([]*db.Share, error) {

	items, err := d.keyedBatchGet(ctx, ids, d.Config.ShareTableName.Get(), false, idToAttributeValueMap)
	if err != nil {
		return nil, err
	}

	shares := make([]*db.Share, 0, len(items))
	for i := 0; i < len(items); i++ {
		share, _ := attributeValueMapToShare(items[i])

		if share == nil || share.DeletedAt != nil {
			continue
		}

		shares = append(shares, share)
	}
	return shares, nil
}

// GetPostsByIDs returns the post for the given ID
func (d *DB) GetPostsByIDs(ctx context.Context, ids []string) ([]*db.Post, error) {
	items, err := d.keyedBatchGet(ctx, ids, d.Config.PostTableName.Get(), false, idToAttributeValueMap)
	if err != nil {
		return nil, err
	}

	posts := make([]*db.Post, 0, len(items))
	for i := 0; i < len(items); i++ {
		post, _ := attributeValueMapToPost(items[i])

		if post == nil || post.DeletedAt != nil {
			continue
		}

		posts = append(posts, post)
	}
	return posts, nil
}

type userIDCreatedAtStartKey struct {
	ID        string `json:"id"`
	CreatedAt string `json:"created_at"`
}

// userIDCreatedAtKeyEncoder facilitates encoding LastEvauluatedKeys and decoding ExclusiveStartKeys for the queries
// using the user_id-created_at-index GSI on the posts table.
type userIDCreatedAtKeyEncoder struct {
	userID string
}

func (c *userIDCreatedAtKeyEncoder) decode(encoded string) (map[string]*dynamodb.AttributeValue, error) {
	b, err := base64.StdEncoding.DecodeString(encoded)
	if err != nil {
		return nil, errors.Wrap(err, "unable to base64 decode")
	}
	startKey := userIDCreatedAtStartKey{}
	if err := json.NewDecoder(bytes.NewReader(b)).Decode(&startKey); err != nil {
		return nil, errors.Wrap(err, "unable to JSON decode")
	}
	return map[string]*dynamodb.AttributeValue{
		"id":         {S: aws.String(startKey.ID)},
		"user_id":    {S: aws.String(c.userID)},
		"created_at": {N: aws.String(startKey.CreatedAt)},
	}, nil
}

func (c *userIDCreatedAtKeyEncoder) encode(m map[string]*dynamodb.AttributeValue) (string, error) {
	const idKey string = "id"
	idMap, err := feedsdynamo.AwsStrings(m, []string{idKey})
	if err != nil {
		return "", errors.Wrap(err, "invalid aws LastEvaluatedKey")
	}

	const createdAtKey string = "created_at"
	createdAtMap, err := feedsdynamo.AwsInts(m, []string{createdAtKey})
	if err != nil {
		return "", errors.Wrap(err, "invalid aws LastEvaluatedKey")
	}

	startKey := userIDCreatedAtStartKey{
		ID:        idMap[idKey],
		CreatedAt: strconv.FormatInt(createdAtMap[createdAtKey], 10),
	}

	b, err := json.Marshal(startKey)
	if err != nil {
		return "", err
	}

	return base64.StdEncoding.EncodeToString(b), nil
}

// GetPostIDsByUser gets the post ID given author id
func (d *DB) GetPostIDsByUser(ctx context.Context, authorID string, limit int64, cursor string) (*db.PostIDsResponse, error) {

	keyEncoder := userIDCreatedAtKeyEncoder{userID: authorID}

	var startKey map[string]*dynamodb.AttributeValue
	if cursor != "" {
		var err error
		if startKey, err = keyEncoder.decode(cursor); err != nil {
			return nil, err
		}
	}

	params := &dynamodb.QueryInput{
		TableName:              aws.String(d.Config.PostTableName.Get()),
		IndexName:              aws.String("user_id-created_at-index"),
		KeyConditionExpression: aws.String("user_id = :user_id"),
		FilterExpression:       aws.String("attribute_not_exists(deleted_at)"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":user_id": {S: &authorID},
		},
		ProjectionExpression: aws.String("id"),
		ExclusiveStartKey:    startKey,
		Limit:                &limit,
	}
	// Cannot set consistency on global secondary indexes
	//d.setQueryInputConsistency(ctx, params)
	resp, err := d.Dynamo.Query(params)
	if err != nil {
		return nil, errors.Wrap(err, "unable to query for post IDs by user")
	}

	ids := make([]string, 0, len(resp.Items))
	for _, item := range resp.Items {
		var idMap map[string]string
		if idMap, err = feedsdynamo.AwsStrings(item, []string{"id"}); err != nil {
			return nil, errors.Wrap(err, "invalid aws item")
		}
		ids = append(ids, idMap["id"])
	}

	var retCursor string
	if resp.LastEvaluatedKey != nil {
		if retCursor, err = keyEncoder.encode(resp.LastEvaluatedKey); err != nil {
			return nil, errors.Wrap(err, "unable to encode cursor")
		}
	}

	return &db.PostIDsResponse{
		PostIDs: ids,
		Cursor:  retCursor,
	}, nil
}

func (d *DB) audreyIDSearch(ctx context.Context, tableName string, audreyID string) (*string, error) {
	params := &dynamodb.QueryInput{
		TableName:              &tableName,
		IndexName:              aws.String("audrey_id-index"),
		KeyConditionExpression: aws.String("audrey_id = :audrey_id"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":audrey_id": {S: &audreyID},
		},
		ProjectionExpression: aws.String("id"),
	}
	// No consistency on global secondary indexes
	//d.setQueryInputConsistency(ctx, params)
	resp, err := d.Dynamo.Query(params)
	if err != nil {
		return nil, err
	} else if len(resp.Items) == 0 {
		return nil, nil
	}

	return resp.Items[0]["id"].S, nil
}

// GetCommentIDByAudreyID gets the post ID given an Audrey ID
func (d *DB) GetCommentIDByAudreyID(ctx context.Context, audreyID string) (*string, error) {
	return d.audreyIDSearch(ctx, d.Config.CommentTableName.Get(), audreyID)
}

// CreateComment creates a comment
func (d *DB) CreateComment(ctx context.Context, userID string, parentEntity entity.Entity, body string, createdAt *time.Time, deletedAt *time.Time, audreyID string, needsApproval bool) (*db.Comment, error) {
	id := uuid.NewV4().String()
	if createdAt == nil {
		createdAt = now()
	}

	comment := &db.Comment{
		ID:            id,
		ParentEntity:  parentEntity,
		UserID:        userID,
		Body:          body,
		CreatedAt:     *createdAt,
		DeletedAt:     deletedAt,
		AudreyID:      audreyID,
		NeedsApproval: needsApproval,
	}

	encodedParentEntity := comment.ParentEntity.Encode()

	item := map[string]*dynamodb.AttributeValue{
		"id":             {S: &comment.ID},
		"parent_entity":  {S: &encodedParentEntity},
		"user_id":        {S: &comment.UserID},
		"body":           {S: &comment.Body},
		"created_at":     timeToAttributeValue(comment.CreatedAt),
		"needs_approval": {BOOL: &comment.NeedsApproval},
	}
	if comment.DeletedAt != nil {
		item["deleted_at"] = timeToAttributeValue(*comment.DeletedAt)
	}
	if len(comment.AudreyID) > 0 {
		item["audrey_id"] = &dynamodb.AttributeValue{S: aws.String(comment.AudreyID)}
	}
	params := &dynamodb.PutItemInput{
		TableName: aws.String(d.Config.CommentTableName.Get()),
		Item:      item,
	}

	req, _ := d.Dynamo.PutItemRequest(params)
	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return nil, err
	}

	if comment.DeletedAt == nil && !comment.NeedsApproval {
		summaryParams := &dynamodb.UpdateItemInput{
			TableName: aws.String(d.Config.CommentsSummaryTableName.Get()),
			Key: map[string]*dynamodb.AttributeValue{
				"parent_entity": {S: aws.String(parentEntity.Encode())},
			},
			UpdateExpression: aws.String("ADD #count :inc_amnt"),
			ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
				":inc_amnt": {N: aws.String("1")},
			},
			ExpressionAttributeNames: map[string]*string{
				"#count": aws.String("count"),
			},
		}
		req, _ = d.Dynamo.UpdateItemRequest(summaryParams)
		if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
			d.Log.Log("err", err, "parent_entity", parentEntity, "could not update comment summary table")
			return nil, err
		}
	}

	return comment, nil
}

type encodedCursor struct {
	ID           string        `json:"id"`
	ParentEntity entity.Entity `json:"parent_entity"`
	CreatedAt    time.Time     `json:"created_at"`
}

func (c *encodedCursor) decode(encoded string) error {
	b, err := base64.StdEncoding.DecodeString(encoded)
	if err != nil {
		return errors.Wrap(err, "unable to base64 decode string")
	}
	return errors.Wrap(json.NewDecoder(bytes.NewReader(b)).Decode(c), "unable to decode JSON")
}

func (c *encodedCursor) encode() (string, error) {
	b, err := json.Marshal(c)
	if err != nil {
		return "", err
	}
	encoded := base64.StdEncoding.EncodeToString(b)
	return encoded, nil
}

// GetCommentIDsByParent returns a list of commentIds associated with the given parent entity
func (d *DB) GetCommentIDsByParent(ctx context.Context, parentEntity entity.Entity, limit int64, cursor string) (*db.CommentIDsResponse, error) {
	resp := db.CommentIDsResponse{}

	// If a cursor was provided, attempt to parse it and use it as the start key for our dynamodb query
	var startKey map[string]*dynamodb.AttributeValue
	if cursor != "" {
		c := encodedCursor{}
		if err := c.decode(cursor); err != nil {
			return nil, err
		}

		startKey = map[string]*dynamodb.AttributeValue{
			"id":            {S: aws.String(c.ID)},
			"created_at":    timeToAttributeValue(c.CreatedAt),
			"parent_entity": {S: aws.String(c.ParentEntity.Encode())},
		}
	}

	items, err := d.getCommentIDsByParent(ctx, parentEntity, nil, limit, startKey)

	if err != nil {
		return nil, err
	} else if len(items.Items) == 0 {
		return &resp, nil
	}

	// If LastEvaluatedKey is not nil, we received a paginated response and not all results were returned to us with
	// our query. Construct a new cursor based on the last key that was processed and include it with the response.
	if items.LastEvaluatedKey != nil {
		var keyErr error
		lastKey := encodedCursor{}
		lastKey.ID = *items.LastEvaluatedKey["id"].S
		lastKey.ParentEntity, keyErr = entity.Decode(*items.LastEvaluatedKey["parent_entity"].S)
		if keyErr != nil {
			return nil, keyErr
		}
		createdAt, keyErr := attributeValueToTime(items.LastEvaluatedKey["created_at"])
		if keyErr != nil {
			return nil, keyErr
		}
		lastKey.CreatedAt = *createdAt
		resp.Cursor, keyErr = lastKey.encode()
		if keyErr != nil {
			return nil, keyErr
		}
	}

	for _, item := range items.Items {
		resp.CommentIds = append(resp.CommentIds, *item["id"].S)
	}

	return &resp, nil
}

func (d *DB) getCommentIDsByParent(ctx context.Context, parentEntity entity.Entity, userID *string, limit int64, startKey map[string]*dynamodb.AttributeValue) (*dynamodb.QueryOutput, error) {
	// Create a query object to find all comments with a given parent entity
	queryParams := &dynamodb.QueryInput{
		TableName: aws.String(d.Config.CommentTableName.Get()),
		IndexName: aws.String("parent_entity-created_at-index-2"),
		Limit:     aws.Int64(limit),
		KeyConditionExpression: aws.String("parent_entity = :parent_entity"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":parent_entity": {S: aws.String(parentEntity.Encode())},
		},
		ProjectionExpression: aws.String("id"),
		ScanIndexForward:     aws.Bool(false),
		ExclusiveStartKey:    startKey,
	}
	filterExpression := "attribute_not_exists(deleted_at)"

	if userID != nil && *userID != "" {
		queryParams.ExpressionAttributeValues[":user_id"] = &dynamodb.AttributeValue{S: aws.String(*userID)}
		filterExpression += " AND user_id = :user_id"
	}
	queryParams.FilterExpression = aws.String(filterExpression)

	req, response := d.Dynamo.QueryRequest(queryParams)
	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return nil, err
	}
	return response, nil
}

// BatchGetComments returns an array of comments for a given array of comment ids
func (d *DB) BatchGetComments(ctx context.Context, ids []string) ([]*db.Comment, error) {
	dynamoComments, err := d.keyedBatchGet(ctx, ids, d.Config.CommentTableName.Get(), false, idToAttributeValueMap)
	if err != nil {
		return nil, err
	}

	// Iterate over each comment and convert it from an attribute value map to a comment object
	comments := make([]*db.Comment, 0, len(dynamoComments))
	for _, dynamoComment := range dynamoComments {
		comment, err := attributeValueMapToComment(dynamoComment)
		if err != nil {
			return nil, err
		}

		// only include comments that have not been deleted
		if comment.DeletedAt == nil {
			comments = append(comments, comment)
		}

	}

	// BatchGetItem does not return items in any order (because parallelized queries). Sort the comments based on the
	// order of ids passed in to this method.
	resp := []*db.Comment{}
	for _, id := range ids {
		for _, comment := range comments {
			if comment.ID == id {
				resp = append(resp, comment)
				break
			}
		}
	}

	return resp, nil
}

type cursorEncoder interface {
	decode(string) (map[string]*dynamodb.AttributeValue, error)
	encode(map[string]*dynamodb.AttributeValue) (string, error)
}

// GetComment returns the comment for the given ID
func (d *DB) GetComment(ctx context.Context, id string) (*db.Comment, error) {
	item, err := d.keyedGet(ctx, id, d.Config.CommentTableName.Get())
	if err != nil {
		return nil, err
	} else if item == nil {
		return nil, nil
	}

	comment, err := attributeValueMapToComment(item)
	if comment != nil && comment.DeletedAt != nil {
		return nil, err
	}
	return comment, err
}

func (d *DB) keyedGet(ctx context.Context, id string, tableName string) (map[string]*dynamodb.AttributeValue, error) {
	params := &dynamodb.GetItemInput{
		TableName: &tableName,
		Key:       idToAttributeValueMap(id),
	}
	d.setGetItemInputConsistency(ctx, params, false)

	req, item := d.Dynamo.GetItemRequest(params)
	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return nil, err
	}
	return item.Item, nil
}

func uniqueStrings(ids []string) []string {
	ret := make([]string, 0, len(ids))
	requireUniqueKeys := make(map[string]struct{}, len(ids))
	for i := 0; i < len(ids); i++ {
		if _, exists := requireUniqueKeys[ids[i]]; exists {
			continue
		}
		requireUniqueKeys[ids[i]] = struct{}{}
		ret = append(ret, ids[i])
	}
	return ret
}

func uniqueEnts(ents []entity.Entity) []entity.Entity {
	ret := make([]entity.Entity, 0, len(ents))
	requireUniqueKeys := make(map[entity.Entity]struct{}, len(ents))
	for i := 0; i < len(ents); i++ {
		if _, exists := requireUniqueKeys[ents[i]]; exists {
			continue
		}
		requireUniqueKeys[ents[i]] = struct{}{}
		ret = append(ret, ents[i])
	}
	return ret
}

func parentEntityToValueMap(encodedParentEntity string) map[string]*dynamodb.AttributeValue {
	return map[string]*dynamodb.AttributeValue{
		"parent_entity": {S: &encodedParentEntity},
	}
}

// loopKeys is a helper and should not be called directly.  It should only be called by keyedBatchGet
func (d *DB) loopKeys(ctx context.Context, ids []string, tableName string, stronglyConsistent bool, keyCreator func(string) map[string]*dynamodb.AttributeValue) ([]map[string]*dynamodb.AttributeValue, error) {
	keys := make([]map[string]*dynamodb.AttributeValue, 0, len(ids))
	for i := 0; i < len(ids); i++ {
		keys = append(keys, keyCreator(ids[i]))
	}

	ret := make([]map[string]*dynamodb.AttributeValue, 0, len(keys))
	unprocessedKeys := map[string]*dynamodb.KeysAndAttributes{
		tableName: {
			Keys: keys,
		},
	}
	for len(unprocessedKeys) > 0 {
		params := &dynamodb.BatchGetItemInput{
			RequestItems: unprocessedKeys,
		}
		d.setBatchGetItemConsistency(ctx, params, stronglyConsistent)

		req, items := d.Dynamo.BatchGetItemRequest(params)

		if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
			return nil, err
		}
		if items.Responses == nil {
			return nil, nil
		}

		ret = append(ret, items.Responses[tableName]...)

		if len(items.UnprocessedKeys) > 0 && items.UnprocessedKeys[tableName] != nil && len(unprocessedKeys[tableName].Keys) >= len(items.UnprocessedKeys[tableName].Keys) {
			return nil, errors.New("appear to be in a batchitemget Ifiniloop")
		}
		unprocessedKeys = items.UnprocessedKeys
	}
	return ret, nil
}

// BatchGetItemInput has a few gotchas
//   1) You cannot pass duplicates
//   2) You cannot pass empty arrays
//   3) You cannot pass more than 100 items
//   4) A response may have unprocessed keys
//
//  People tend to forget all these, so this does them all at once.  It assumes a unique []string is the primary constraint
// on the batch get.
func (d *DB) keyedBatchGet(ctx context.Context, ids []string, tableName string, stronglyConsistent bool, keyCreator func(string) map[string]*dynamodb.AttributeValue) ([]map[string]*dynamodb.AttributeValue, error) {
	ids = uniqueStrings(ids)
	if len(ids) == 0 {
		return nil, nil
	}

	// Can only call BatchGetItemInput on 100 items at once
	if len(ids) > 100 {
		firstResult, err := d.keyedBatchGet(ctx, ids[:100], tableName, stronglyConsistent, keyCreator)
		if err != nil {
			return nil, err
		}
		secondResult, err := d.keyedBatchGet(ctx, ids[100:], tableName, stronglyConsistent, keyCreator)
		if err != nil {
			return nil, err
		}
		return append(firstResult, secondResult...), nil
	}

	return d.loopKeys(ctx, ids, tableName, stronglyConsistent, keyCreator)
}

// BatchGetCommentsSummary returns comments summaries for a given set of parent entities
func (d *DB) BatchGetCommentsSummary(ctx context.Context, parentEntities []entity.Entity) ([]*db.CommentsSummary, error) {
	dbSummaries, err := d.keyedBatchGet(ctx, entity.EncodeAll(parentEntities), d.Config.CommentsSummaryTableName.Get(), false, parentEntityToValueMap)
	if err != nil {
		return nil, err
	}

	summaries := make([]*db.CommentsSummary, 0, len(dbSummaries))
	for _, dbSummary := range dbSummaries {
		summary, err := attributeValueMapToCommentsSummary(dbSummary)
		if err != nil {
			return nil, err
		}
		summaries = append(summaries, summary)
	}

	return summaries, nil
}

// BatchGetReactions returns the reactions for a user and group of entities
func (d *DB) BatchGetReactions(ctx context.Context, parentEntities []entity.Entity, userID string) ([]*db.Reactions, error) {
	dbReactions, err := d.keyedBatchGet(ctx, entity.EncodeAll(parentEntities), d.Config.ReactionTableName.Get(), false, func(s string) map[string]*dynamodb.AttributeValue {
		return map[string]*dynamodb.AttributeValue{
			"user_id":       {S: &userID},
			"parent_entity": {S: &s},
		}
	})
	if err != nil {
		return nil, err
	}

	ret := make([]*db.Reactions, 0, len(dbReactions))
	for _, dbReaction := range dbReactions {
		reaction, err := newDBReactionsFromDynamoItem(dbReaction)
		if err != nil {
			return nil, err
		}
		ret = append(ret, reaction)
	}
	return ret, nil
}

// DeleteComment deletes the comment for the given ID
func (d *DB) DeleteComment(ctx context.Context, id string, deletedAt *time.Time) (*db.Comment, error) {
	if deletedAt == nil {
		deletedAt = now()
	}
	key := idToAttributeValueMap(id)
	getParams := &dynamodb.GetItemInput{
		TableName:      aws.String(d.Config.CommentTableName.Get()),
		Key:            key,
		ConsistentRead: aws.Bool(true),
	}

	d.setGetItemInputConsistency(ctx, getParams, false)
	req, item := d.Dynamo.GetItemRequest(getParams)
	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return nil, err
	} else if item.Item == nil {
		return nil, nil
	}

	comment, err := attributeValueMapToComment(item.Item)
	if err != nil {
		return nil, err
	} else if comment.DeletedAt != nil {
		return nil, nil
	}

	if err := d.deleteComment(ctx, key, comment.ParentEntity, deletedAt); err != nil {
		return nil, err
	}
	comment.DeletedAt = deletedAt
	return comment, nil
}

func (d *DB) deleteComment(ctx context.Context, key map[string]*dynamodb.AttributeValue, parentEntity entity.Entity, deletedAt *time.Time) error {
	updateParams := &dynamodb.UpdateItemInput{
		TableName:        aws.String(d.Config.CommentTableName.Get()),
		Key:              key,
		UpdateExpression: aws.String("set deleted_at = :deleted_at"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":deleted_at": timeToAttributeValue(*deletedAt),
		},
		ReturnValues: aws.String("ALL_OLD"),
	}

	req, output := d.Dynamo.UpdateItemRequest(updateParams)
	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return err
	}

	// Grab a copy of the comment as it was before it was deleted.
	oldComment, err := attributeValueMapToComment(output.Attributes)
	if err != nil {
		return err
	}

	// nil-check for oldComment.DeletedAt to prevent multiple summary updates from concurrent requests.
	if oldComment.DeletedAt == nil {
		return d.updateCommentSummaryForNeedsApproval(ctx, oldComment.ParentEntity, oldComment.NeedsApproval, true)
	}

	return nil
}

// DeleteCommentsByParentAndUser deletes the comments for the given parent entity and user
func (d *DB) DeleteCommentsByParentAndUser(ctx context.Context, parentEntity entity.Entity, userID string) error {
	deletedAt := now()

	var cursor map[string]*dynamodb.AttributeValue
	for {
		items, err := d.getCommentIDsByParent(ctx, parentEntity, &userID, 100, cursor)
		if err != nil {
			return errors.Wrap(err, "could not get comments")
		}
		for _, item := range items.Items {
			key := idToAttributeValueMap(*item["id"].S)
			if err := d.deleteComment(ctx, key, parentEntity, deletedAt); err != nil {
				return err
			}
		}
		if items.LastEvaluatedKey == nil {
			break
		}
		cursor = items.LastEvaluatedKey
	}

	return nil
}

func attributeValueMapToComment(item map[string]*dynamodb.AttributeValue) (*db.Comment, error) {
	var s map[string]string
	var err error

	if s, err = feedsdynamo.AwsStrings(item, []string{"id", "user_id", "body", "parent_entity"}); err != nil {
		return nil, err
	}

	createdAt, err := attributeValueToTime(item["created_at"])
	if err != nil {
		return nil, err
	}

	parentEntity, err := entity.Decode(s["parent_entity"])
	if err != nil {
		return nil, errors.Wrap(err, "malformed parent_entity")
	}

	optional, err := getOptionalCommentAttributesFromMap(item)
	if err != nil {
		return nil, err
	}

	return &db.Comment{
		ID:            s["id"],
		ParentEntity:  parentEntity,
		UserID:        s["user_id"],
		Body:          s["body"],
		CreatedAt:     *createdAt,
		DeletedAt:     optional.DeletedAt,
		AudreyID:      optional.AudreyID,
		NeedsApproval: optional.NeedsApproval,
		Emotes:        optional.Emotes,
	}, nil
}

type optionalCommentAttributes struct {
	DeletedAt     *time.Time
	AudreyID      string
	NeedsApproval bool
	Emotes        []db.Emote
}

func getOptionalCommentAttributesFromMap(item map[string]*dynamodb.AttributeValue) (*optionalCommentAttributes, error) {
	var deletedAt *time.Time
	if val, ok := item["deleted_at"]; ok {
		var err error
		if deletedAt, err = attributeValueToTime(val); err != nil {
			return nil, err
		}
	}

	var audreyID string
	if val, ok := item["audrey_id"]; ok && val.S != nil {
		audreyID = *val.S
	}

	var needsApproval bool
	if val, ok := item["needs_approval"]; ok && val.BOOL != nil {
		needsApproval = *val.BOOL
	}

	var emotes []db.Emote
	if val, ok := item["emotes"]; ok {
		emotes = emotesFromDynamo(val)
	}

	return &optionalCommentAttributes{
		DeletedAt:     deletedAt,
		AudreyID:      audreyID,
		NeedsApproval: needsApproval,
		Emotes:        emotes,
	}, nil

}

func attributeValueMapToCommentsSummary(item map[string]*dynamodb.AttributeValue) (*db.CommentsSummary, error) {
	total, err := strconv.Atoi(*item["count"].N)
	if err != nil {
		return nil, err
	}

	parentEntity, err := entity.Decode(*item["parent_entity"].S)
	if err != nil {
		return nil, errors.Wrap(err, "malformed parent_entity")
	}

	summary := &db.CommentsSummary{
		ParentEntity: parentEntity,
		Total:        total,
	}

	return summary, nil
}

// GetReactions retrieves a user's reactions for a given parentEntity
func (d *DB) GetReactions(ctx context.Context, parentEntity entity.Entity, userID string) (*db.Reactions, error) {
	getParams := &dynamodb.GetItemInput{
		TableName: aws.String(d.Config.ReactionTableName.Get()),
		Key: map[string]*dynamodb.AttributeValue{
			"parent_entity": {S: aws.String(parentEntity.Encode())},
			"user_id":       {S: aws.String(userID)},
		},
	}

	d.setGetItemInputConsistency(ctx, getParams, false)
	req, reaction := d.Dynamo.GetItemRequest(getParams)
	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return nil, err
	} else if reaction.Item == nil {
		return nil, nil
	}

	return newDBReactionsFromDynamoItem(reaction.Item)
}

func stringInSlice(a string, list []string) bool {
	for _, b := range list {
		if b == a {
			return true
		}
	}
	return false
}

func removeFromSlice(a string, list []string) []string {
	v := make([]string, 0, len(list))
	for _, b := range list {
		if b != a {
			v = append(v, b)
		}
	}
	return v
}

// CreateReaction adds a reaction to a given Parent Entity ID
func (d *DB) CreateReaction(ctx context.Context, parentEntity entity.Entity, userID string, emoteID string) (*db.Reactions, error) {

	// create reaction in user reaction table
	userParams := &dynamodb.UpdateItemInput{
		TableName: aws.String(d.Config.ReactionTableName.Get()),
		Key: map[string]*dynamodb.AttributeValue{
			"parent_entity": {S: aws.String(parentEntity.Encode())},
			"user_id":       {S: aws.String(userID)},
		},
		UpdateExpression: aws.String("ADD emote_ids :emote_id"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":emote_id": {SS: aws.StringSlice([]string{emoteID})},
		},
		// Have the UpdateItem operation return the old versions of the updated attributes.
		ReturnValues: aws.String("UPDATED_OLD"),
	}

	req, item := d.Dynamo.UpdateItemRequest(userParams)
	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return nil, err
	}

	// Check to see if the emoteID we're trying to add previously existed in the user's list of emote reactions.
	// If it does (meaning the user is trying to add a reaction that already is present) don't increment the
	// reaction summary count for that emote.
	var emoteIDs []string
	if err := dynamodbattribute.Unmarshal(item.Attributes["emote_ids"], &emoteIDs); err != nil {
		return nil, err
	}

	if !stringInSlice(emoteID, emoteIDs) {
		if err := d.changeReactionsSummary(ctx, parentEntity, emoteID, 1); err != nil {
			return nil, err
		}
		emoteIDs = append(emoteIDs, emoteID)
	}

	return &db.Reactions{
		ParentEntity: parentEntity,
		UserID:       userID,
		EmoteIDs:     emoteIDs,
	}, nil
}

// changeReactionsSummary increments or decrements the number of the given emote by the given delta
func (d *DB) changeReactionsSummary(ctx context.Context, parentEntity entity.Entity, emoteID string, delta int) error {
	emoteKey := "emote_" + emoteID

	summaryParams := &dynamodb.UpdateItemInput{
		TableName: aws.String(d.Config.ReactionsSummaryTableName.Get()),
		Key: map[string]*dynamodb.AttributeValue{
			"parent_entity": {S: aws.String(parentEntity.Encode())},
		},
		UpdateExpression: aws.String("ADD #emote_key :incr_amnt"),
		ExpressionAttributeNames: map[string]*string{
			"#emote_key": &emoteKey,
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":incr_amnt": {N: aws.String(strconv.Itoa(delta))},
		},
	}

	req, _ := d.Dynamo.UpdateItemRequest(summaryParams)
	return service_common.ContextSend(ctx, req, d.Log)
}

// DeleteReaction deletes a user reaction from a parent entity
func (d *DB) DeleteReaction(ctx context.Context, parentEntity entity.Entity, userID string, emoteID string) (*db.Reactions, error) {
	// Remove the reaction from the user's list of reactions
	reactionParams := &dynamodb.UpdateItemInput{
		TableName: aws.String(d.Config.ReactionTableName.Get()),
		Key: map[string]*dynamodb.AttributeValue{
			"parent_entity": {S: aws.String(parentEntity.Encode())},
			"user_id":       {S: aws.String(userID)},
		},
		UpdateExpression: aws.String("DELETE emote_ids :emote_id"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":emote_id": {SS: aws.StringSlice([]string{emoteID})},
		},
		// Have the UpdateItem operation return the old versions of the updated attributes.
		ReturnValues: aws.String("UPDATED_OLD"),
	}
	req, item := d.Dynamo.UpdateItemRequest(reactionParams)
	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return nil, err
	}

	// Check to see if the emoteID we're trying to delete previously existed in the user's list of emote reactions.
	// If it did not (meaning the user is trying to delete a non-existent reaction) don't decrement the reaction
	// summary count for that emote.
	emoteIDsBeforeUpdate := []string{}
	if err := dynamodbattribute.Unmarshal(item.Attributes["emote_ids"], &emoteIDsBeforeUpdate); err != nil {
		return nil, err
	} else if !stringInSlice(emoteID, emoteIDsBeforeUpdate) {
		return nil, nil
	}

	// Decrement the reaction summary count for the emote.
	if err := d.changeReactionsSummary(ctx, parentEntity, emoteID, -1); err != nil {
		return nil, err
	}

	return &db.Reactions{
		ParentEntity: parentEntity,
		UserID:       userID,
		EmoteIDs:     removeFromSlice(emoteID, emoteIDsBeforeUpdate),
	}, nil
}

// GetReactionsSummary returns the reaction summary for the parent entity ID
func (d *DB) GetReactionsSummary(ctx context.Context, parentEntity entity.Entity) (*db.ReactionsSummary, error) {
	getParams := &dynamodb.GetItemInput{
		TableName: aws.String(d.Config.ReactionsSummaryTableName.Get()),
		Key: map[string]*dynamodb.AttributeValue{
			"parent_entity": {S: aws.String(parentEntity.Encode())},
		},
	}
	d.setGetItemInputConsistency(ctx, getParams, false)

	req, reactionsSummaryItem := d.Dynamo.GetItemRequest(getParams)
	if err := service_common.ContextSend(ctx, req, d.Log); err != nil {
		return nil, err
	} else if reactionsSummaryItem.Item == nil {
		return nil, err
	}

	dbReactionsSummary, err := newDBReactionsSummaryFromDynamoItem(reactionsSummaryItem.Item)

	if err != nil {
		return nil, err
	}
	return dbReactionsSummary, nil
}

// BatchGetReactionsSummary returns an array of reactions summaries for a given array of parent entities
func (d *DB) BatchGetReactionsSummary(ctx context.Context, parentEntities []entity.Entity, stronglyConsistent bool) ([]*db.ReactionsSummary, error) {

	dbReactionSummaries, err := d.keyedBatchGet(ctx, entity.EncodeAll(parentEntities), d.Config.ReactionsSummaryTableName.Get(), stronglyConsistent, parentEntityToValueMap)
	if err != nil {
		return nil, err
	}

	summaries := make([]*db.ReactionsSummary, 0, len(dbReactionSummaries))
	for _, dbReactionSummary := range dbReactionSummaries {
		summary, err := newDBReactionsSummaryFromDynamoItem(dbReactionSummary)
		if err != nil {
			return nil, err
		}
		summaries = append(summaries, summary)
	}

	return summaries, nil
}

func newDBReactionsFromDynamoItem(item map[string]*dynamodb.AttributeValue) (*db.Reactions, error) {
	emoteIDs := []string{}
	err := dynamodbattribute.Unmarshal(item["emote_ids"], &emoteIDs)
	if err != nil {
		return nil, err
	}

	parentEntity, err := entity.Decode(*item["parent_entity"].S)
	if err != nil {
		return nil, errors.Wrap(err, "malformed parent_entity")
	}

	reaction := &db.Reactions{
		UserID:       *item["user_id"].S,
		ParentEntity: parentEntity,
		EmoteIDs:     emoteIDs,
	}

	return reaction, nil
}

func newDBShareSummaryFromDynamoItem(item map[string]*dynamodb.AttributeValue) (*db.ShareSummary, error) {
	vals, err := feedsdynamo.AwsInts(item, []string{"share_count"})
	if err != nil {
		return nil, err
	}
	valStrings, err := feedsdynamo.AwsStrings(item, []string{"parent_entity"})
	if err != nil {
		return nil, err
	}
	decodedEntity, err := entity.Decode(valStrings["parent_entity"])
	if err != nil {
		return nil, err
	}

	return &db.ShareSummary{
		Total:        int(vals["share_count"]),
		ParentEntity: decodedEntity,
	}, nil
}

func newDBReactionsSummaryFromDynamoItem(item map[string]*dynamodb.AttributeValue) (*db.ReactionsSummary, error) {
	emotes := map[string]int{}
	for key, val := range item {
		if strings.Contains(key, "emote_") {
			i, err := strconv.Atoi(*val.N)
			if err != nil {
				continue
			}
			key = strings.Replace(key, "emote_", "", 1)
			// Filter emotes that are back to zero
			// TODO: We should clean up zero emotes
			if i > 0 {
				emotes[key] = i
			}
		}
	}

	parentEntity, err := entity.Decode(*item["parent_entity"].S)
	if err != nil {
		return nil, errors.Wrap(err, "malformed parent_entity")
	}

	reaction := &db.ReactionsSummary{
		ParentEntity: parentEntity,
		Emotes:       emotes,
	}

	return reaction, nil
}
