package db

import (
	"fmt"
	"time"

	"code.justin.tv/feeds/feeds-common/entity"
	"github.com/aws/aws-sdk-go/aws/awsutil"
)

// Post is a database Post object
type Post struct {
	ID        string
	UserID    string
	Body      string
	CreatedAt time.Time
	DeletedAt *time.Time
	AudreyID  string
	Emotes    []Emote
	// Deprecated
	EmbedURLs     *[]string
	EmbedEntities *[]entity.Entity
}

// Share is a feed item that was shared by someone on their own feed
type Share struct {
	ID           string
	UserID       string
	TargetEntity entity.Entity
	CreatedAt    time.Time
	DeletedAt    *time.Time
}

// ShareSummary contains total share amount for a single target entity
type ShareSummary struct {
	ParentEntity entity.Entity
	Total        int
}

func (t *Post) String() string {
	return awsutil.Prettify(t)
}

// Emote is the emote object for the API
type Emote struct {
	ID    int `json:"id"`
	Start int `json:"start"`
	End   int `json:"end"`
	Set   int `json:"set"`
}

// Comment is a database Comment object
type Comment struct {
	ID            string
	ParentEntity  entity.Entity
	UserID        string
	Body          string
	CreatedAt     time.Time
	DeletedAt     *time.Time
	AudreyID      string
	NeedsApproval bool
	Emotes        []Emote
}

func (t *Comment) String() string {
	return awsutil.Prettify(t)
}

// CommentIDsResponse is an envelope that is returned from the db to the api
type CommentIDsResponse struct {
	CommentIds []string
	Cursor     string
}

// CommentsSummary is a database CommentsSummary object
type CommentsSummary struct {
	ParentEntity entity.Entity
	Total        int
}

// PostIDsResponse is an envelope that is returned from the db to the api
type PostIDsResponse struct {
	PostIDs []string
	Cursor  string
}

// Reactions is a database object containing the reactions a given user has made for a parent entity
type Reactions struct {
	ParentEntity entity.Entity
	UserID       string
	EmoteIDs     []string
}

func (t *Reactions) String() string {
	if t == nil {
		return "<nil>"
	}
	return fmt.Sprintf("{ParentEntity: \"%s\", UserID: \"%s\", Emotes: %s}", t.ParentEntity.String(), t.UserID, awsutil.Prettify(t.EmoteIDs))
}

// ReactionsSummary is a database object which contains all reactions all users have made for a given parent entity
type ReactionsSummary struct {
	ParentEntity entity.Entity
	Emotes       map[string]int
}

func (t *ReactionsSummary) String() string {
	if t == nil {
		return "<nil>"
	}
	return fmt.Sprintf("{ParentEntity: \"%s\", Emotes: %s}", t.ParentEntity.String(), awsutil.Prettify(t.Emotes))
}
