package main

import (
	"net"
	"os"
	"os/signal"
	"strings"
	"time"

	"net/http"

	"code.justin.tv/common/config"
	"code.justin.tv/feeds/clients/fanout"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/dnscache"
	"code.justin.tv/feeds/duplo/cmd/duplo/internal/activity"
	"code.justin.tv/feeds/duplo/cmd/duplo/internal/api"
	"code.justin.tv/feeds/duplo/cmd/duplo/internal/db/dupdynamo"
	"code.justin.tv/feeds/metrics/sfx/sfxstatsd"
	"code.justin.tv/feeds/metrics/statsdim"
	"code.justin.tv/feeds/service-common"
	xraydynamo "code.justin.tv/feeds/xray/plugins/dynamodb"
	"code.justin.tv/feeds/xray/plugins/ec2"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sqs"
)

const (
	teamName    = "feeds"
	serviceName = "duplo"
)

// CodeVersion is set by build script
var CodeVersion string

func sfxStastdConfig() sfxstatsd.SetupConfig {
	return sfxstatsd.SetupConfig{
		CounterMetricRules: []*statsdim.ConfigurableDelimiterMetricRule{
			{
				MetricPath:    "http.*.code.*",
				DimensionsMap: "-.handler.-.http_code",
				MetricName:    "http.hits",
			},
		},
		TimingMetricRules: []*statsdim.ConfigurableDelimiterMetricRule{
			{
				MetricPath:    "http.*.time",
				DimensionsMap: "%.handler.%",
			},
		},
	}
}

var instance = service{
	osExit: os.Exit,
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:       teamName,
			Service:    serviceName,
			OsGetenv:   os.Getenv,
			OsHostname: os.Hostname,
		},
		CodeVersion:    CodeVersion,
		SfxSetupConfig: sfxStastdConfig(),
	},
}

type asyncClientConfig struct {
	AsyncQueueSize *distconf.Int
}

func (a *asyncClientConfig) Load(d *distconf.Distconf) error {
	a.AsyncQueueSize = d.Int("fanout.client.async_queue_size", 1024)
	return nil
}

type injectables struct {
	CommentActivityPublisher api.ActivityPublisher
}

type service struct {
	injectables
	osExit        func(code int)
	onListen      func(net.Addr)
	sigChan       chan os.Signal
	serviceCommon service_common.ServiceCommon
	runner        service_common.ServiceRunner

	server            api.HTTPServer
	asyncFanoutClient fanout.AsyncClient

	configs struct {
		HTTPConfig            service_common.BaseHTTPServerConfig
		DupdynamoConfig       dupdynamo.Config
		FanoutConfig          fanout.Config
		AsyncClientConfig     asyncClientConfig
		CommentActivityConfig activity.Config
	}
	skipGlobalConfigParse bool
}

func (f *service) setup() error {
	if err := f.serviceCommon.Setup(); err != nil {
		return err
	}
	if err := service_common.LoadConfigs(f.serviceCommon.Config, &f.configs.DupdynamoConfig, &f.configs.FanoutConfig, &f.configs.AsyncClientConfig); err != nil {
		return err
	}
	if err := f.configs.CommentActivityConfig.Verify("duplo.comment_activity", f.serviceCommon.Config); err != nil {
		return err
	}
	if err := f.configs.HTTPConfig.Verify(f.serviceCommon.Config, "duplo"); err != nil {
		return err
	}

	if !f.skipGlobalConfigParse {
		if err := config.Parse(); err != nil {
			return err
		}
	}

	if err := f.serviceCommon.XRay.WithPlugin(&ec2.Plugin{}); err != nil {
		f.serviceCommon.Log.Log("err", err, "unable to enable ec2 plugin")
	}
	if err := f.serviceCommon.XRay.WithPlugin(&xraydynamo.Plugin{}); err != nil {
		f.serviceCommon.Log.Log("err", err, "unable to enable ec2 plugin")
	}

	f.serviceCommon.ConfigCommon.DNSCache = dnscache.Cache{
		OnErr: func(err error, host string) {
			f.serviceCommon.Log.Log("host", host, "err", err, "unable to use DNS cache")
		},
		OnCacheUpsert: func(host string, oldAddrs []string, newAddrs []string, lookupDuration time.Duration) {
			oldIPs := strings.Join(oldAddrs, ",")
			newIPs := strings.Join(newAddrs, ",")
			f.serviceCommon.Log.Debug("host", host, "oldIPs", oldIPs, "newIPs", newIPs, "lookupDuration", lookupDuration, "updated DNS cache")
		},
	}
	return nil
}

func (f *service) inject() {
	tr := service_common.DefaultTransport()
	tr.DialContext = f.serviceCommon.ConfigCommon.DNSCache.DialContext
	cl := &http.Client{
		Transport: tr,
	}

	session, awsConf := service_common.CreateAWSSession(f.serviceCommon.Config)
	awsConf = append(awsConf, aws.NewConfig().WithHTTPClient(cl))

	sqsClient := sqs.New(session, awsConf...)
	f.serviceCommon.XRay.AWS(sqsClient.Client)
	f.asyncFanoutClient = fanout.AsyncClient{
		Client: &fanout.Client{
			Sqs:    sqsClient,
			Config: &f.configs.FanoutConfig,
			Ch:     &f.serviceCommon.Ctxlog,
			Log:    f.serviceCommon.Log,
		},
		ActivityQueue: make(chan *fanout.ActivityWithContext, f.configs.AsyncClientConfig.AsyncQueueSize.Get()),
		OnErr: func(err error, activity *fanout.Activity) {
			f.serviceCommon.Log.Log("err", err, "activity", activity, "error sending fanout SQS activity")
		},
	}

	snsClient := sns.New(session, awsConf...)
	f.serviceCommon.XRay.AWS(snsClient.Client)
	if f.CommentActivityPublisher == nil {
		f.CommentActivityPublisher = &activity.Publisher{
			SNS:    snsClient,
			Config: &f.configs.CommentActivityConfig,
			Log:    f.serviceCommon.Log,
		}
	}

	dynamoClient := dynamodb.New(session, awsConf...)
	f.serviceCommon.XRay.AWS(dynamoClient.Client)

	perOpTimeout := service_common.PerOperationTimeout{}
	perOpTimeout.SetTimeouts(service_common.DynamoDBReadOperations(), f.configs.DupdynamoConfig.ReadOperationTimeout.Get)
	perOpTimeout.AddToClient(dynamoClient.Client)

	f.server = api.HTTPServer{
		BaseHTTPServer: service_common.BaseHTTPServer{
			Config: &f.configs.HTTPConfig,
			Stats: &service_common.StatSender{
				SubStatter:   f.serviceCommon.Statsd.NewSubStatter("http"),
				ErrorTracker: &f.serviceCommon.ErrorTracker,
			},
			Dims:        &f.serviceCommon.CtxDimensions,
			Log:         f.serviceCommon.Log,
			ElevateKey:  f.serviceCommon.ElevateLogKey,
			Ctxlog:      &f.serviceCommon.Ctxlog,
			OnListen:    f.onListen,
			PanicLogger: f.serviceCommon.PanicLogger,
			XRay:        f.serviceCommon.XRay,
		},
		DB: &dupdynamo.DB{
			Dynamo: dynamoClient,
			Config: &f.configs.DupdynamoConfig,
			Log:    f.serviceCommon.Log,
			RequireConsistentReads: f.serviceCommon.Config.Bool("dynamo.consistent_reads", false).Get(),
		},
		Fanout:          &f.asyncFanoutClient,
		CommentActivity: f.CommentActivityPublisher,
	}
	f.server.SetupRoutes = f.server.DuploRoutes
	f.server.HealthCheck = f.server.CheckAWS

	f.runner = service_common.ServiceRunner{
		Log: f.serviceCommon.Log,
		Services: []service_common.Service{
			&f.server, &f.asyncFanoutClient, &f.serviceCommon,
		},
		SigChan:      f.sigChan,
		SignalNotify: signal.Notify,
	}

	// Cannot just pass f because f contains private members that I cannot nil check via reflection
	res := (&service_common.NilCheck{
		IgnoredPackages: []string{"aws-sdk-go", "net/http"},
	}).Check(f, &f.server, f.asyncFanoutClient, f.configs, f.runner)
	res.MustBeEmpty(os.Stderr)
}

func (f *service) main() {
	if err := f.setup(); err != nil {
		if f.serviceCommon.Log != nil {
			f.serviceCommon.Log.Log("err", err, "Unable to load initial config")
		}
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		time.Sleep(time.Millisecond * 150)
		f.osExit(1)
		return
	}
	f.inject()
	if err := f.runner.Execute(); err != nil {
		f.serviceCommon.Log.Log("err", err, "wait to end finished with an error")
		time.Sleep(time.Millisecond * 150)
		f.osExit(1)
		return
	}
	f.serviceCommon.Log.Log("Finished main")
}

func main() {
	instance.main()
}
