// +build concurrency

package main

import (
	"math/rand"
	"sync"
	"testing"
	"time"

	"code.justin.tv/feeds/duplo/cmd/duplo/internal/api"
	. "github.com/smartystreets/goconvey/convey"
)

var testEmoteIDs = []string{
	"kappa",
	"SSSsss",
	":)",
	":(",
}

var testUserIDs = []string{
	"111111111",
	"222222222",
	"333333333",
	"444444444",
}

func createEmoteThread(ts *testSetup, channel chan map[string]map[string]int, wg *sync.WaitGroup, parent string, userIDs []string, emoteIDs []string, numReactions int) {
	emotes := map[string]map[string]int{}

	for i := 0; i < numReactions; i++ {
		emoteID := emoteIDs[rand.Intn(len(emoteIDs))]
		userID := userIDs[rand.Intn(len(userIDs))]

		createReaction(ts, parent, userID, emoteID)

		// Create keys in emotes map if they don't exist
		if _, ok := emotes[userID]; !ok {
			emotes[userID] = map[string]int{}
		}
		if _, ok := emotes[userID][emoteID]; !ok {
			emotes[userID][emoteID] = 0
		}

		emotes[userID][emoteID]++

	}

	channel <- emotes
	wg.Done()
}

func deleteEmoteThread(ts *testSetup, wg *sync.WaitGroup, parent string, userID string, emoteIDs []string, numReactions int) {
	for i := 0; i < numReactions; i++ {
		emoteID := emoteIDs[rand.Intn(len(emoteIDs))]

		deleteReaction(ts, parent, userID, emoteID)
	}

	wg.Done()
}

func mergeEmotes(existingEmotes map[string]map[string]int, newEmotes map[string]map[string]int) {
	for userID, emoteMap := range newEmotes {
		if _, ok := existingEmotes[userID]; !ok {
			existingEmotes[userID] = map[string]int{}
		}
		for emoteID, val := range emoteMap {
			if _, ok := existingEmotes[userID][emoteID]; !ok {
				existingEmotes[userID][emoteID] = 0
			}
			existingEmotes[userID][emoteID] += val
		}
	}
}

func loopReactions(requestedReactions map[string]map[string]int) map[string]*api.EmoteSummary {
	// Calculate the reaction count we expect to see stored in the db. There should be a maximum of
	// one reaction per user.
	expectedReactions := map[string]*api.EmoteSummary{}
	for _, userEmoteMap := range requestedReactions {
		for emoteID := range userEmoteMap {
			if _, ok := expectedReactions[emoteID]; !ok {
				expectedReactions[emoteID] = &api.EmoteSummary{}
			}
			expectedReactions[emoteID].Count++
		}
	}
	return expectedReactions
}

func TestIntegration_ReactionsConcurrency(t *testing.T) {
	testInfo, cleanShutdown := startServer(t, injectables{})
	if testInfo == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+testInfo.listenAddr, t, func(c C) {
		ts := &testSetup{host: testInfo.listenAddr}
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)

		Convey("Should not allow a user to react more than once with the same emote", func() {
			parent := "test:parent:" + timestamp()
			numThreads := 10
			numReactionsPerThread := 10

			channel := make(chan map[string]map[string]int, numThreads)
			var wg sync.WaitGroup
			for i := 0; i < numThreads; i++ {
				wg.Add(1)
				go createEmoteThread(ts, channel, &wg, parent, testUserIDs, testEmoteIDs, numReactionsPerThread)
			}
			wg.Wait()

			// Store the actual number of reaction requests (this will differ from the expected reaction count)
			requestedReactions := map[string]map[string]int{}
			for i := 0; i < numThreads; i++ {
				mergeEmotes(requestedReactions, <-channel)
			}
			expectedReactions := loopReactions(requestedReactions)

			summary, err := getReactionsSummary(ts, parent)
			So(err, ShouldBeNil)
			So(summary, ShouldNotBeNil)
			So(summary.EmoteSummaries, ShouldResemble, expectedReactions)
		})

		Convey("Should not allow a user to delete an emote more than once", func() {
			parent := "test:parent:" + timestamp()
			numThreads := 5
			numIterationsPerThread := 5

			// Iterate through each user and have them react with every emote in the testEmotesIDs array
			for _, userID := range testUserIDs {
				for _, emoteID := range testEmoteIDs {
					createReaction(ts, parent, userID, emoteID)
				}
			}

			// Spawn a bunch of threads for one user to try and delete different reactions multiple times at
			// the same time
			var wg sync.WaitGroup
			for j := 0; j < numThreads; j++ {
				wg.Add(1)
				go deleteEmoteThread(ts, &wg, parent, testUserIDs[0], testEmoteIDs, numIterationsPerThread)
			}
			wg.Wait()

			summary, err := getReactionsSummary(ts, parent)
			So(err, ShouldBeNil)
			So(summary, ShouldNotBeNil)
			for _, val := range summary.EmoteSummaries {
				So(val.Count, ShouldBeGreaterThanOrEqualTo, len(testUserIDs)-1)
			}
		})
	})

	cleanShutdown(time.Second * 3)
}
