// +build integration

package main

import (
	"bytes"
	"fmt"
	"io"
	"math/rand"
	"net"
	"net/http"
	"os"
	"strconv"
	"strings"
	"syscall"
	"testing"
	"time"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/duplo/cmd/duplo/internal/api"
	"code.justin.tv/feeds/duplo/cmd/duplo/internal/api/mocks"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"github.com/satori/go.uuid"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"
)

const (
	testUserID    = "27184041" //login: testfeed
	testUserIDAlt = "20122977" //login: testf
)

func timestamp() string {
	return time.Now().Format("2006-01-02-15.04.05.000000000-MST")
}

func intPtr(v int) *int {
	return &v
}

type createPostParams struct {
	UserID    string       `json:"user_id"`
	Body      string       `json:"body"`
	AudreyID  string       `json:"audrey_id,omitempty"`
	CreateAt  *time.Time   `json:"created_at,omitempty"`
	DeletedAt *time.Time   `json:"deleted_at,omitempty"`
	Emotes    *[]api.Emote `json:"emotes,omitempty"`
	EmbedURLs *[]string    `json:"embed_urls,omitempty"`
}

type updatePostParams struct {
	Emotes        *[]api.Emote     `json:"emotes,omitempty"`
	EmbedURLs     *[]string        `json:"embed_urls,omitempty"`
	EmbedEntities *[]entity.Entity `json:"embed_entities,omitempty"`
}

func createPost(ts *testSetup, userID string, body string, opts *createPostParams) (*api.Post, error) {
	url := "/v1/posts"
	params := &createPostParams{
		UserID: userID,
		Body:   body,
	}
	if opts != nil {
		params.AudreyID = opts.AudreyID
		params.CreateAt = opts.CreateAt
		params.DeletedAt = opts.DeletedAt
		params.Emotes = opts.Emotes
		params.EmbedURLs = opts.EmbedURLs
	}
	var ret api.Post
	err := request(ts, "POST", url, params, &ret)
	return &ret, err
}

func createShare(ts *testSetup, userID string, targetEntity entity.Entity) (*api.Share, error) {
	var ret api.Share
	err := request(ts, "POST", "/v1/shares", struct {
		UserID       string        `json:"user_id"`
		TargetEntity entity.Entity `json:"target_entity"`
	}{
		UserID:       userID,
		TargetEntity: targetEntity,
	}, &ret)
	return &ret, err
}

func getShare(ts *testSetup, shareID string) (*api.Share, error) {
	var ret api.Share
	err := request(ts, "GET", "/v1/shares/"+shareID, nil, &ret)
	return &ret, err
}

func getShares(ts *testSetup, shareIDs []string) (*api.Shares, error) {
	var ret api.Shares
	err := request(ts, "GET", "/v1/shares?ids="+strings.Join(shareIDs, ","), nil, &ret)
	return &ret, err
}

func getSharesByAuthor(ts *testSetup, entityID string, userID string) (*api.Shares, error) {
	var ret api.Shares
	err := request(ts, "GET", "/v1/shares_by_author?author_id="+userID+"&target_entities="+entityID, nil, &ret)
	return &ret, err
}

func getSharesSummary(ts *testSetup, entityID string) (*api.SharesSummaries, error) {
	var ret api.SharesSummaries
	err := request(ts, "GET", "/v1/shares_summaries?parent_entities="+entityID, nil, &ret)
	return &ret, err
}

func deleteShare(ts *testSetup, shareID string) (*api.Share, error) {
	var ret api.Share
	err := request(ts, "DELETE", "/v1/shares/"+shareID, nil, &ret)
	return &ret, err
}

func updatePost(ts *testSetup, postID string, params updatePostParams) error {
	url := "/v1/posts/" + postID
	var into string
	err := request(ts, "PUT", url, params, &into)
	return err
}

func getPostIDByAudreyID(ts *testSetup, audreyID string) (*api.ID, error) {
	url := "/v1/posts/legacy/" + audreyID + "/id"
	var ret api.ID
	err := request(ts, "GET", url, nil, &ret)
	return &ret, err
}

func getPost(ts *testSetup, postID string) (*api.Post, error) {
	url := "/v1/posts/" + postID
	var ret api.Post
	err := request(ts, "GET", url, nil, &ret)
	return &ret, err
}

func getPostIDsByUser(ts *testSetup, userID string, limit int, cursor string) (*api.PaginatedPostIDs, error) {
	url := "/v1/posts/ids_by_user/" + userID + "?limit=" + strconv.Itoa(limit)
	if cursor != "" {
		url = url + "&cursor=" + cursor
	}

	var ret api.PaginatedPostIDs
	err := request(ts, "GET", url, nil, &ret)
	return &ret, err
}

func getPosts(ts *testSetup, postIDs []string) (api.Posts, error) {
	return getPostsWithShares(ts, postIDs, false, nil)
}

func getPostsWithShares(ts *testSetup, postIDs []string, shares bool, share_users []string) (api.Posts, error) {
	url := "/v1/posts?ids=" + strings.Join(postIDs, ",")
	if shares {
		url += "&shares=true"
	}
	if len(share_users) > 0 {
		url += "&share_users=" + strings.Join(share_users, ",")
	}
	var ret api.Posts
	err := request(ts, "GET", url, nil, &ret)
	return ret, err
}

func deletePost(ts *testSetup, postID string, deletedAt *time.Time) (*api.Post, error) {
	url := fmt.Sprintf("/v1/posts/%s?", postID)
	if deletedAt != nil {
		url += fmt.Sprintf("&deleted_at=%s", deletedAt.Format(time.RFC3339))
	}
	var ret api.Post
	err := request(ts, "DELETE", url, nil, &ret)
	return &ret, err
}

type createCommentParams struct {
	ParentEntity  string     `json:"parent_entity"`
	UserID        string     `json:"user_id"`
	Body          string     `json:"body"`
	AudreyID      string     `json:"audrey_id,omitempty"`
	CreateAt      *time.Time `json:"created_at,omitempty"`
	DeletedAt     *time.Time `json:"deleted_at,omitempty"`
	NeedsApproval bool       `json:"needs_approval,omitempty"`
}

func createComment(ts *testSetup, parentEntity string, userID string, body string, opts *createCommentParams) (*api.Comment, error) {
	url := "/v1/comments"
	params := &createCommentParams{
		ParentEntity: parentEntity,
		UserID:       userID,
		Body:         body,
	}
	if opts != nil {
		params.AudreyID = opts.AudreyID
		params.CreateAt = opts.CreateAt
		params.DeletedAt = opts.DeletedAt
		params.NeedsApproval = opts.NeedsApproval
	}
	var ret api.Comment
	err := request(ts, "POST", url, params, &ret)
	return &ret, err
}

func getCommentIDByAudreyID(ts *testSetup, audreyID string) (*api.ID, error) {
	url := "/v1/comments/legacy/" + audreyID + "/id"
	var ret api.ID
	err := request(ts, "GET", url, nil, &ret)
	return &ret, err
}

func getComment(ts *testSetup, commentID string) (*api.Comment, error) {
	url := "/v1/comments/" + commentID
	var ret api.Comment
	err := request(ts, "GET", url, nil, &ret)
	return &ret, err
}

type updateCommentParams struct {
	NeedsApproval *bool        `json:"needs_approval,omitempty"`
	Emotes        *[]api.Emote `json:"emotes,omitempty"`
}

func updateComment(ts *testSetup, commentID string, opts *updateCommentParams) (*api.Comment, error) {
	url := "/v1/comments/" + commentID
	var ret api.Comment
	err := request(ts, "PUT", url, opts, &ret)
	return &ret, err
}

func getComments(ts *testSetup, parentEntity string, limit *int, cursor *string) (*api.PaginatedComments, error) {
	url := "/v1/comments?parent_entity=" + parentEntity
	if limit != nil {
		url += fmt.Sprintf("&limit=%d", *limit)
	}
	if cursor != nil {
		url += "&cursor=" + *cursor
	}
	var ret api.PaginatedComments
	err := request(ts, "GET", url, nil, &ret)
	return &ret, err
}

func getCommentsSummaries(ts *testSetup, parentEntities []string) (*api.CommentsSummariesResponse, error) {
	url := "/v1/comments/summary?parent_entity=" + strings.Join(parentEntities, ",")

	var ret *api.CommentsSummariesResponse
	err := request(ts, "GET", url, nil, &ret)
	return ret, err
}

func deleteComment(ts *testSetup, commentID string, deletedAt *time.Time) (*api.Comment, error) {
	url := fmt.Sprintf("/v1/comments/%s?", commentID)
	if deletedAt != nil {
		url += fmt.Sprintf("&deleted_at=%s", deletedAt.Format(time.RFC3339))
	}
	var ret api.Comment
	err := request(ts, "DELETE", url, nil, &ret)
	return &ret, err
}

func deleteCommentsByParentAndUser(ts *testSetup, parentEntity string, userID string) error {
	url := fmt.Sprintf("/v1/comments/by_parent_and_user/%s/%s", parentEntity, userID)
	err := request(ts, "DELETE", url, nil, nil)
	return err
}

func getReactions(ts *testSetup, parentEntity string, userID string) (*api.Reactions, error) {
	url := fmt.Sprintf("/v1/reactions/%s/%s", parentEntity, userID)
	var ret api.Reactions
	err := request(ts, "GET", url, nil, &ret)
	return &ret, err
}

func getReactionsByUser(ts *testSetup, entities []string, userID string) (*api.ReactionList, error) {
	url := fmt.Sprintf("/v1/reactions/user_reactions/%s?parent_entity="+strings.Join(entities, ","), userID)
	var ret api.ReactionList
	err := request(ts, "GET", url, nil, &ret)
	return &ret, err
}

func createReaction(ts *testSetup, parentEntity string, userID string, emoteID string) (*api.Reactions, error) {
	url := fmt.Sprintf("/v1/reactions/%s/%s/%s", parentEntity, userID, emoteID)
	var ret api.Reactions
	err := request(ts, "PUT", url, nil, &ret)
	return &ret, err
}

func deleteReaction(ts *testSetup, parentEntity string, userID string, emoteID string) (*api.Reactions, error) {
	url := fmt.Sprintf("/v1/reactions/%s/%s/%s", parentEntity, userID, emoteID)
	var ret api.Reactions
	err := request(ts, "DELETE", url, nil, &ret)
	return &ret, err
}

func getReactionsSummary(ts *testSetup, parentEntity string) (*api.ReactionsSummary, error) {
	url := "/v1/reactions/summary/" + parentEntity
	var ret api.ReactionsSummary
	err := request(ts, "GET", url, nil, &ret)
	return &ret, err
}

func getReactionsSummaries(ts *testSetup, parentEntities []string) (*api.ReactionsSummariesResponse, error) {
	url := "/v1/reactions/summary?parent_entity=" + strings.Join(parentEntities, ",")
	var ret *api.ReactionsSummariesResponse
	err := request(ts, "GET", url, nil, &ret)
	return ret, err
}

func getReactionsSummariesForUsers(ts *testSetup, parentEntities []string, userIDs []string) (*api.ReactionsSummariesResponse, error) {
	url := "/v1/reactions/summary?parent_entity=" + strings.Join(parentEntities, ",") + "&user_ids=" + strings.Join(userIDs, ",")
	var ret *api.ReactionsSummariesResponse
	err := request(ts, "GET", url, nil, &ret)
	return ret, err
}

func TestIntegrations(t *testing.T) {
	eachTest := []struct {
		Name        string
		testFunc    func(t *testing.T, ts *testSetup)
		injectables injectables
	}{
		{"testIntegration_Post", testIntegration_Post, injectables{}},
		{"testIntegration_Comment", testIntegration_Comment, injectables{}},
		{"TestIntegration_Publish_WithMock", testIntegration_Publish_WithMock, injectables{&apimocks.ActivityPublisher{}}},
		{"TestIntegration_Shares", testIntegration_Shares, injectables{}},
		{"TestIntegration_Reactions", testIntegration_Reactions, injectables{}},
		{"testIntegration_ReactionsSummary", testIntegration_ReactionsSummary, injectables{}},
		{"testClientIntegrationAllClientTests", testClientIntegrationAllClientTests, injectables{}},
	}

	for _, f := range eachTest {
		f := f
		t.Run(f.Name, func(t *testing.T) {
			t.Parallel()
			testInfo, cleanShutdown := startServer(t, f.injectables)
			if testInfo == nil {
				t.Error("Unable to setup testing server")
				return
			}
			Convey("With "+testInfo.listenAddr, t, func(c C) {
				ts := &testSetup{
					host:        testInfo.listenAddr,
					injectables: f.injectables,
				}
				So(ts.Setup(), ShouldBeNil)
				c.Reset(ts.cancelFunc)

				f.testFunc(t, ts)
			})
			cleanShutdown(time.Second * 3)
		})
	}
}

func testIntegration_Post(t *testing.T, ts *testSetup) {
	r := rand.New(rand.NewSource(time.Now().UnixNano()))
	Convey("Should not be able to update a non existent post", func() {
		err := updatePost(ts, strconv.Itoa(int(r.Int63())), updatePostParams{
			Emotes: &[]api.Emote{
				{
					ID:    10,
					Start: 20,
					End:   30,
					Set:   40,
				},
			}})
		So(err, ShouldNotBeNil)
	})

	Convey("Should be able to create, get, update, and delete a post", func() {
		testBody := "TestClientIntegration_Post " + timestamp()
		created, err := createPost(ts, testUserID, testBody, nil)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.ID, ShouldNotBeNil)
		So(created.Body, ShouldEqual, testBody)
		So(created.Emotes, ShouldBeNil)

		_, err = createPost(ts, testUserID, testBody, nil)
		So(err, ShouldBeNil)

		post, err := getPost(ts, created.ID)
		So(err, ShouldBeNil)
		So(post, ShouldNotBeNil)
		So(post.Body, ShouldEqual, testBody)
		So(post.Emotes, ShouldBeNil)

		comment, err := createComment(ts, "post:"+created.ID, testUserID, "A comment", nil)
		So(err, ShouldBeNil)
		So(comment.ParentEntity.ID(), ShouldEqual, created.ID)

		err = updatePost(ts, created.ID, updatePostParams{
			Emotes: &[]api.Emote{
				{
					ID:    10,
					Start: 20,
					End:   30,
					Set:   40,
				},
			}})
		So(err, ShouldBeNil)

		post, err = getPost(ts, created.ID)
		So(err, ShouldBeNil)
		So(post, ShouldNotBeNil)
		So(len(*post.Emotes), ShouldEqual, 1)
		So((*post.Emotes)[0].Set, ShouldEqual, 40)

		_, err = getPostIDsByUser(ts, testUserID, 100, "")
		So(err, ShouldBeNil)

		deleted, err := deletePost(ts, created.ID, nil)
		So(err, ShouldBeNil)
		So(deleted, ShouldNotBeNil)
		So(deleted.DeletedAt, ShouldNotBeNil)

		post, err = getPost(ts, created.ID)
		So(err, ShouldNotBeNil)
		So(errorCode(err), ShouldEqual, 404)
	})

	Convey("Should be able to create a post explicit no emotes", func() {
		testBody := "TestClientIntegration_Post " + timestamp()
		createParams := &createPostParams{
			Emotes: &[]api.Emote{},
		}
		t.Log("Starting a empty create")
		created, err := createPost(ts, testUserID, testBody, createParams)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(len(*created.Emotes), ShouldEqual, 0)
		So(created.Emotes, ShouldNotBeNil)

		gottenPost, err := getPost(ts, created.ID)
		So(err, ShouldBeNil)
		So(len(*gottenPost.Emotes), ShouldEqual, 0)
		So(gottenPost.Emotes, ShouldNotBeNil)
	})

	Convey("Should be able to create a post with emotes", func() {
		testBody := "TestClientIntegration_Post " + timestamp()
		createParams := &createPostParams{
			Emotes: &[]api.Emote{
				{
					ID:    0,
					Start: 0,
					End:   1,
					Set:   2,
				},
			},
		}
		created, err := createPost(ts, testUserID, testBody, createParams)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.ID, ShouldNotBeNil)
		So(len(*created.Emotes), ShouldEqual, 1)
		So((*created.Emotes)[0].End, ShouldEqual, 1)

		Convey("Then change the emotes", func() {
			err := updatePost(ts, created.ID, updatePostParams{
				Emotes: &[]api.Emote{
					{
						ID:    1,
						Start: 2,
						End:   3,
						Set:   4,
					},
				}})
			So(err, ShouldBeNil)
			post, err := getPost(ts, created.ID)
			So(err, ShouldBeNil)
			So(len(*post.Emotes), ShouldEqual, 1)
			So((*post.Emotes)[0].Start, ShouldEqual, 2)
		})
	})

	Convey("Should be able to create a post with embed urls", func() {
		testBody := "TestClientIntegration_Post " + timestamp()
		testURL := "https://www.twitch.tv"
		createParams := &createPostParams{
			EmbedURLs: &[]string{testURL},
		}
		created, err := createPost(ts, testUserID, testBody, createParams)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.ID, ShouldNotBeNil)
		So(created.EmbedURLs, ShouldNotBeNil)
		So((*created.EmbedURLs), ShouldHaveLength, 1)
		So(created.EmbedEntities, ShouldBeNil)

		Convey("Then change the embed urls", func() {
			testURL2 := "https://www.twitchout.com"
			err := updatePost(ts, created.ID, updatePostParams{
				EmbedURLs: &[]string{testURL2},
			})
			So(err, ShouldBeNil)
			post, err := getPost(ts, created.ID)
			So(err, ShouldBeNil)
			So(post.EmbedURLs, ShouldNotBeNil)
			So((*post.EmbedURLs), ShouldHaveLength, 1)
			So(post.EmbedEntities, ShouldBeNil)
		})
		Convey("Then change the embed entities", func() {
			err := updatePost(ts, created.ID, updatePostParams{
				EmbedEntities: &[]entity.Entity{entity.New("vod", "1234")},
			})
			So(err, ShouldBeNil)
			post, err := getPost(ts, created.ID)
			So(err, ShouldBeNil)
			So(post.EmbedEntities, ShouldNotBeNil)
			So(*post.EmbedEntities, ShouldHaveLength, 1)
			So((*post.EmbedEntities)[0].Encode(), ShouldEqual, "vod:1234")
		})
	})

	Convey("Should not be able to create a post with an empty body and no embedURLs", func() {
		var emptyPost api.Post
		created, err := createPost(ts, testUserID, "", nil)
		So(created, ShouldResemble, &emptyPost)
		So(err, ShouldNotBeNil)

		Convey("Should be able to create a post with an empty body so long as embedURLs are provided", func() {
			testURL := "https://www.twitch.tv"
			createParams := &createPostParams{
				EmbedURLs: &[]string{testURL},
			}
			created, err := createPost(ts, testUserID, "", createParams)
			So(created, ShouldNotBeNil)
			So(err, ShouldBeNil)

			post, err := getPost(ts, created.ID)
			So(err, ShouldBeNil)
			So(post.Body, ShouldEqual, "")
			So(*post.EmbedURLs, ShouldHaveLength, 1)
		})
	})

	Convey("Updating a post with nil parameters should return an error and the post shouldn't be updated", func() {
		testBody := "TestClientIntegration_Post " + timestamp()
		createParams := &createPostParams{
			EmbedURLs: &[]string{"https://www.twitch.tv"},
			Emotes: &[]api.Emote{
				{
					ID:    1,
					Start: 2,
					End:   3,
					Set:   4,
				},
			},
		}
		created, err := createPost(ts, testUserID, testBody, createParams)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.ID, ShouldNotBeNil)
		So(created.EmbedURLs, ShouldNotBeNil)
		So((*created.EmbedURLs), ShouldHaveLength, 1)
		So(created.Emotes, ShouldNotBeNil)
		So((*created.Emotes), ShouldHaveLength, 1)

		err = updatePost(ts, created.ID, updatePostParams{})
		So(err, ShouldNotBeNil)

		post, _ := getPost(ts, created.ID)
		So(post.EmbedURLs, ShouldNotBeNil)
		So((*post.EmbedURLs), ShouldHaveLength, 1)
	})

	Convey("Should be able to create a post with a defined created and deleted at", func() {
		testBody := "TestClientIntegration_Post " + timestamp()
		t := time.Now().Add(-time.Hour)
		createParams := &createPostParams{
			CreateAt:  &t,
			DeletedAt: &t,
		}
		created, err := createPost(ts, testUserID, testBody, createParams)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.ID, ShouldNotBeNil)
		So(created.CreatedAt.Unix(), ShouldEqual, t.Unix())
		So(created.DeletedAt.Unix(), ShouldEqual, t.Unix())
	})

	Convey("Should be able to delete a post with a defined deleted at", func() {
		testBody := "TestClientIntegration_Post " + timestamp()
		created, err := createPost(ts, testUserID, testBody, nil)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)

		deletedAt := time.Now().Add(-time.Hour)
		deleted, err := deletePost(ts, created.ID, &deletedAt)
		So(err, ShouldBeNil)
		So(deleted, ShouldNotBeNil)
		So(deleted.DeletedAt.Unix(), ShouldEqual, deletedAt.Unix())
	})

	Convey("Should be able to create a post with an audrey_id", func() {
		testBody := "TestClientIntegration_Post " + timestamp()
		audreyID := "audrey:" + timestamp()
		createParams := &createPostParams{
			AudreyID: audreyID,
		}
		created, err := createPost(ts, testUserID, testBody, createParams)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.ID, ShouldNotBeNil)
		So(created.AudreyID, ShouldEqual, audreyID)

		post, err := getPost(ts, created.ID)
		So(err, ShouldBeNil)
		So(post, ShouldNotBeNil)
		So(post.AudreyID, ShouldEqual, audreyID)

		idI, err := tryXTimes(3, func() (interface{}, error) {
			return getPostIDByAudreyID(ts, audreyID)
		})
		So(err, ShouldBeNil)
		id := idI.(*api.ID)
		So(id, ShouldNotBeNil)
		So(id.ID, ShouldEqual, created.ID)
	})

	Convey("Should be able to create a post with EmbedURLs", func() {
		testBody := "TestClientIntegration_Post " + timestamp()

		embedURLs := []string{"url1", "url2", "url3"}

		createParams := &createPostParams{
			EmbedURLs: &embedURLs,
		}
		created, err := createPost(ts, testUserID, testBody, createParams)
		So(err, ShouldBeNil)
		So(created.EmbedURLs, ShouldNotBeNil)

		postEmbedURLs := *created.EmbedURLs
		So(postEmbedURLs, ShouldHaveLength, len(embedURLs))
		for _, embedURL := range embedURLs {
			So(postEmbedURLs, ShouldContain, embedURL)
		}

		post, err := getPost(ts, created.ID)
		So(err, ShouldBeNil)
		So(post.EmbedURLs, ShouldNotBeNil)

		postEmbedURLs = *post.EmbedURLs
		So(postEmbedURLs, ShouldHaveLength, len(embedURLs))
		for _, embedURL := range embedURLs {
			So(postEmbedURLs, ShouldContain, embedURL)
		}

		Convey("When given an empty slice of embed urls", func() {
			embedURLs := []string{}

			createParams = &createPostParams{
				EmbedURLs: &embedURLs,
			}
			created, err = createPost(ts, testUserID, testBody, createParams)
			So(err, ShouldBeNil)
			So(created.EmbedURLs, ShouldNotBeNil)
			So(*created.EmbedURLs, ShouldBeEmpty)
		})

		Convey("When no embed urls are passed in", func() {
			// When no EmbedURLs are passed in on creation, the field is not set on the post
			createParams = &createPostParams{}
			created, err = createPost(ts, testUserID, testBody, createParams)
			So(err, ShouldBeNil)
			So(created.EmbedURLs, ShouldBeNil)
		})
	})

	Convey("Should be able to retrieve multiple posts by ids", func() {
		testBody1 := "TestClientIntegration_Post " + timestamp()
		created1, err := createPost(ts, testUserID, testBody1, nil)
		So(err, ShouldBeNil)
		So(created1, ShouldNotBeNil)

		testBody2 := "TestClientIntegration_Post " + timestamp()
		created2, err := createPost(ts, testUserID, testBody2, nil)
		So(err, ShouldBeNil)
		So(created2, ShouldNotBeNil)

		posts, err := getPosts(ts, []string{created1.ID, created2.ID})
		So(err, ShouldBeNil)
		So(posts, ShouldNotBeNil)
		So(len(posts.Items), ShouldEqual, 2)
		So(created1, ShouldBeIn, posts.Items)
		So(created2, ShouldBeIn, posts.Items)

		// And with the same ID twice
		posts, err = getPosts(ts, []string{created1.ID, created1.ID})
		So(err, ShouldBeNil)
		So(posts, ShouldNotBeNil)
		So(len(posts.Items), ShouldEqual, 1)
		So(created1, ShouldBeIn, posts.Items)
	})

	Convey("getPostByUser Should be able to get multiple pages of post IDs by user ID", func() {

		userID := newUserID()

		Convey("Should allow iterating through pages using a cursor", func() {

			postIDSet := make(map[string]struct{})

			// Create two posts and record their post IDs
			for i := 0; i < 2; i++ {
				testBody := "TestClientIntegration_Post " + timestamp()
				created, err := createPost(ts, userID, testBody, nil)
				So(err, ShouldBeNil)
				So(created, ShouldNotBeNil)

				postIDSet[created.ID] = struct{}{}
			}

			// Get the first page of post IDs
			page, err := getPostIDsByUser(ts, userID, 1, "")
			So(err, ShouldBeNil)
			So(page, ShouldNotBeNil)
			So(page.Cursor, ShouldNotEqual, "")
			So(page.PostIDs, ShouldHaveLength, 1)
			page1PostID := page.PostIDs[0]
			So(postIDSet, ShouldContainKey, page1PostID)

			// Get the second page of post IDs using the cursor
			page, err = getPostIDsByUser(ts, userID, 1, page.Cursor)
			So(err, ShouldBeNil)
			So(page, ShouldNotBeNil)
			So(page.Cursor, ShouldEqual, "")
			So(page.PostIDs, ShouldHaveLength, 1)
			page2PostID := page.PostIDs[0]
			So(postIDSet, ShouldContainKey, page2PostID)
			So(page1PostID, ShouldNotEqual, page2PostID)
		})

		Convey("Should not return deleted post IDs", func() {
			testBody := "TestClientIntegration_Post " + timestamp()
			created, err := createPost(ts, userID, testBody, nil)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)

			now := time.Now()
			_, err = deletePost(ts, created.ID, &now)
			So(err, ShouldBeNil)

			page, err := getPostIDsByUser(ts, userID, 100, "")
			So(err, ShouldBeNil)
			So(page, ShouldNotBeNil)
			So(page.Cursor, ShouldEqual, "")
			So(page.PostIDs, ShouldHaveLength, 0)
		})

	})

	Convey("Should not return posts that have been deleted within batch get posts", func() {
		testBody1 := "TestClientIntegration_Post " + timestamp()
		created1, err := createPost(ts, testUserID, testBody1, nil)
		So(err, ShouldBeNil)
		So(created1, ShouldNotBeNil)

		testBody2 := "TestClientIntegration_Post " + timestamp()
		created2, err := createPost(ts, testUserID, testBody2, nil)
		So(err, ShouldBeNil)
		So(created2, ShouldNotBeNil)

		deletedAt := time.Now().Add(-time.Hour)
		deleted, err := deletePost(ts, created1.ID, &deletedAt)
		So(err, ShouldBeNil)
		So(deleted, ShouldNotBeNil)

		posts, err := getPosts(ts, []string{created1.ID, created2.ID})
		So(err, ShouldBeNil)
		So(posts, ShouldNotBeNil)
		So(len(posts.Items), ShouldEqual, 1)
		So(created1, ShouldNotBeIn, posts.Items)
		So(created2, ShouldBeIn, posts.Items)
	})

	Convey("Should return an error when no ids are provided to get multi post", func() {
		posts, err := getPosts(ts, []string{""})
		So(err, ShouldNotBeNil)
		So(posts, ShouldNotBeNil)
		So(len(posts.Items), ShouldEqual, 0)
	})

	Convey("Should return an error when try to update deleted post", func() {
		testBody := "TestClientIntegration_Post " + timestamp()
		created, err := createPost(ts, testUserID, testBody, nil)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.ID, ShouldNotBeNil)
		So(created.Body, ShouldEqual, testBody)
		So(created.Emotes, ShouldBeNil)
		So(created.DeletedAt, ShouldBeNil)

		post, err := getPost(ts, created.ID)
		So(post, ShouldNotBeNil)
		So(err, ShouldBeNil)

		deleted, err := deletePost(ts, created.ID, nil)
		So(err, ShouldBeNil)
		So(deleted, ShouldNotBeNil)
		So(deleted.DeletedAt, ShouldNotBeNil)

		post, err = getPost(ts, created.ID)
		So(post, ShouldNotBeNil)
		So(err, ShouldNotBeNil)
		So(errorCode(err), ShouldEqual, 404)

		err = updatePost(ts, created.ID, updatePostParams{
			Emotes: &[]api.Emote{
				{
					ID:    10,
					Start: 20,
					End:   30,
					Set:   40,
				},
			}})
		So(err, ShouldNotBeNil)
		So(errorCode(err), ShouldEqual, 404)
	})
}

func tryXTimes(count int, f func() (interface{}, error)) (interface{}, error) {
	var a interface{}
	var b error
	sleepTime := time.Millisecond * 100
	for i := 0; i < count; i++ {
		a, b = f()
		if b == nil {
			return a, b
		}
		time.Sleep(sleepTime)
		sleepTime *= 2
	}
	return a, b
}

func testIntegration_Comment(t *testing.T, ts *testSetup) {
	Convey("Should be able to create, get, and delete a comment", func() {
		parent := "test:parent:" + timestamp()
		testBody := "TestClientIntegration_Comment " + timestamp()
		created, err := createComment(ts, parent, testUserID, testBody, nil)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.ID, ShouldNotBeNil)
		So(created.ParentEntity.Encode(), ShouldEqual, parent)
		So(created.Body, ShouldEqual, testBody)

		comment, err := getComment(ts, created.ID)
		So(err, ShouldBeNil)
		So(comment, ShouldNotBeNil)
		So(comment.Body, ShouldEqual, testBody)
		So(comment.NeedsApproval, ShouldBeFalse)

		commentsSummaries, err := getCommentsSummaries(ts, []string{parent})
		So(err, ShouldBeNil)
		So(commentsSummaries, ShouldNotBeNil)
		So(len(commentsSummaries.Items), ShouldEqual, 1)
		So(commentsSummaries.Items[0].Total, ShouldEqual, 1)

		deleted, err := deleteComment(ts, created.ID, nil)
		So(err, ShouldBeNil)
		So(deleted, ShouldNotBeNil)
		So(deleted.DeletedAt, ShouldNotBeNil)

		comment, err = getComment(ts, created.ID)
		So(err, ShouldNotBeNil)
		So(errorCode(err), ShouldEqual, 404)

		commentsSummaries, err = getCommentsSummaries(ts, []string{parent})
		So(err, ShouldBeNil)
		So(commentsSummaries, ShouldNotBeNil)
		So(len(commentsSummaries.Items), ShouldEqual, 1)
		So(commentsSummaries.Items[0].Total, ShouldEqual, 0)
	})

	Convey("Should be able to create a comment with a defined created and deleted at", func() {
		parent := "test:parent:" + timestamp()
		testBody := "TestClientIntegration_Comment " + timestamp()
		t := time.Now().Add(-time.Hour)
		params := &createCommentParams{
			CreateAt:  &t,
			DeletedAt: &t,
		}
		created, err := createComment(ts, parent, testUserID, testBody, params)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.ID, ShouldNotBeNil)
		So(created.CreatedAt.Unix(), ShouldEqual, t.Unix())
		So(created.DeletedAt.Unix(), ShouldEqual, t.Unix())
	})

	Convey("Should be able to delete a comment with a defined deleted at", func() {
		parent := "test:parent:" + timestamp()
		testBody := "TestClientIntegration_Comment " + timestamp()
		created, err := createComment(ts, parent, testUserID, testBody, nil)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)

		deletedAt := time.Now().Add(-time.Hour)
		deleted, err := deleteComment(ts, created.ID, &deletedAt)
		So(err, ShouldBeNil)
		So(deleted, ShouldNotBeNil)
		So(deleted.DeletedAt.Unix(), ShouldEqual, deletedAt.Unix())
	})

	Convey("Should be able to create a comment with an audrey_id", func() {
		parent := "test:parent:" + timestamp()
		testBody := "TestClientIntegration_Comment " + timestamp()
		audreyID := "test:audrey:" + timestamp()
		createParams := &createCommentParams{
			AudreyID: audreyID,
		}
		created, err := createComment(ts, parent, testUserID, testBody, createParams)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.ID, ShouldNotBeNil)
		So(created.AudreyID, ShouldEqual, audreyID)

		comment, err := getComment(ts, created.ID)
		So(err, ShouldBeNil)
		So(comment, ShouldNotBeNil)
		So(comment.AudreyID, ShouldEqual, audreyID)

		idI, err := tryXTimes(3, func() (interface{}, error) {
			return getCommentIDByAudreyID(ts, audreyID)
		})
		So(err, ShouldBeNil)
		id := idI.(*api.ID)
		So(id, ShouldNotBeNil)
		So(id.ID, ShouldEqual, created.ID)
	})

	Convey("Should be able to create a comment with a defined needs approval", func() {
		parent := "test:parent:" + timestamp()
		testBody := "TestClientIntegration_Comment " + timestamp()
		params := &createCommentParams{
			NeedsApproval: true,
		}
		created, err := createComment(ts, parent, testUserID, testBody, params)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.ID, ShouldNotBeNil)
		So(created.NeedsApproval, ShouldBeTrue)
	})

	Convey("Should be able to update if a comment needs approval", func() {
		parent := "test:parent:" + timestamp()
		testBody := "TestClientIntegration_Comment " + timestamp()
		created, err := createComment(ts, parent, testUserID, testBody, nil)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.NeedsApproval, ShouldBeFalse)

		updateParams := &updateCommentParams{
			NeedsApproval: Bool(true),
		}
		updated, err := updateComment(ts, created.ID, updateParams)
		So(err, ShouldBeNil)
		So(updated, ShouldNotBeNil)
		So(updated.NeedsApproval, ShouldBeTrue)

		updateParams2 := &updateCommentParams{
			NeedsApproval: Bool(false),
		}
		updated2, err := updateComment(ts, created.ID, updateParams2)
		So(err, ShouldBeNil)
		So(updated2, ShouldNotBeNil)
		So(updated2.NeedsApproval, ShouldBeFalse)
	})

	Convey("Should update comments summary count when needs approval changes", func() {
		parent := "test:parent:" + timestamp()
		testBody := "TestClientIntegration_Comment " + timestamp()
		created, err := createComment(ts, parent, testUserID, testBody, nil)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)

		commentsSummaries, err := getCommentsSummaries(ts, []string{parent})
		So(err, ShouldBeNil)
		So(commentsSummaries, ShouldNotBeNil)
		So(len(commentsSummaries.Items), ShouldEqual, 1)
		So(commentsSummaries.Items[0].Total, ShouldEqual, 1)

		// update the comment to flag it for approval
		updateParams := &updateCommentParams{
			NeedsApproval: Bool(true),
		}
		updated, err := updateComment(ts, created.ID, updateParams)
		So(err, ShouldBeNil)
		So(updated, ShouldNotBeNil)

		commentsSummariesEdit, err := getCommentsSummaries(ts, []string{parent})
		So(err, ShouldBeNil)
		So(commentsSummariesEdit, ShouldNotBeNil)
		So(len(commentsSummariesEdit.Items), ShouldEqual, 1)
		So(commentsSummariesEdit.Items[0].Total, ShouldEqual, 0)

		// update the comment to deny it
		updateParams2 := &updateCommentParams{
			NeedsApproval: Bool(false),
		}
		updated2, err := updateComment(ts, created.ID, updateParams2)
		So(err, ShouldBeNil)
		So(updated2, ShouldNotBeNil)

		commentsSummariesEdit2, err := getCommentsSummaries(ts, []string{parent})
		So(err, ShouldBeNil)
		So(commentsSummariesEdit2, ShouldNotBeNil)
		So(len(commentsSummariesEdit2.Items), ShouldEqual, 1)
		So(commentsSummariesEdit2.Items[0].Total, ShouldEqual, 1)
	})

	Convey("Should not be able to update a comment if it doesn't exist", func() {
		updateParams := &updateCommentParams{
			NeedsApproval: Bool(true),
		}
		_, err := updateComment(ts, uuid.NewV4().String(), updateParams)
		So(err, ShouldNotBeNil)
		So(errorCode(err), ShouldEqual, http.StatusNotFound)
	})

	Convey("Updating comment emotes", func() {
		parent := "test:parent:" + timestamp()
		testBody := "TestClientIntegration_Comment " + timestamp()
		created, err := createComment(ts, parent, testUserID, testBody, nil)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.Emotes, ShouldBeNil)

		Convey("when explicitly given no emotes", func() {
			noEmotes := []api.Emote{}
			updatedComment, err := updateComment(ts, created.ID, &updateCommentParams{
				Emotes: &noEmotes,
			})
			So(err, ShouldBeNil)
			So(updatedComment, ShouldNotBeNil)
			So(updatedComment.Emotes, ShouldNotBeNil)
			So(*updatedComment.Emotes, ShouldBeEmpty)
		})

		emotes := []api.Emote{
			{
				ID:    10,
				Start: 20,
				End:   30,
				Set:   40,
			}, {
				ID:    11,
				Start: 40,
				End:   50,
				Set:   40,
			},
		}

		Convey("when given emotes", func() {
			updatedComment, err := updateComment(ts, created.ID, &updateCommentParams{
				Emotes: &emotes,
			})
			So(err, ShouldBeNil)
			So(updatedComment, ShouldNotBeNil)
			So(updatedComment.Emotes, ShouldNotBeNil)
			So(*updatedComment.Emotes, ShouldResemble, emotes)

			comment, err := getComment(ts, created.ID)
			So(err, ShouldBeNil)
			So(comment, ShouldNotBeNil)
			So(comment.Emotes, ShouldNotBeNil)
			So(*comment.Emotes, ShouldResemble, emotes)
		})

		Convey("should not affect other fields", func() {
			// Set the comment's NeedApproval field to true.
			updatedComment, err := updateComment(ts, created.ID, &updateCommentParams{
				NeedsApproval: Bool(true),
			})
			So(err, ShouldBeNil)
			So(updatedComment, ShouldNotBeNil)
			So(updatedComment.NeedsApproval, ShouldBeTrue)

			commentSummaries, err := getCommentsSummaries(ts, []string{parent})
			So(err, ShouldBeNil)
			So(commentSummaries, ShouldNotBeNil)
			So(commentSummaries.Items, ShouldHaveLength, 1)
			So(commentSummaries.Items[0].Total, ShouldEqual, 0)

			// Update the emotes.
			updatedComment, err = updateComment(ts, created.ID, &updateCommentParams{
				Emotes: &emotes,
			})
			So(err, ShouldBeNil)
			So(updatedComment, ShouldNotBeNil)
			So(updatedComment.NeedsApproval, ShouldBeTrue)

			// Verify that NeedsApproval remains unchanged.
			comment, err := getComment(ts, created.ID)
			So(err, ShouldBeNil)
			So(comment, ShouldNotBeNil)
			So(updatedComment.NeedsApproval, ShouldBeTrue)

			// Verify comment summary remains unchanged.
			commentSummaries, err = getCommentsSummaries(ts, []string{parent})
			So(err, ShouldBeNil)
			So(commentSummaries, ShouldNotBeNil)
			So(commentSummaries.Items, ShouldHaveLength, 1)
			So(commentSummaries.Items[0].Total, ShouldEqual, 0)
		})
	})

	Convey("Should be able to approve or deny a comment multiple times and keep an accurate comments summary count", func() {
		parent := "test:parent:" + timestamp()
		testBody := "TestClientIntegration_Comment " + timestamp()
		params := &createCommentParams{
			NeedsApproval: true,
		}
		created, err := createComment(ts, parent, testUserID, testBody, params)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.NeedsApproval, ShouldBeTrue)

		commentsSummaries, err := getCommentsSummaries(ts, []string{parent})
		So(err, ShouldBeNil)
		So(commentsSummaries, ShouldNotBeNil)
		So(len(commentsSummaries.Items), ShouldEqual, 0)

		approveParams := &updateCommentParams{
			NeedsApproval: Bool(false),
		}
		_, err = updateComment(ts, created.ID, approveParams)
		So(err, ShouldBeNil)
		_, err = updateComment(ts, created.ID, approveParams)
		So(err, ShouldBeNil)

		commentsSummariesEdit, err := getCommentsSummaries(ts, []string{parent})
		So(err, ShouldBeNil)
		So(commentsSummariesEdit, ShouldNotBeNil)
		So(len(commentsSummariesEdit.Items), ShouldEqual, 1)
		So(commentsSummariesEdit.Items[0].Total, ShouldEqual, 1)

		denyParams := &updateCommentParams{
			NeedsApproval: Bool(true),
		}
		_, err = updateComment(ts, created.ID, denyParams)
		So(err, ShouldBeNil)
		_, err = updateComment(ts, created.ID, denyParams)
		So(err, ShouldBeNil)

		commentsSummariesEdit2, err := getCommentsSummaries(ts, []string{parent})
		So(err, ShouldBeNil)
		So(commentsSummariesEdit2, ShouldNotBeNil)
		So(len(commentsSummariesEdit2.Items), ShouldEqual, 1)
		So(commentsSummariesEdit2.Items[0].Total, ShouldEqual, 0)
	})

	Convey("Should be able to retrieve comments by parent entity", func() {
		parent := "test:parent:" + timestamp()

		testBody1 := "TestClientIntegration_Comment " + timestamp()
		c1, err := createComment(ts, parent, testUserID, testBody1, nil)
		So(err, ShouldBeNil)
		So(c1, ShouldNotBeNil)

		testBody2 := "TestClientIntegration_Comment " + timestamp()
		c2, err := createComment(ts, parent, testUserID, testBody2, nil)
		So(err, ShouldBeNil)
		So(c2, ShouldNotBeNil)

		comments, err := getComments(ts, parent, nil, nil)
		So(err, ShouldBeNil)
		So(len(comments.Items), ShouldEqual, 2)
		So(comments.Cursor, ShouldEqual, "")

		// Comments are returned in descending order
		So(comments.Items[0].Body, ShouldEqual, testBody2)
		So(comments.Items[1].Body, ShouldEqual, testBody1)
	})

	Convey("Should return an empty array of comments for a parent entity doesn't exist", func() {
		parent := "test:parent:" + timestamp()

		comments, err := getComments(ts, parent, nil, nil)
		So(err, ShouldBeNil)
		So(len(comments.Items), ShouldEqual, 0)
		So(comments.Cursor, ShouldEqual, "")
	})

	Convey("Should be able to retrieve a list of comments summaries by parent entity", func() {
		parent1 := "test:parent:" + timestamp()
		testBody1 := "TestClientIntegration_Comment " + timestamp()
		c1, err := createComment(ts, parent1, testUserID, testBody1, nil)
		So(err, ShouldBeNil)
		So(c1, ShouldNotBeNil)

		testBody2 := "TestClientIntegration_Comment " + timestamp()
		c2, err := createComment(ts, parent1, testUserID, testBody2, nil)
		So(err, ShouldBeNil)
		So(c2, ShouldNotBeNil)

		parent2 := "test:parent:" + timestamp()
		testBody3 := "TestClientIntegration_Comment " + timestamp()
		c3, err := createComment(ts, parent2, testUserID, testBody3, nil)
		So(err, ShouldBeNil)
		So(c3, ShouldNotBeNil)

		commentsSummaries, err := getCommentsSummaries(ts, []string{parent1, parent2})
		So(err, ShouldBeNil)
		So(commentsSummaries, ShouldNotBeNil)
		So(len(commentsSummaries.Items), ShouldEqual, 2)
		seen := []bool{false, false}
		for _, item := range commentsSummaries.Items {
			if item.ParentEntity.Encode() == parent1 {
				So(item.Total, ShouldEqual, 2)
				seen[0] = true
			}
			if item.ParentEntity.Encode() == parent2 {
				So(item.Total, ShouldEqual, 1)
				seen[1] = true
			}
		}
		So(seen[0], ShouldBeTrue)
		So(seen[1], ShouldBeTrue)
	})

	Convey("Should not return deleted comments", func() {
		parent := "test:parent:" + timestamp()

		testBody1 := "TestClientIntegration_Comment " + timestamp()
		c1, err := createComment(ts, parent, testUserID, testBody1, nil)
		So(err, ShouldBeNil)
		So(c1, ShouldNotBeNil)

		testBody2 := "TestClientIntegration_Comment " + timestamp()
		c2, err := createComment(ts, parent, testUserID, testBody2, nil)
		So(err, ShouldBeNil)
		So(c2, ShouldNotBeNil)

		comments1, err := getComments(ts, parent, nil, nil)
		So(err, ShouldBeNil)
		So(len(comments1.Items), ShouldEqual, 2)

		deletedAt := time.Now().Add(-time.Hour)
		deleted, err := deleteComment(ts, c1.ID, &deletedAt)
		So(err, ShouldBeNil)
		So(deleted, ShouldNotBeNil)
		So(deleted.DeletedAt.Unix(), ShouldEqual, deletedAt.Unix())

		comments2, err := getComments(ts, parent, nil, nil)
		So(err, ShouldBeNil)
		So(len(comments2.Items), ShouldEqual, 1)
	})

	Convey("Should be able to use a cursor to retrieve paginated comments by parent entity", func() {
		parent := "test:parent:" + timestamp()

		testBody1 := "TestClientIntegration_Comment " + timestamp()
		c1, err := createComment(ts, parent, testUserID, testBody1, nil)
		So(err, ShouldBeNil)
		So(c1, ShouldNotBeNil)

		testBody2 := "TestClientIntegration_Comment " + timestamp()
		c2, err := createComment(ts, parent, testUserID, testBody2, nil)
		So(err, ShouldBeNil)
		So(c2, ShouldNotBeNil)

		comments1, err := getComments(ts, parent, intPtr(1), nil)
		So(err, ShouldBeNil)
		So(len(comments1.Items), ShouldEqual, 1)
		So(comments1.Cursor, ShouldNotEqual, "")
		So(comments1.Items[0].Body, ShouldEqual, testBody2)

		comments2, err := getComments(ts, parent, intPtr(2), &comments1.Cursor)
		So(err, ShouldBeNil)
		So(len(comments2.Items), ShouldEqual, 1)
		So(comments2.Cursor, ShouldEqual, "")
		So(comments2.Items[0].Body, ShouldEqual, testBody1)
	})

	Convey("Should return an error if given a malformed parent entity", func() {
		_, err := createComment(ts, "MalformedParent", testUserID, "TestClientIntegration_Comment", nil)
		So(err, ShouldNotBeNil)
		So(errorCode(err), ShouldEqual, http.StatusBadRequest)
	})

	Convey("Should be able to delete comments by parent_entity and user_id", func() {
		parent := "test:parent:" + timestamp()

		testBody1 := "TestClientIntegration_Comment " + timestamp()
		c1, err := createComment(ts, parent, testUserID, testBody1, nil)
		So(err, ShouldBeNil)
		So(c1, ShouldNotBeNil)

		testBody2 := "TestClientIntegration_Comment " + timestamp()
		c2, err := createComment(ts, parent, testUserIDAlt, testBody2, nil)
		So(err, ShouldBeNil)
		So(c2, ShouldNotBeNil)

		comments1, err := getComments(ts, parent, nil, nil)
		So(err, ShouldBeNil)
		So(len(comments1.Items), ShouldEqual, 2)

		err = deleteCommentsByParentAndUser(ts, parent, testUserID)
		So(err, ShouldBeNil)

		_, err = getComment(ts, c1.ID)
		So(err, ShouldNotBeNil)
		So(errorCode(err), ShouldEqual, http.StatusNotFound)

		comments2, err := getComments(ts, parent, nil, nil)
		So(err, ShouldBeNil)
		So(len(comments2.Items), ShouldEqual, 1)

	})

	Convey("Shouldn't increment/decrement comment count if you create/delete a comment that needs approval", func() {
		parent := "test:parent:" + timestamp()
		testBody := "TestClientIntegration_Comment " + timestamp()

		params := &createCommentParams{
			NeedsApproval: false,
		}
		_, err := createComment(ts, parent, testUserID, testBody, params)
		So(err, ShouldBeNil)

		// Comment does not need approval, so parent should report one comment.
		commentsSummaries, err := getCommentsSummaries(ts, []string{parent})
		So(err, ShouldBeNil)
		So(commentsSummaries.Items[0].Total, ShouldEqual, 1)

		params.NeedsApproval = true
		createdNeedsApproval, err := createComment(ts, parent, testUserID, testBody, params)
		So(err, ShouldBeNil)

		// Comment needs approval, so creating it should not affect total reported comments by parent.
		commentsSummaries, err = getCommentsSummaries(ts, []string{parent})
		So(err, ShouldBeNil)
		So(commentsSummaries.Items[0].Total, ShouldEqual, 1)

		_, err = deleteComment(ts, createdNeedsApproval.ID, &createdNeedsApproval.CreatedAt)
		So(err, ShouldBeNil)

		// Comment needs approval, so deleting it should not affect total reported comments by parent.
		commentsSummaries, err = getCommentsSummaries(ts, []string{parent})
		So(err, ShouldBeNil)
		So(commentsSummaries.Items[0].Total, ShouldEqual, 1)
	})
}

func testIntegration_Publish_WithMock(t *testing.T, ts *testSetup) {
	pub := ts.injectables.CommentActivityPublisher.(*apimocks.ActivityPublisher)

	Convey("Should be able to create a comment and publish the activity", func() {
		pub.On("Publish", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(nil)

		// We need to create a post because of the whitelist logic.  Once we get rid of the whitelist
		// logic we can get rid of this and the duplo.comment_activity_users setting in main
		testPostBody := "TestClientIntegration_Post " + timestamp()
		p, err := createPost(ts, testUserID, testPostBody, nil)
		So(err, ShouldBeNil)
		So(p, ShouldNotBeNil)

		//parent := "test:post:" + timestamp()
		parent := "post:" + p.ID
		testCommentBody := "TestClientIntegration_Comment " + timestamp()
		c, err := createComment(ts, parent, testUserID, testCommentBody, nil)
		So(err, ShouldBeNil)
		So(c, ShouldNotBeNil)

		ent := entity.New(entity.NamespaceComment, c.ID)
		verb := verb.Create
		author := entity.New(entity.NamespaceUser, c.UserID)
		pub.AssertCalled(t, "Publish", mock.Anything, ent, verb, author)
	})
}

func testIntegration_Shares(t *testing.T, ts *testSetup) {
	Convey("Should fail matching a share that doesn't exist", func() {
		_, err := getShare(ts, "share:does_not_exist")
		So(err, ShouldNotBeNil)
	})
	Convey("Should be able to create a share", func() {
		post, err := createPost(ts, testUserID, "hello world", nil)
		So(err, ShouldBeNil)
		So(post, ShouldNotBeNil)
		postEnt := entity.New(entity.NamespacePost, post.ID)

		share, err := createShare(ts, testUserID, postEnt)
		So(err, ShouldBeNil)
		So(share.ID, ShouldNotEqual, "")
		So(share.UserID, ShouldEqual, testUserID)
		So(share.TargetEntity, ShouldResemble, postEnt)

		Convey("then get it back", func() {
			shareBack, err := getShare(ts, share.ID)
			So(err, ShouldBeNil)
			So(shareBack.TargetEntity.Encode(), ShouldEqual, share.TargetEntity.Encode())
		})
		Convey("then get back multi", func() {
			sharesBack, err := getShares(ts, []string{share.ID})
			So(err, ShouldBeNil)
			So(len(sharesBack.Items), ShouldEqual, 1)
			So(sharesBack.Items[0].TargetEntity.Encode(), ShouldEqual, share.TargetEntity.Encode())
		})
		Convey("then get back the share by parent", func() {
			sharesBack, err := getSharesSummary(ts, postEnt.Encode())
			So(err, ShouldBeNil)
			So(sharesBack.Items[0].ParentEntity, ShouldResemble, postEnt)
			So(sharesBack.Items[0].Total, ShouldEqual, 1)
		})
		Convey("then get back the share for a userID", func() {
			sharesBack, err := getSharesByAuthor(ts, postEnt.Encode(), testUserID)
			So(err, ShouldBeNil)
			So(sharesBack.Items[0].ID, ShouldEqual, share.ID)
		})
		Convey("and share should be empty for other user IDs", func() {
			shares, err := getSharesByAuthor(ts, postEnt.Encode(), testUserIDAlt)
			So(err, ShouldBeNil)
			So(len(shares.Items), ShouldEqual, 0)
		})

		Convey("then remove it", func() {
			shareBack, err := deleteShare(ts, share.ID)
			So(err, ShouldBeNil)
			So(shareBack.TargetEntity.Encode(), ShouldEqual, share.TargetEntity.Encode())
			Convey("and it should stay gone", func() {
				_, err := getShare(ts, share.ID)
				So(err, ShouldNotBeNil)
			})
			Convey("and share by userID should be empty", func() {
				shares, err := getSharesByAuthor(ts, postEnt.Encode(), testUserID)
				So(err, ShouldBeNil)
				So(len(shares.Items), ShouldEqual, 0)
			})
			Convey("and share summary should be zero", func() {
				sharesBack, err := getSharesSummary(ts, postEnt.Encode())
				So(err, ShouldBeNil)
				So(sharesBack.Items[0].ParentEntity, ShouldResemble, postEnt)
				So(sharesBack.Items[0].Total, ShouldEqual, 0)
			})
		})
	})
}

func testIntegration_Reactions(t *testing.T, ts *testSetup) {

	Convey("Should be able to create, get, and delete a reaction", func() {
		parent := "test:parent:" + timestamp()
		testEmoteID := "Kappa"
		created, err := createReaction(ts, parent, testUserID, testEmoteID)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.ParentEntity.Encode(), ShouldEqual, parent)
		So(created.EmoteIDs, ShouldContain, testEmoteID)

		reaction, err := getReactions(ts, parent, testUserID)
		So(err, ShouldBeNil)
		So(reaction, ShouldNotBeNil)

		deleted, err := deleteReaction(ts, parent, testUserID, testEmoteID)
		So(err, ShouldBeNil)
		So(deleted, ShouldNotBeNil)
		So(len(deleted.EmoteIDs), ShouldEqual, 0)
		Convey("Should be able to get a reaction by user", func() {
			reactions, err := getReactionsByUser(ts, []string{created.ParentEntity.Encode()}, testUserID)
			So(err, ShouldBeNil)
			So(len(reactions.Reactions), ShouldEqual, 1)
		})
	})

	Convey("Should get a 404 if deleting a reaction that does not exist.", func() {
		parent := "test:parent:" + timestamp()
		testEmoteID := "Kappa"

		_, err := deleteReaction(ts, parent, testUserID, testEmoteID)
		So(err, ShouldNotBeNil)
		So(errorCode(err), ShouldEqual, http.StatusNotFound)
	})

	Convey("Should get a 404 if deleting an emote that isn't part of a reaction", func() {
		parent := "test:parent:" + timestamp()
		testEmoteID := "Kappa"
		_, err := createReaction(ts, parent, testUserID, testEmoteID)
		So(err, ShouldBeNil)

		anotherEmoteID := "101"
		_, err = deleteReaction(ts, parent, testUserID, anotherEmoteID)
		So(err, ShouldNotBeNil)
		So(errorCode(err), ShouldEqual, http.StatusNotFound)
	})

	Convey("Should return a 404 for get user reactions for parents where the user has no reactions", func() {
		parent := "test:parent:" + timestamp()
		testEmoteID := "101"

		reaction, err := getReactions(ts, parent, testUserID)
		So(errorCode(err), ShouldEqual, 404)
		So(reaction.EmoteIDs, ShouldBeEmpty)

		created, err := createReaction(ts, parent, testUserID, testEmoteID)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.ParentEntity.Encode(), ShouldEqual, parent)
		So(created.EmoteIDs, ShouldContain, testEmoteID)

		deleted, err := deleteReaction(ts, parent, testUserID, testEmoteID)
		So(err, ShouldBeNil)
		So(deleted, ShouldNotBeNil)
		So(len(deleted.EmoteIDs), ShouldEqual, 0)

		reaction2, err := getReactions(ts, parent, testUserID)
		So(errorCode(err), ShouldEqual, 404)
		So(reaction2.EmoteIDs, ShouldBeEmpty)
	})

	Convey("Setting the same reaction twice should be okay", func() {
		parent := "test:parent:" + timestamp()
		testEmoteID := "Kappa"
		c1, err := createReaction(ts, parent, testUserID, testEmoteID)
		So(err, ShouldBeNil)
		So(c1, ShouldNotBeNil)
		So(c1.EmoteIDs, ShouldContain, testEmoteID)

		c2, err := createReaction(ts, parent, testUserID, testEmoteID)
		So(err, ShouldBeNil)
		So(c2, ShouldNotBeNil)
		So(c2.EmoteIDs, ShouldContain, testEmoteID)
	})

	Convey("Should be able to react with multiple emotes", func() {
		parent := "test:parent:" + timestamp()
		a, err := createReaction(ts, parent, testUserID, "kappa")
		So(err, ShouldBeNil)
		So(a, ShouldNotBeNil)
		So(a.EmoteIDs, ShouldContain, "kappa")

		b, err := createReaction(ts, parent, testUserID, "pogchamp")
		So(err, ShouldBeNil)
		So(b, ShouldNotBeNil)
		So(b.EmoteIDs, ShouldContain, "kappa")
		So(b.EmoteIDs, ShouldContain, "pogchamp")
	})

	Convey("Should return a StatusBadRequest if given a malformed parent entity", func() {
		_, err := createReaction(ts, "malformedParent", testUserID, "Kappa")
		So(err, ShouldNotBeNil)
		So(errorCode(err), ShouldEqual, http.StatusBadRequest)
	})
}

func testIntegration_ReactionsSummary(t *testing.T, ts *testSetup) {
	Convey("ReactionSummary of unknown parent entity should return empty response", func() {
		parent := "test:parent:" + timestamp()

		summary, err := getReactionsSummary(ts, parent)
		So(err, ShouldBeNil)
		So(summary, ShouldNotBeNil)
		So(summary.EmoteSummaries, ShouldNotBeNil)
		So(summary.DeprecatedEmotes, ShouldNotBeNil)
		So(len(summary.EmoteSummaries), ShouldEqual, 0)
	})

	Convey("Should update the reaction summary when reactions are created and deleted", func() {
		parent := "test:parent:" + timestamp()

		_, err := createReaction(ts, parent, testUserID, "kappa")
		So(err, ShouldBeNil)
		summary, err := getReactionsSummary(ts, parent)
		So(err, ShouldBeNil)
		So(summary, ShouldNotBeNil)
		So(summary.EmoteSummaries, ShouldNotBeNil)
		So(summary.EmoteSummaries["kappa"].Count, ShouldEqual, 1)

		_, err = deleteReaction(ts, parent, testUserID, "kappa")
		So(err, ShouldBeNil)
		summary, err = getReactionsSummary(ts, parent)
		So(err, ShouldBeNil)
		So(summary, ShouldNotBeNil)
		So(len(summary.EmoteSummaries), ShouldEqual, 0)
	})

	Convey("Should NOT increment the emote count when reacting with the same emote", func() {
		parent := "test:parent:" + timestamp()
		_, err := createReaction(ts, parent, testUserID, "kappa")
		So(err, ShouldBeNil)
		s1, err := getReactionsSummary(ts, parent)
		So(err, ShouldBeNil)
		So(s1, ShouldNotBeNil)
		So(s1.EmoteSummaries["kappa"].Count, ShouldEqual, 1)

		_, err = createReaction(ts, parent, testUserID, "kappa")
		So(err, ShouldBeNil)
		s2, err := getReactionsSummary(ts, parent)
		So(err, ShouldBeNil)
		So(s2.EmoteSummaries["kappa"].Count, ShouldEqual, 1)
	})

	// Note: dynamodb does not allow AttributeValues and Attribute Names to equal empty string, meaning this
	// edge case is currently handled at the db level.
	Convey("Should NOT allow the user to save a emote ID of empty string", func() {
		parent := "test:parent:" + timestamp()
		_, err := createReaction(ts, parent, testUserID, "")
		So(err, ShouldNotBeNil)

		summary, err := getReactionsSummary(ts, parent)
		So(summary, ShouldNotBeNil)
		So(err, ShouldBeNil)
		So(len(summary.EmoteSummaries), ShouldEqual, 0)
	})

	Convey("Should be able to get reactions for multiple parent entities", func() {
		parent1 := "test:parent:" + timestamp()
		created1, err := createReaction(ts, parent1, testUserID, "120")
		So(err, ShouldBeNil)
		So(created1, ShouldNotBeNil)

		parent2 := "test:parent:" + timestamp()
		created2, err := createReaction(ts, parent2, testUserID, "200")
		So(err, ShouldBeNil)
		So(created2, ShouldNotBeNil)

		reactionsSummaries, err := getReactionsSummaries(ts, []string{parent1, parent2})
		So(err, ShouldBeNil)
		So(reactionsSummaries, ShouldNotBeNil)
		So(len(reactionsSummaries.Items), ShouldEqual, 2)
		ShouldContainReaction(reactionsSummaries.Items, created1)
		ShouldContainReaction(reactionsSummaries.Items, created2)

		Convey("and fill in user", func() {
			reactionsSummaries, err := getReactionsSummariesForUsers(ts, []string{parent1, parent2}, []string{testUserID})
			So(err, ShouldBeNil)
			So(reactionsSummaries, ShouldNotBeNil)
			So(len(reactionsSummaries.Items), ShouldEqual, 2)
			ShouldContainReaction(reactionsSummaries.Items, created1)
			ShouldContainReaction(reactionsSummaries.Items, created2)
			shouldHaveUserReaction(reactionsSummaries.Items, parent1, "120", testUserID)
			shouldHaveUserReaction(reactionsSummaries.Items, parent2, "200", testUserID)
		})
	})

	Convey("Should be not return reactions with a count of zero for batch get reactions", func() {
		parent := "test:parent:" + timestamp()
		emoteID := "101"
		created, err := createReaction(ts, parent, testUserID, emoteID)
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(created.EmoteIDs[0], ShouldEqual, emoteID)

		summary, err := getReactionsSummaries(ts, []string{parent})
		So(err, ShouldBeNil)
		ShouldContainReaction(summary.Items, created)
		So(summary.Items[0].EmoteSummaries[emoteID].Count, ShouldEqual, 1)

		created, err = deleteReaction(ts, parent, testUserID, "101")
		So(err, ShouldBeNil)
		So(created, ShouldNotBeNil)
		So(len(created.EmoteIDs), ShouldEqual, 0)

		summary1, err := getReactionsSummaries(ts, []string{parent})
		So(err, ShouldBeNil)
		So(len(summary1.Items), ShouldEqual, 1)
		So(summary1.Items[0].EmoteSummaries, ShouldNotContainKey, emoteID)
	})
}

func shouldHaveUserReaction(actual []*api.ReactionsSummary, parentEntity string, emoteID string, userID string) {
	// find the expected reaction parent entity inside the list of reactions summaries
	var reactionSummary *api.ReactionsSummary
	for _, summary := range actual {
		if parentEntity == summary.ParentEntity.Encode() {
			reactionSummary = summary
			break
		}
	}

	So(reactionSummary.ParentEntity, ShouldNotBeEmpty)
	So(reactionSummary.EmoteSummaries, ShouldContainKey, emoteID)
	So(reactionSummary.EmoteSummaries[emoteID].UserIDs, ShouldContain, userID)
}

// ShouldContainReaction receives exactly two parameters. The first is a slice containing reactions summaries and the
// second is a user reaction. The user reaction's parent entity should exist within reactions summaries as should all
// emote ids.
func ShouldContainReaction(actual []*api.ReactionsSummary, expected *api.Reactions) {
	// find the expected reaction parent entity inside the list of reactions summaries
	var reactionSummary *api.ReactionsSummary
	for _, summary := range actual {
		if expected.ParentEntity == summary.ParentEntity {
			reactionSummary = summary
			break
		}
	}

	So(reactionSummary.ParentEntity, ShouldNotBeEmpty)

	for _, emoteID := range expected.EmoteIDs {
		So(reactionSummary.DeprecatedEmotes, ShouldContainKey, emoteID)
		So(reactionSummary.DeprecatedEmotes[emoteID], ShouldBeGreaterThan, 0)
	}
	for _, emoteID := range expected.EmoteIDs {
		So(reactionSummary.EmoteSummaries, ShouldContainKey, emoteID)
		So(reactionSummary.EmoteSummaries[emoteID].Count, ShouldBeGreaterThan, 0)
	}
}

type codedError interface {
	HTTPCode() int
}

func errorCode(err error) int {
	if coded, ok := errors.Cause(err).(codedError); ok {
		return coded.HTTPCode()
	}
	return 0
}

type testSetup struct {
	ctx         context.Context
	injectables injectables
	cancelFunc  func()
	client      *http.Client
	host        string
}

func (t *testSetup) Setup() error {
	t.ctx, t.cancelFunc = context.WithTimeout(context.Background(), time.Second*3)
	t.client = &http.Client{}
	return nil
}

func request(ts *testSetup, method, url string, body interface{}, into interface{}) error {
	return clients.DoHTTP(ts.ctx, ts.client, method, ts.host+url, nil, body, into, nil)
}

func addMapValues(m *distconf.InMemory, vals map[string][]byte) error {
	for k, v := range vals {
		if err := m.Write(k, v); err != nil {
			return err
		}
	}
	return nil
}

type panicPanic struct{}

func (p panicPanic) OnPanic(pnc interface{}) {
	panic(pnc)
}

type testServerInfo struct {
	listenAddr string
}

func startServer(t *testing.T, i injectables) (*testServerInfo, func(time.Duration)) {
	localConf := &distconf.InMemory{}
	err := addMapValues(localConf, map[string][]byte{
		"duplo.listen_addr":            []byte(":0"),
		"rollbar.access_token":         []byte(""),
		"statsd.hostport":              []byte(""),
		"debug.addr":                   []byte(":0"),
		"logging.to_stdout":            []byte("false"),
		"logging.to_stderr":            []byte("true"),
		"dynamo.consistent_reads":      []byte("true"),
		"duplo.comment_activity_users": []byte(testUserID),
	})
	if err != nil {
		t.Error(err)
		return nil, nil
	}

	started := make(chan string)
	finished := make(chan struct{})
	signalToClose := make(chan os.Signal)
	exitCalled := make(chan struct{})
	elevateKey := "hi"
	thisInstance := service{
		injectables: i,
		osExit: func(i int) {
			if i != 0 {
				t.Error("Invalid osExit status code", i)
			}
			close(exitCalled)
		},
		serviceCommon: service_common.ServiceCommon{
			ConfigCommon: service_common.ConfigCommon{
				Team:          teamName,
				Service:       serviceName,
				CustomReaders: []distconf.Reader{localConf},
				BaseDirectory: "../../",
				ElevateLogKey: elevateKey,
				OsGetenv:      os.Getenv,
				OsHostname:    os.Hostname,
			},
			CodeVersion: CodeVersion,
			Log: &log.ElevatedLog{
				ElevateKey: elevateKey,
				NormalLog: log.ContextLogger{
					Logger: t,
				},
				DebugLog: log.ContextLogger{
					Logger: log.Discard,
				},
				LogToDebug: func(_ ...interface{}) bool {
					return false
				},
			},
			PanicLogger:    panicPanic{},
			SfxSetupConfig: sfxStastdConfig(),
		},
		skipGlobalConfigParse: true,
		sigChan:               signalToClose,
		onListen: func(listeningAddr net.Addr) {
			started <- fmt.Sprintf("http://localhost:%d", listeningAddr.(*net.TCPAddr).Port)
		},
	}
	thisInstance.serviceCommon.Log.NormalLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	thisInstance.serviceCommon.Log.DebugLog.Dims = &thisInstance.serviceCommon.CtxDimensions

	healthCheckCtx, cancelFunc := context.WithCancel(context.Background())
	go func() {
		defer cancelFunc()
		thisInstance.main()
		close(finished)
	}()

	var addressForIntegrationTests string
	select {
	case <-exitCalled:
		return nil, nil
	case addressForIntegrationTests = <-started:
	case <-time.After(time.Second * 35):
		t.Error("Took to long to start service")
		return nil, nil
	}

	if err := tryHealthCheck(t, addressForIntegrationTests, healthCheckCtx); err != nil {
		t.Error(err)
		return nil, nil
	}
	onFinish := func(timeToWait time.Duration) {
		signalToClose <- syscall.SIGTERM
		select {
		case <-finished:
			return
		case <-time.After(timeToWait):
			t.Error("Timed out waiting for server to end")
		}
	}
	return &testServerInfo{
		listenAddr: addressForIntegrationTests,
	}, onFinish
}

func tryHealthCheck(t *testing.T, addressForIntegrationTests string, healthCheckCtx context.Context) error {
	t.Log("Fetching from ", addressForIntegrationTests)
	c := http.Client{}
	req, err := http.NewRequest("GET", addressForIntegrationTests+"/debug/health", nil)
	if err != nil {
		return err
	}
	req = req.WithContext(healthCheckCtx)
	resp, err := c.Do(req)
	if err != nil {
		return err
	}
	if resp.StatusCode != http.StatusOK {
		t.Error("Invalid status code ", resp.StatusCode)
		buf := &bytes.Buffer{}
		_, err := io.Copy(buf, resp.Body)
		t.Error(buf.String(), err)
		return fmt.Errorf("Invalid status code %d", resp.StatusCode)
	}
	return nil
}

func TestMainFunc(t *testing.T) {
	ret := 0
	a := instance.osExit
	b := instance.serviceCommon.BaseDirectory
	c := instance.serviceCommon.Log
	defer func() {
		instance.osExit = a
		instance.serviceCommon.BaseDirectory = b
		instance.serviceCommon.Log = c
	}()
	instance.osExit = func(i int) { ret = i }
	instance.serviceCommon.BaseDirectory = "INTENTIONAL_MISSING_DIRECTORY"
	instance.serviceCommon.Log = nil
	main()
	if ret == 0 {
		t.Error("expected instance to fail")
	}
}

func newUserID() string {
	return "user" + timestamp()
}

// Bool returns a pointer to the bool value passed in.
func Bool(v bool) *bool {
	return &v
}
