locals {
  owner   = "jlindamo@twitch.tv"
  team    = "feeds"
  region  = "us-west-2"
  service = "duplo"

  account_ids = {
    integration = "724951484461"
    staging     = "724951484461"
    production  = "914569885343"
  }

  profiles = {
    integration = "twitch-feed-dev"
    staging     = "twitch-feed-dev"
    production  = "twitch-feed-aws"
  }

  post_read_capacities = {
    staging     = 200
    integration = 200
    production  = 3000
  }

  post_read_capacity = "${lookup(local.post_read_capacities, terraform.workspace)}"

  post_write_capacities = {
    staging     = 100
    integration = 100
    production  = 150
  }

  post_write_capacity = "${lookup(local.post_write_capacities, terraform.workspace)}"

  post_audrey_index_read_capacities = {
    staging     = 100
    integration = 100
    production  = 1000
  }

  post_audrey_index_read_capacity = "${lookup(local.post_audrey_index_read_capacities, terraform.workspace)}"

  post_audrey_index_write_capacities = {
    staging     = 100
    integration = 100
    production  = 200
  }

  post_audrey_index_write_capacity = "${lookup(local.post_audrey_index_write_capacities, terraform.workspace)}"

  post_userid_index_read_capacities = {
    staging     = 100
    integration = 100
    production  = 1000
  }

  post_userid_index_read_capacity = "${lookup(local.post_userid_index_read_capacities, terraform.workspace)}"

  post_userid_index_write_capacities = {
    staging     = 100
    integration = 100
    production  = 200
  }

  post_userid_index_write_capacity = "${lookup(local.post_userid_index_write_capacities, terraform.workspace)}"

  comment_read_capacities = {
    staging     = 400
    integration = 400
    production  = 500
  }

  comment_read_capacity = "${lookup(local.comment_read_capacities, terraform.workspace)}"

  comment_write_capacities = {
    staging     = 100
    integration = 100
    production  = 200
  }

  comment_write_capacity = "${lookup(local.comment_write_capacities, terraform.workspace)}"

  comment_parent_index_read_capacities = {
    integration = 100
    staging     = 100
    production  = 3000
  }

  comment_parent_index_read_capacity = "${lookup(local.comment_parent_index_read_capacities, terraform.workspace)}"

  comment_parent_index_write_capacities = {
    integration = 100
    staging     = 100
    production  = 300
  }

  comment_parent_index_write_capacity = "${lookup(local.comment_parent_index_write_capacities, terraform.workspace)}"

  comment_audrey_index_read_capacities = {
    integration = 100
    staging     = 100
    production  = 2000
  }

  comment_audrey_index_read_capacity = "${lookup(local.comment_audrey_index_read_capacities, terraform.workspace)}"

  comment_audrey_index_write_capacities = {
    integration = 100
    staging     = 100
    production  = 300
  }

  comment_audrey_index_write_capacity = "${lookup(local.comment_audrey_index_write_capacities, terraform.workspace)}"

  comment_summary_read_capacities = {
    integration = 200
    staging     = 200
    production  = 3000
  }

  comment_summary_read_capacity = "${lookup(local.comment_summary_read_capacities, terraform.workspace)}"

  comment_summary_write_capacities = {
    integration = 100
    staging     = 100
    production  = 250
  }

  comment_summary_write_capacity = "${lookup(local.comment_summary_write_capacities, terraform.workspace)}"

  reaction_read_capacities = {
    integration = 400
    staging     = 400
    production  = 12500
  }

  reaction_read_capacity = "${lookup(local.reaction_read_capacities, terraform.workspace)}"

  reaction_write_capacities = {
    integration = 200
    staging     = 200
    production  = 500
  }

  reaction_write_capacity = "${lookup(local.reaction_write_capacities, terraform.workspace)}"

  reaction_summary_read_capacities = {
    integration = 200
    staging     = 200
    production  = 12500
  }

  reaction_summary_read_capacity = "${lookup(local.reaction_summary_read_capacities, terraform.workspace)}"

  reaction_summary_write_capacities = {
    integration = 200
    staging     = 200
    production  = 1000
  }

  reaction_summary_write_capacity = "${lookup(local.reaction_summary_write_capacities, terraform.workspace)}"

  share_read_capacities = {
    integration = 200
    staging     = 200
    production  = 3000
  }

  share_read_capacity = "${lookup(local.share_read_capacities, terraform.workspace)}"

  share_write_capacities = {
    integration = 200
    staging     = 200
    production  = 500
  }

  share_write_capacity = "${lookup(local.share_write_capacities, terraform.workspace)}"

  share_entity_write_capacities = {
    integration = 200
    staging     = 200
    production  = 500
  }

  share_entity_write_capacity = "${lookup(local.share_entity_write_capacities, terraform.workspace)}"

  share_entity_read_capacities = {
    integration = 200
    staging     = 200
    production  = 3000
  }

  share_entity_read_capacity = "${lookup(local.share_entity_read_capacities, terraform.workspace)}"

  share_summary_read_capacities = {
    integration = 200
    staging     = 200
    production  = 3000
  }

  share_summary_read_capacity = "${lookup(local.share_summary_read_capacities, terraform.workspace)}"

  share_summary_write_capacities = {
    integration = 200
    staging     = 200
    production  = 1000
  }

  share_summary_write_capacity = "${lookup(local.share_summary_write_capacities, terraform.workspace)}"

  alarm_threshold_percents = {
    integration = 70
    staging     = 70
    production  = 70
  }

  alarm_threshold_percent = "${lookup(local.alarm_threshold_percents, terraform.workspace)}"

  alarm_periods = {
    integration = 60
    staging     = 60
    production  = 60
  }

  alarm_period = "${lookup(local.alarm_periods, terraform.workspace)}"

  profile = "${lookup(local.profiles, terraform.workspace)}"
}

provider "aws" {
  profile             = "${local.profile}"
  allowed_account_ids = ["${lookup(local.account_ids, terraform.workspace)}"]
  region              = "${local.region}"
}

terraform {
  backend "s3" {
    bucket  = "twitch-feed-aws"
    key     = "feeds/duplo-dynamodb-us-west-2.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "region_core" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/core-region-${local.region}.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "account_global" {
  backend   = "s3"
  workspace = "${local.profile}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/core-global.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "duplo" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/duplo-us-west-2.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

# Posts
#	Hash: id (ID, string)
resource "aws_dynamodb_table" "post_dynamodb" {
  name           = "${local.service}_${terraform.workspace}_post"
  read_capacity  = "${local.post_read_capacity}"
  write_capacity = "${local.post_write_capacity}"
  hash_key       = "id"

  attribute {
    name = "id"
    type = "S"
  }

  attribute {
    name = "created_at"
    type = "N"
  }

  attribute {
    name = "audrey_id"
    type = "S"
  }

  attribute {
    name = "user_id"
    type = "S"
  }

  global_secondary_index {
    name            = "audrey_id-index"
    projection_type = "KEYS_ONLY"
    hash_key        = "audrey_id"
    read_capacity   = "${local.post_audrey_index_read_capacity}"
    write_capacity  = "${local.post_audrey_index_write_capacity}"
  }

  global_secondary_index {
    name               = "user_id-created_at-index"
    projection_type    = "INCLUDE"
    non_key_attributes = ["deleted_at"]
    hash_key           = "user_id"
    range_key          = "created_at"
    read_capacity      = "${local.post_userid_index_read_capacity}"
    write_capacity     = "${local.post_userid_index_write_capacity}"
  }

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Environment = "${terraform.workspace}"
    Service     = "${local.service}"
    Team        = "${local.team}"
  }
}

module post_dynamodb_cloudwatch {
  source         = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//dynamodb/table_cloudwatch?ref=v0.8.20"
  table_name     = "${aws_dynamodb_table.post_dynamodb.name}"
  alerts_sns_arn = "${data.terraform_remote_state.region_core.pagerduty_alarm_arn}"
  read_capacity  = "${aws_dynamodb_table.post_dynamodb.read_capacity}"
  write_capacity = "${aws_dynamodb_table.post_dynamodb.write_capacity}"
}

resource "aws_dynamodb_table" "comment_dynamodb" {
  name           = "${local.service}_${terraform.workspace}_comment"
  read_capacity  = "${local.comment_read_capacity}"
  write_capacity = "${local.comment_write_capacity}"
  hash_key       = "id"

  attribute {
    name = "id"
    type = "S"
  }

  attribute {
    name = "parent_entity"
    type = "S"
  }

  attribute {
    name = "created_at"
    type = "N"
  }

  attribute {
    name = "audrey_id"
    type = "S"
  }

  global_secondary_index {
    name               = "parent_entity-created_at-index-2"
    projection_type    = "INCLUDE"
    non_key_attributes = ["user_id", "deleted_at"]
    hash_key           = "parent_entity"
    range_key          = "created_at"
    read_capacity      = "${local.comment_parent_index_read_capacity}"
    write_capacity     = "${local.comment_parent_index_write_capacity}"
  }

  global_secondary_index {
    name            = "audrey_id-index"
    projection_type = "KEYS_ONLY"
    hash_key        = "audrey_id"
    read_capacity   = "${local.comment_audrey_index_read_capacity}"
    write_capacity  = "${local.comment_audrey_index_write_capacity}"
  }

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Environment = "${terraform.workspace}"
    Service     = "${local.service}"
    Team        = "${local.team}"
  }
}

module comment_dynamodb_cloudwatch {
  source         = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//dynamodb/table_cloudwatch?ref=v0.8.20"
  table_name     = "${aws_dynamodb_table.comment_dynamodb.name}"
  alerts_sns_arn = "${data.terraform_remote_state.region_core.pagerduty_alarm_arn}"
  read_capacity  = "${aws_dynamodb_table.comment_dynamodb.read_capacity}"
  write_capacity = "${aws_dynamodb_table.comment_dynamodb.write_capacity}"
}

resource "aws_dynamodb_table" "comment_summary_dynamodb" {
  name           = "${local.service}_${terraform.workspace}_comment_summary"
  read_capacity  = "${local.comment_summary_read_capacity}"
  write_capacity = "${local.comment_summary_write_capacity}"
  hash_key       = "parent_entity"

  attribute {
    name = "parent_entity"
    type = "S"
  }

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Environment = "${terraform.workspace}"
    Service     = "${local.service}"
    Team        = "${local.team}"
  }
}

module comment_summary_dynamodb_cloudwatch {
  source         = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//dynamodb/table_cloudwatch?ref=v0.8.20"
  table_name     = "${aws_dynamodb_table.comment_summary_dynamodb.name}"
  alerts_sns_arn = "${data.terraform_remote_state.region_core.pagerduty_alarm_arn}"
  read_capacity  = "${aws_dynamodb_table.comment_summary_dynamodb.read_capacity}"
  write_capacity = "${aws_dynamodb_table.comment_summary_dynamodb.write_capacity}"
}

resource "aws_dynamodb_table" "reaction_dynamodb" {
  name           = "${local.service}_${terraform.workspace}_reaction"
  read_capacity  = "${local.reaction_read_capacity}"
  write_capacity = "${local.reaction_write_capacity}"
  hash_key       = "parent_entity"
  range_key      = "user_id"

  attribute {
    name = "parent_entity"
    type = "S"
  }

  attribute {
    name = "user_id"
    type = "S"
  }

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Environment = "${terraform.workspace}"
    Service     = "${local.service}"
    Team        = "${local.team}"
  }
}

module reaction_dynamodb_cloudwatch {
  source         = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//dynamodb/table_cloudwatch?ref=v0.8.20"
  table_name     = "${aws_dynamodb_table.reaction_dynamodb.name}"
  alerts_sns_arn = "${data.terraform_remote_state.region_core.pagerduty_alarm_arn}"
  read_capacity  = "${aws_dynamodb_table.reaction_dynamodb.read_capacity}"
  write_capacity = "${aws_dynamodb_table.reaction_dynamodb.write_capacity}"
}

resource "aws_dynamodb_table" "reaction_summary_dynamodb" {
  name           = "${local.service}_${terraform.workspace}_reaction_summary"
  read_capacity  = "${local.reaction_summary_read_capacity}"
  write_capacity = "${local.reaction_summary_write_capacity}"
  hash_key       = "parent_entity"

  attribute {
    name = "parent_entity"
    type = "S"
  }

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Environment = "${terraform.workspace}"
    Service     = "${local.service}"
    Team        = "${local.team}"
  }
}

module reaction_summary_dynamodb_cloudwatch {
  source         = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//dynamodb/table_cloudwatch?ref=v0.8.20"
  table_name     = "${aws_dynamodb_table.reaction_summary_dynamodb.name}"
  alerts_sns_arn = "${data.terraform_remote_state.region_core.pagerduty_alarm_arn}"
  read_capacity  = "${aws_dynamodb_table.reaction_summary_dynamodb.read_capacity}"
  write_capacity = "${aws_dynamodb_table.reaction_summary_dynamodb.write_capacity}"
}

resource "aws_dynamodb_table" "share_dynamodb" {
  name           = "${local.service}_${terraform.workspace}_share"
  read_capacity  = "${local.share_read_capacity}"
  write_capacity = "${local.share_write_capacity}"
  hash_key       = "id"

  attribute {
    name = "id"
    type = "S"
  }

  attribute {
    name = "target_entity"
    type = "S"
  }

  attribute {
    name = "user_id"
    type = "S"
  }

  global_secondary_index {
    name            = "target_entity-index"
    projection_type = "ALL"
    hash_key        = "target_entity"
    range_key       = "user_id"
    read_capacity   = "${local.share_entity_read_capacity}"
    write_capacity  = "${local.share_entity_write_capacity}"
  }

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Environment = "${terraform.workspace}"
    Service     = "${local.service}"
    Team        = "${local.team}"
  }
}

module share_dynamodb_cloudwatch {
  source         = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//dynamodb/table_cloudwatch?ref=v0.8.20"
  table_name     = "${aws_dynamodb_table.share_dynamodb.name}"
  alerts_sns_arn = "${data.terraform_remote_state.region_core.pagerduty_alarm_arn}"
  read_capacity  = "${aws_dynamodb_table.share_dynamodb.read_capacity}"
  write_capacity = "${aws_dynamodb_table.share_dynamodb.write_capacity}"
}

resource "aws_dynamodb_table" "share_summary" {
  name           = "${local.service}_${terraform.workspace}_share_summary"
  read_capacity  = "${local.share_summary_read_capacity}"
  write_capacity = "${local.share_summary_write_capacity}"
  hash_key       = "parent_entity"

  attribute {
    name = "parent_entity"
    type = "S"
  }

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Environment = "${terraform.workspace}"
    Service     = "${local.service}"
    Team        = "${local.team}"
  }
}

module share_summary_cloudwatch {
  source         = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//dynamodb/table_cloudwatch?ref=v0.8.20"
  table_name     = "${aws_dynamodb_table.share_summary.name}"
  alerts_sns_arn = "${data.terraform_remote_state.region_core.pagerduty_alarm_arn}"
  read_capacity  = "${aws_dynamodb_table.share_summary.read_capacity}"
  write_capacity = "${aws_dynamodb_table.share_summary.write_capacity}"
}
