locals {
  owner   = "jlindamo@twitch.tv"
  team    = "feeds"
  region  = "us-west-2"
  service = "duplo"

  account_ids = {
    integration = "724951484461"
    staging     = "724951484461"
    production  = "914569885343"
  }

  profiles = {
    integration = "twitch-feed-dev"
    staging     = "twitch-feed-dev"
    production  = "twitch-feed-aws"
  }

  min_counts = {
    integration = "1"
    staging     = "2"
    production  = "4"
  }

  desired_counts = {
    integration = "2"
    staging     = "3"
    production  = "7"
  }

  max_counts = {
    integration = "3"
    staging     = "4"
    production  = "12"
  }

  auto_promotes = {
    integration = "true"
    staging     = "true"
    production  = "false"
  }

  promote_froms = {
    integration = "latest"
    staging     = "integration"
    production  = "canary"
  }

  promote_from  = "${lookup(local.promote_froms, terraform.workspace)}"
  auto_promote  = "${lookup(local.auto_promotes, terraform.workspace)}"
  min_count     = "${lookup(local.min_counts, terraform.workspace)}"
  max_count     = "${lookup(local.max_counts, terraform.workspace)}"
  desired_count = "${lookup(local.desired_counts, terraform.workspace)}"
  profile       = "${lookup(local.profiles, terraform.workspace)}"
}

provider "aws" {
  profile             = "${local.profile}"
  allowed_account_ids = ["${lookup(local.account_ids, terraform.workspace)}"]
  region              = "${local.region}"
}

terraform {
  backend "s3" {
    bucket  = "twitch-feed-aws"
    key     = "feeds/duplo-us-west-2.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

provider "jenkins" {
  url                = "https://jenkins.internal.justin.tv"
  shared_credentials = "${file(pathexpand("~/.jenkins_auth"))}"
}

provider "consul" {
  address    = "consul.internal.justin.tv"
  datacenter = "us-west2"
}

data "terraform_remote_state" "account" {
  backend = "s3"

  config {
    bucket  = "${local.profile}"
    key     = "tfstate/feed/terraform/accounts/${local.profile}"
    region  = "us-west-2"
    profile = "${local.profile}"
  }
}

data "terraform_remote_state" "account_core" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/core-account.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "region_core" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/core-region-${local.region}.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

module "duplo" {
  source           = "../../modules/cluster"
  environment      = "${terraform.workspace}"
  target_group_arn = "${module.alb.target_group_arn}"
  cluster_name     = "${data.terraform_remote_state.region_core.common_cluster_name}"

  min_size                  = "${local.min_count}"
  desired_count             = "${local.desired_count}"
  max_size                  = "${local.max_count}"
  service_iam_role          = "${data.terraform_remote_state.account_core.service_iam_role}"
  task_autoscale_iam_role   = "${data.terraform_remote_state.account_core.task_autoscale_iam_role}"
  profile                   = "${local.profile}"
  promote_from              = "${local.promote_from}"
  auto_promote              = "${local.auto_promote}"
  deploy_aws_role           = "${data.terraform_remote_state.account_core.build_role}"
  container_loggroup_region = "${data.terraform_remote_state.region_core.container_loggroup_region}"
  container_loggroup        = "${data.terraform_remote_state.region_core.container_loggroup}"
  aws_creds                 = "${data.terraform_remote_state.account_core.jenkins_build_secret}"
  testing_assume_role       = "${data.terraform_remote_state.account_core.build_user_arn}"
}

module "permissions" {
  source      = "../../modules/permissions"
  environment = "${terraform.workspace}"
  iam_id      = "${module.duplo.iam_id}"
}

module "alb" {
  source            = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//alb?ref=v0.8.20"
  owner             = "${local.owner}"
  environment       = "${terraform.workspace}"
  access_log_bucket = "${local.profile}-logs"
  team              = "${local.team}"
  service           = "${local.service}"
  subnets           = "${data.terraform_remote_state.account.private_subnets}"
  security_groups   = "${data.terraform_remote_state.account.twitch_subnets_sg}"
  vpc_id            = "${data.terraform_remote_state.account.vpc_id}"
}

module alb_cloudwatch {
  source           = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//alb/cloudwatch?ref=v0.8.20"
  alb_arn          = "${module.alb.alb_arn}"
  alarm_sns_arn    = "${data.terraform_remote_state.region_core.pagerduty_alarm_arn}"
  target_group_arn = "${module.alb.target_group_arn}"
}

output target_group_arn {
  value = "${module.alb.target_group_arn}"
}

output task_role_arn {
  value = "${module.duplo.task_role_arn}"
}
