package rollbar

import (
	"errors"
	"net/http"
	"os"
	"testing"

	"code.justin.tv/common/golibs/errorlogger/rollbar/mocks"
	"github.com/heroku/rollbar"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type RollbarSuite struct {
	suite.Suite
	rollbar *mocks.Client
	logger  ErrorLogger
}

func (s *RollbarSuite) SetupTest() {
	rollbar := &mocks.Client{}
	rollbar.On("Message", mock.Anything, mock.Anything).Return()
	rollbar.On("ErrorWithStackSkip", mock.Anything, mock.Anything, mock.Anything).Return()
	rollbar.On("RequestErrorWithStackSkip", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return()
	rollbar.On("SetCustom", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return()
	s.rollbar = rollbar
	logger := ErrorLogger{rollbar}
	s.logger = logger
}

func (s *RollbarSuite) TestNew() {
	logger := NewRollbarLogger("token", "development")
	s.NotNil(logger)
	l, ok := logger.(*ErrorLogger)
	hostname, err := os.Hostname()
	if err != nil {
		hostname = "unknown"
	}

	s.True(ok)
	s.Equal("token", l.rollbar.GetToken())
	s.Equal("development", l.rollbar.GetEnvironment())
	s.Equal(hostname, l.rollbar.GetServerHost())
}

func (s *RollbarSuite) TestInfo() {
	s.logger.Info("hello")
	s.rollbar.AssertCalled(s.T(), "Message", rollbar.INFO, "hello")
}

func (s *RollbarSuite) TestError() {
	err := errors.New("uhoh")
	s.logger.Error(err)
	s.rollbar.AssertCalled(s.T(), "ErrorWithStackSkip", rollbar.ERR, err, entireStack)
}

func (s *RollbarSuite) TestRequestError() {
	err := errors.New("uhoh")
	request := &http.Request{}
	s.logger.RequestError(request, err)
	s.rollbar.AssertCalled(s.T(), "RequestErrorWithStackSkip", rollbar.ERR, request, err, entireStack)
}

func (s *RollbarSuite) TestRequestPanic() {
	request := &http.Request{}
	expectedErr := errors.New("panic: do not panic")
	s.logger.RequestPanic(request, "do not panic")
	s.rollbar.AssertCalled(s.T(), "RequestErrorWithStackSkip", rollbar.ERR, request, expectedErr, truncatedStack)
}

func (s *RollbarSuite) TestNewErrorLogger() {
	logger := NewErrorLogger("token", "development")
	logger.SetCustom(map[string]interface{}{
		"region": "us-west-2",
	})

	s.NotNil(logger)
	hostname, err := os.Hostname()
	if err != nil {
		hostname = "unknown"
	}

	s.Equal("token", logger.rollbar.GetToken())
	s.Equal("development", logger.rollbar.GetEnvironment())
	s.Equal(hostname, logger.rollbar.GetServerHost())
	s.Equal("us-west-2", logger.rollbar.GetCustom()["region"])
}

func TestRollbarSuite(t *testing.T) {
	suite.Run(t, new(RollbarSuite))
}
