package fanout

import (
	"bytes"
	"encoding/json"
	"fmt"
	"strconv"

	"code.justin.tv/feeds/clients/feeddataflow"
	"code.justin.tv/feeds/ctxlog"
	"code.justin.tv/feeds/ctxlog/ctxlogaws"
	"code.justin.tv/feeds/ctxlog/ctxlogaws/ctxlogsqs"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"code.justin.tv/feeds/log"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sqs"
	"golang.org/x/net/context"
)

// Config configures Fanout client
type Config struct {
	QueueURL *distconf.Str
}

// Load configuration information
func (c *Config) Load(dconf *distconf.Distconf) error {
	c.QueueURL = dconf.Str("fanout.sqssource.queue_url", "")
	return nil
}

// Activity is the feed object that comes into fanout service
type Activity struct {
	Entity   entity.Entity          `json:"entity"`
	Verb     verb.Verb              `json:"verb"`
	Actor    entity.Entity          `json:"actor"`
	Metadata *feeddataflow.Metadata `json:"metadata,omitempty"`
}

func (i *Activity) String() string {
	return fmt.Sprintf("Actor:[%v] Verb:[%v] Entity:[%v]", i.Actor, i.Verb, i.Entity)
}

// Client can send messages to fanout via SQS
type Client struct {
	Sqs    *sqs.SQS
	Config *Config
	Ch     *ctxlog.Ctxlog
	Log    *log.ElevatedLog
}

// AddActivity adds a single feed activity to fanout
func (c *Client) AddActivity(ctx context.Context, activity *Activity) error {
	return c.itemsToQueue(ctx, []interface{}{activity}, c.Config.QueueURL.Get())
}

func (c *Client) itemsToQueue(ctx context.Context, items []interface{}, queueName string) error {
	msgAttributes := ctxlogsqs.ModifyRequest(ctx, nil, c.Ch)
	if len(items) == 1 {
		item := items[0]
		msgBuffer := &bytes.Buffer{}
		if err := json.NewEncoder(msgBuffer).Encode(item); err != nil {
			return err
		}
		msgBody := msgBuffer.String()
		input := &sqs.SendMessageInput{
			QueueUrl:          &queueName,
			MessageBody:       &msgBody,
			MessageAttributes: msgAttributes,
		}
		req, _ := c.Sqs.SendMessageRequest(input)
		req.HTTPRequest = req.HTTPRequest.WithContext(ctx)
		return ctxlogaws.DoAWSSend(req, c.Log)
	}

	entries := make([]*sqs.SendMessageBatchRequestEntry, 0, len(items))
	for i := 0; i < len(items); i++ {
		msgBuffer := &bytes.Buffer{}
		if err := json.NewEncoder(msgBuffer).Encode(items[i]); err != nil {
			return err
		}
		msgBody := msgBuffer.String()
		entries = append(entries, &sqs.SendMessageBatchRequestEntry{
			Id:                aws.String(strconv.Itoa(i)),
			MessageBody:       &msgBody,
			MessageAttributes: msgAttributes,
		})
	}
	input := &sqs.SendMessageBatchInput{
		QueueUrl: aws.String(c.Config.QueueURL.Get()),
		Entries:  entries,
	}
	req, _ := c.Sqs.SendMessageBatchRequest(input)
	req.HTTPRequest = req.HTTPRequest.WithContext(ctx)
	return ctxlogaws.DoAWSSend(req, c.Log)
}

// AddActivities adds multiple feed activities to fanout
func (c *Client) AddActivities(ctx context.Context, activity []*Activity) error {
	items := make([]interface{}, 0, len(activity))
	for _, a := range activity {
		items = append(items, a)
	}
	return c.itemsToQueue(ctx, items, c.Config.QueueURL.Get())
}

// HasValue is anything, including a context, that can extract values
type HasValue interface {
	Value(key interface{}) interface{}
}

// ActivityWithContext is the combined queue type for Async client's queue
type ActivityWithContext struct {
	Activity *Activity
	ValueCtx HasValue
}

// AsyncClient adds activity to fanout async via a channel.  Assumes Setup() then Start() are called.
type AsyncClient struct {
	Client        *Client
	OnErr         func(err error, activity *Activity)
	ActivityQueue chan *ActivityWithContext
	closeCalled   chan struct{}
}

// Setup the async client for use
func (c *AsyncClient) Setup() error {
	c.closeCalled = make(chan struct{})
	return nil
}

// Start the async client.  Runs till Close()
func (c *AsyncClient) Start() error {
	ctx := context.Background()
	for {
		select {
		case <-c.closeCalled:
			return nil
		case activity := <-c.ActivityQueue:
			c.addActivity(&ctxWithValues{Context: ctx, valueSack: activity.ValueCtx}, activity.Activity)
		}
	}
}

// ctxWithValues is a context that can load values from an old context, but doesn't use its deadline.
type ctxWithValues struct {
	context.Context
	valueSack HasValue
}

func (c *ctxWithValues) Value(key interface{}) interface{} {
	v := c.Context.Value(key)
	if v != nil {
		return v
	}
	return c.valueSack.Value(key)
}

func (c *AsyncClient) addActivity(ctx context.Context, activity *Activity) {
	err := c.Client.AddActivity(ctx, activity)
	if err != nil && c.OnErr != nil {
		c.OnErr(err, activity)
	}
}

// DrainThenClose will finish draining any activity then close the channel
func (c *AsyncClient) DrainThenClose() error {
	ctx := context.Background()
	for {
		select {
		case activity := <-c.ActivityQueue:
			c.addActivity(&ctxWithValues{Context: ctx, valueSack: activity.ValueCtx}, activity.Activity)
		default:
			return c.Close()
		}
	}
}

// Close ends the async client
func (c *AsyncClient) Close() error {
	close(c.closeCalled)
	return nil
}

// AddActivity returns immediately when activity is eventually added to the channel.  Blocks until Close() or ctx is
// closed.
func (c *AsyncClient) AddActivity(ctx context.Context, activity *Activity) error {
	select {
	case c.ActivityQueue <- &ActivityWithContext{Activity: activity, ValueCtx: ctx}:
		return nil
	case <-c.closeCalled:
		return errors.New("async client already closed")
	case <-ctx.Done():
		return ctx.Err()
	}
}
