GIT_COMMIT ?= $(shell git rev-parse --verify HEAD)
ENVIRONMENT ?= development
GOIMPORTS_REV ?= 5e2ae75
GOCOVMERGE_REV ?= b5bfa59
GOMETALINTER_REV ?= 0262fb2
TEAM ?= feeds
MAIN_DIR ?= ./cmd/$(SERVICE)
GOTEST ?= .
REPO = code.justin.tv/$(TEAM)/$(SERVICE)
LOCAL_DEPLOY_ENV ?= staging
LOCAL_DEPLOY_TAG ?= $(GIT_COMMIT)-$(shell whoami)

GOMETALINTER_ARGS ?= --vendor --dupl-threshold=150 --min-confidence=.3 --tests --deadline=90s --disable-all \
	-Egolint -Etest -Eineffassign -Etestify -Eunconvert -Estaticcheck -Egoconst -Egocyclo -Eerrcheck \
	-Egofmt -Evet -Edupl -Einterfacer -Estructcheck -Evetshadow -Egosimple -Egoimports -Evarcheck -Emisspell \
	-Emaligned -Etest --exclude=/usr/local/go/src --exclude='/mocks/[^/]+\.go.*\((dupl)|(golint)\)' ./...

go_get_rev = go get -d $(1)/... && cd "$(GOPATH)/src/$(1)" && git fetch -av && git reset --hard $(2) && go install ./...

precommit: fix fast_lint test

setup:
	$(call go_get_rev,"golang.org/x/tools/cmd/goimports",$(GOIMPORTS_REV))
	$(call go_get_rev,"github.com/wadey/gocovmerge",$(GOCOVMERGE_REV))
	$(call go_get_rev,"github.com/alecthomas/gometalinter",$(GOMETALINTER_REV))
	go get -u github.com/jstemmer/go-junit-report

	gometalinter --install
	echo $(PATH)
	which go
	which gcc
	go version
	which gometalinter

lint:
	gometalinter $(GOMETALINTER_ARGS)

fast_lint:
	gometalinter --fast $(GOMETALINTER_ARGS)

test:
	go test -v -race ./... | go-junit-report -set-exit-code > test_report.xml

static_build_docker:
	docker run -w /go/src/$(REPO) -v `pwd`:/go/src/$(REPO) docker.internal.justin.tv/feeds/gobuilder:1.10.1-0 make static_build

static_build:
	go build -ldflags "-linkmode external -extldflags -static" -o ./cmd/$(SERVICE)/$(SERVICE) $(MAIN_DIR)
	go build -ldflags "-linkmode external -extldflags -static" -race -o ./cmd/$(SERVICE)/$(SERVICE)_race $(MAIN_DIR)
	touch -mat 201701010000.00 ./cmd/$(SERVICE)/$(SERVICE)
	touch -mat 201701010000.00 ./cmd/$(SERVICE)/$(SERVICE)_race
	# Config file mod times are modified by each git checkout. We reset the mod times so the docker images build with the same SHA
	[ ! -e config ] || touch -mat 201701010000.00 config/

integration_test:
	env ENVIRONMENT=$(ENVIRONMENT) go test -race -run=$(GOTEST) -tags=integration ./...

# use this to make development faster as tests will be cached.
# be sure to run tests with the -race flag enabled before submitting PRs
integration_test_short:
	env ENVIRONMENT=$(ENVIRONMENT) go test -run=$(GOTEST) -tags=integration ./...

build:
	go build -ldflags "-X main.CodeVersion=$(GIT_COMMIT)" $(MAIN_DIR)

build_tag_push: static_build_docker
	./vendor/code.justin.tv/twitch/ecs-deploy/helpers/build-tag-push.sh $(TEAM) $(SERVICE) $(LOCAL_DEPLOY_TAG) ./cmd/$(SERVICE)/Dockerfile

local_deploy: build_tag_push
	docker run --read-only -v $(HOME)/.aws:/root/.aws:ro docker.internal.justin.tv/twitch/ecs-deploy:v0.4.6 /ecs-deploy pipeline user-deploy --user=$(shell whoami) -r $(TEAM) $(SERVICE) $(LOCAL_DEPLOY_ENV) $(LOCAL_DEPLOY_TAG)

revendor:
	rm -rf ./vendor
	dep ensure
	dep prune
	# Do helpers
	mkdir -p vendor/code.justin.tv/twitch/ecs-deploy/helpers/
	cd ../../twitch/ecs-deploy && git fetch -av
	git -C ../../twitch/ecs-deploy show origin/master:helpers/build-tag-push.sh > vendor/code.justin.tv/twitch/ecs-deploy/helpers/build-tag-push.sh
	git -C ../../twitch/ecs-deploy show origin/master:helpers/is-new-deploy.sh > vendor/code.justin.tv/twitch/ecs-deploy/helpers/is-new-deploy.sh
	chmod a+x vendor/code.justin.tv/twitch/ecs-deploy/helpers/is-new-deploy.sh vendor/code.justin.tv/twitch/ecs-deploy/helpers/build-tag-push.sh
	# Now do service common
	mkdir -p vendor/code.justin.tv/feeds/service-common
	cd ../service-common && git fetch -av
	git -C ../service-common/ show origin/master:CommonMakefile.mk > vendor/code.justin.tv/feeds/service-common/CommonMakefile.mk

clean:
	rm -f $(MAIN_DIR)/$(SERVICE)_race $(MAIN_DIR)/$(SERVICE) test_report.xml


run:
	env ENVIRONMENT=$(ENVIRONMENT) go run -race $(MAIN_DIR)/main.go

fix:
	find . -iname '*.go' -not -path '*/vendor/*' -print0 | xargs -0 gofmt -s -w
	find . -iname '*.go' -not -path '*/vendor/*' -print0 | xargs -0 goimports -w
	terraform fmt

# WARNING: WILL `git clean` YOUR REPOSITORY, removing all ignored and not checked in files.
# This simulates a 'fresh' install.
# Will find all terraform files for our systems and verify that, from clean, they plan nothing.  Use this after pushing to master to verify your terraform
# is applied.
terraform_check:
	git clean -dx -f -f
	find . -name main.tf -exec dirname {} \; | grep -v '/.terraform' | grep -v modules | xargs -I {} bash -c 'cd {} && echo {} && terraform init && terraform plan -detailed-exitcode && cd -'

integration_test_cover:
	rm -f cover.out uncovered_functions.txt uncovered_functions.txt function_coverage.txt
	go list -f "env ENVIRONMENT=$(ENVIRONMENT) go test -race -timeout 3m -tags=integration -coverprofile {{.Name}}_{{len .Imports}}_{{len .Deps}}.coverprofile -coverpkg ./cmd/... {{.ImportPath}}" ./cmd/... | xargs -I {} bash -c {}
	gocovmerge `ls *.coverprofile` > cover.out
	rm *.coverprofile
	go tool cover -func cover.out > function_coverage.txt
	# pipe to cat to swallow the error code
	grep '\s0.0%' function_coverage.txt | cat > uncovered_functions.txt
	cat function_coverage.txt
	rm cover.out
	if [ -s uncovered_functions.txt ]; then (echo "There are uncovered functions" && cat uncovered_functions.txt && exit 1) fi;
	rm -f cover.out uncovered_functions.txt uncovered_functions.txt function_coverage.txt
