package dynamodb

import (
	"code.justin.tv/feeds/xray"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

// Plugin allows adding ECS hostname information to xray
type Plugin struct {
}

// ModifyMetadata attaches the hostname to the ECS container name
func (p *Plugin) ModifyMetadata(meta *xray.OptionalMetadata) error {
	meta.DynamoDBTableDetector = p.tableDetector
	return nil
}

var opNameToDiscovery = map[string]func(interface{}) string{
	"BatchGetItem":   batchGetItem,
	"BatchWriteItem": batchWriteItem,
	"CreateTable":    createTable,
	"DeleteItem":     deleteItem,
	"DeleteTable":    deleteTable,
	"DescribeTable":  describeTable,
	"GetItem":        getItem,
	"PutItem":        putItem,
	"Query":          query,
	"Scan":           scan,
	"UpdateItem":     updateItem,
	"UpdateTable":    updateTable,
}

func emptyOnNil(s *string) string {
	if s == nil {
		return ""
	}
	return *s
}

func getItem(i interface{}) string {
	if op, ok := i.(*dynamodb.GetItemInput); ok {
		return emptyOnNil(op.TableName)
	}
	return ""
}

func query(i interface{}) string {
	if op, ok := i.(*dynamodb.QueryInput); ok {
		return emptyOnNil(op.TableName)
	}
	return ""
}

func scan(i interface{}) string {
	if op, ok := i.(*dynamodb.ScanInput); ok {
		return emptyOnNil(op.TableName)
	}
	return ""
}

func putItem(i interface{}) string {
	if op, ok := i.(*dynamodb.PutItemInput); ok {
		return emptyOnNil(op.TableName)
	}
	return ""
}

func updateItem(i interface{}) string {
	if op, ok := i.(*dynamodb.UpdateItemInput); ok {
		return emptyOnNil(op.TableName)
	}
	return ""
}

func updateTable(i interface{}) string {
	if op, ok := i.(*dynamodb.UpdateTableInput); ok {
		return emptyOnNil(op.TableName)
	}
	return ""
}

func batchGetItem(i interface{}) string {
	if op, ok := i.(*dynamodb.BatchGetItemInput); ok {
		// Across tables, not sure which table to return.  Return empty(??)
		if len(op.RequestItems) != 1 {
			return ""
		}
		for tableName := range op.RequestItems {
			return tableName
		}
	}
	return ""
}

func batchWriteItem(i interface{}) string {
	if op, ok := i.(*dynamodb.BatchWriteItemInput); ok {
		// Across tables, not sure which table to return.  Return empty(??)
		if len(op.RequestItems) != 1 {
			return ""
		}
		for tableName := range op.RequestItems {
			return tableName
		}
	}
	return ""
}

func createTable(i interface{}) string {
	if op, ok := i.(*dynamodb.CreateTableInput); ok {
		return emptyOnNil(op.TableName)
	}
	return ""
}

func deleteItem(i interface{}) string {
	if op, ok := i.(*dynamodb.DeleteItemInput); ok {
		return emptyOnNil(op.TableName)
	}
	return ""
}

func deleteTable(i interface{}) string {
	if op, ok := i.(*dynamodb.DeleteTableInput); ok {
		return emptyOnNil(op.TableName)
	}
	return ""
}

func describeTable(i interface{}) string {
	if op, ok := i.(*dynamodb.DescribeTableInput); ok {
		return emptyOnNil(op.TableName)
	}
	return ""
}

func (p *Plugin) tableDetector(r *request.Request) string {
	if r == nil {
		return ""
	}
	if r.Operation == nil {
		return ""
	}
	if r.Params == nil {
		return ""
	}
	discoveryFunction := opNameToDiscovery[r.Operation.Name]
	if discoveryFunction == nil {
		return ""
	}
	return discoveryFunction(r.Params)
}
