package ondemand

import (
	"fmt"
	"io"
	"net/http"
	"strings"

	"code.justin.tv/feeds/service-common"
	"github.com/aws/aws-sdk-go/aws/awsutil"
	"github.com/pkg/errors"
	"goji.io"
	"goji.io/pat"
)

// HTTPServer handles responding to API requests via HTTP
type HTTPServer struct {
	service_common.BaseHTTPServer
	TaskCreation    TaskCreation
	TaskDB          *TaskDB
	httpClient      http.Client
	localServiceMux *goji.Mux
}

func (h *HTTPServer) Routes(mux *goji.Mux) {
	mux.Handle(pat.New("/*"), http.HandlerFunc(h.forwardRequests))
	h.localServiceMux = goji.NewMux()
	h.localServiceMux.Handle(pat.Get("/"), http.HandlerFunc(h.listTasks))
	h.localServiceMux.Handle(pat.Get("/task/:task_id"), http.HandlerFunc(h.showTask))
	h.localServiceMux.Handle(pat.Delete("/task/:task_id"), http.HandlerFunc(h.deleteTask))
	h.localServiceMux.Handle(pat.Post("/delete_task/:task_id"), http.HandlerFunc(h.deleteTask))
}

func (h *HTTPServer) deleteTask(rw http.ResponseWriter, req *http.Request) {
	taskID := pat.Param(req, "task_id")
	task, err := h.TaskDB.GetTaskByID(req.Context(), taskID)
	if err != nil {
		rw.WriteHeader(http.StatusInternalServerError)
		fmt.Fprintf(rw, "Unable to list tasks: %s", err.Error())
		return
	}
	if err := h.TaskCreation.RemoveTask(req.Context(), task); err != nil {
		rw.WriteHeader(http.StatusInternalServerError)
		fmt.Fprintf(rw, "Unable to reload ECS task: %s", err.Error())
		return
	}
	fmt.Fprint(rw, "OK")
}

func (h *HTTPServer) showTask(rw http.ResponseWriter, req *http.Request) {
	taskID := pat.Param(req, "task_id")
	task, err := h.TaskDB.GetTaskByID(req.Context(), taskID)
	if err != nil {
		rw.WriteHeader(http.StatusInternalServerError)
		fmt.Fprintf(rw, "Unable to list tasks: %s", err.Error())
		return
	}
	if task == nil {
		rw.WriteHeader(http.StatusNotFound)
		fmt.Fprint(rw, "Unable to find task")
		return
	}
	ecsTask, err := h.TaskCreation.reloadTask(req.Context(), task)
	if err != nil {
		rw.WriteHeader(http.StatusInternalServerError)
		fmt.Fprintf(rw, "Unable to reload ECS task: %s", err.Error())
		return
	}
	data := ShowTaskData{
		Task:    task,
		ECSTask: ecsTask,
	}
	if err := showTaskTextTemplate.Execute(rw, data); err != nil {
		rw.WriteHeader(http.StatusInternalServerError)
		fmt.Fprintf(rw, "Unable to render tasks template: %s", err.Error())
		return
	}
}

func (h *HTTPServer) listTasks(rw http.ResponseWriter, req *http.Request) {
	tasks, err := h.TaskDB.GetTasks(req.Context())
	if err != nil {
		rw.WriteHeader(http.StatusInternalServerError)
		fmt.Fprintf(rw, "Unable to list tasks: %s", err.Error())
		return
	}
	data := ListTasksData{
		Tasks: tasks,
	}
	if err := listTasksTemplate.Execute(rw, data); err != nil {
		rw.WriteHeader(http.StatusInternalServerError)
		fmt.Fprintf(rw, "Unable to render tasks template: %s", err.Error())
		return
	}
}

func (h *HTTPServer) forwardRequests(rw http.ResponseWriter, req *http.Request) {
	requestingHost, err := parseTaskFromHost(req.Host)
	if err != nil {
		h.localServiceMux.ServeHTTP(rw, req)
		return
	}
	nextHop, err := h.TaskCreation.CreateTask(req.Context(), requestingHost)
	if err != nil {
		rw.WriteHeader(http.StatusServiceUnavailable)
		fmt.Fprintf(rw, "unable to create task: %s\n", err.Error())
		return
	}
	defer func() {
		if err2 := req.Body.Close(); err2 != nil {
			fmt.Println("Error on close", err2)
		}
	}()
	fwdReq, err := http.NewRequest(req.Method, fmt.Sprintf("http://%s:%d", nextHop.host, nextHop.port), req.Body)
	if err != nil {
		rw.WriteHeader(http.StatusServiceUnavailable)
		fmt.Fprintf(rw, "unable to create request: %s\n", err.Error())
		return
	}
	fwdReq.URL.Path = req.URL.Path
	fwdReq.URL.RawQuery = req.URL.RawQuery
	fwdReq.URL.ForceQuery = true
	fwdReq.Header = req.Header
	fmt.Println(awsutil.Prettify(fwdReq))
	resp, err := h.httpClient.Do(fwdReq)
	if err != nil {
		rw.WriteHeader(http.StatusServiceUnavailable)
		fmt.Fprintf(rw, "unable to send backend response: %s\n", err.Error())
		return
	}
	for k, v := range map[string][]string(resp.Header) {
		for _, v2 := range v {
			rw.Header().Add(k, v2)
		}
	}
	rw.WriteHeader(resp.StatusCode)
	if _, err := io.Copy(rw, resp.Body); err != nil {
		rw.WriteHeader(http.StatusServiceUnavailable)
		fmt.Fprintf(rw, "unable to copy response body: %s\n", err.Error())
		return
	}
}

type parsedHost struct {
	team        string
	service     string
	environment string
	imageName   string
}

func (p *parsedHost) String() string {
	return fmt.Sprintf("team: %s service %s env %s image %s", p.team, p.service, p.environment, p.imageName)
}

func parseTaskFromHost(host string) (*parsedHost, error) {
	parts := strings.Split(host, ".")
	if len(parts) < 2 {
		return nil, errors.Errorf("unable to split host %s", host)
	}
	if parts[1] == "us-west2" {
		return nil, errors.Errorf("us-west2 as team is just normal: %s", host)
	}
	imageSha := parts[0]
	secondPart := strings.SplitN(parts[1], "-", 2)
	if len(secondPart) < 2 {
		return nil, errors.Errorf("unable to split host %s", host)
	}
	return &parsedHost{
		team:        secondPart[0],
		service:     secondPart[1],
		environment: "staging",
		imageName:   imageSha,
	}, nil
}
