package ondemand

import (
	"context"
	"fmt"
	"strconv"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/service-common/feedsdynamo"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

type TaskDB struct {
	DynamoClient *dynamodb.DynamoDB
	Config       *TaskDBConfig
}

type TaskDBConfig struct {
	deployedTasksTable *distconf.Str
}

func (t *TaskDBConfig) Load(dconf *distconf.Distconf) error {
	t.deployedTasksTable = dconf.Str("ecs-on-demand.deployed_tasks_table", "")
	if t.deployedTasksTable.Get() == "" {
		return errors.New("unable to find deployed tasks table")
	}
	return nil
}

type RunningTask struct {
	Team        string
	Service     string
	Environment string
	ImageName   string
	TaskCluster string
	TaskARN     string
	Version     int64
	ForwardHost string
	ForwardPort int32

	creationTime time.Time
	updateTime   time.Time
}

func (r *RunningTask) CreationTime() time.Time {
	return r.creationTime
}

func (r *RunningTask) UpdateTime() time.Time {
	return r.updateTime
}

func (r *RunningTask) DynamoID() string {
	return fmt.Sprintf("%s-%s-%s", r.Team, r.Service, r.ImageName)
}

func (r *RunningTask) dynamoKey() map[string]*dynamodb.AttributeValue {
	// Just the "id" part
	fullItem := r.toDynamo()
	return map[string]*dynamodb.AttributeValue{
		"id": fullItem["id"],
	}
}

func (r *RunningTask) toDynamo() map[string]*dynamodb.AttributeValue {
	return map[string]*dynamodb.AttributeValue{
		"team":          {S: &r.Team},
		"environment":   {S: &r.Environment},
		"service":       {S: &r.Service},
		"image_name":    {S: &r.ImageName},
		"task_cluster":  {S: &r.TaskCluster},
		"task_arn":      {S: &r.TaskARN},
		"version":       {N: aws.String(strconv.FormatInt(r.Version, 10))},
		"forward_port":  {N: aws.String(strconv.FormatInt(int64(r.ForwardPort), 10))},
		"forward_host":  {S: &r.ForwardHost},
		"id":            {S: aws.String(r.DynamoID())},
		"creation_time": {N: aws.String(strconv.FormatInt(r.creationTime.Unix(), 10))},
		"update_time":   {N: aws.String(strconv.FormatInt(time.Now().Unix(), 10))},
	}
}

func taskFromDynamo(in map[string]*dynamodb.AttributeValue) (*RunningTask, error) {
	if in == nil {
		return nil, nil
	}
	var m map[string]string
	var n map[string]int64
	var err error
	if m, err = feedsdynamo.AwsStrings(in, []string{"team", "service", "image_name", "task_cluster", "task_arn", "forward_host", "environment"}); err != nil {
		return nil, err
	}
	if n, err = feedsdynamo.AwsInts(in, []string{"version", "forward_port", "creation_time", "update_time"}); err != nil {
		return nil, err
	}
	return &RunningTask{
		Team:         m["team"],
		Service:      m["service"],
		Environment:  m["environment"],
		ImageName:    m["image_name"],
		TaskCluster:  m["task_cluster"],
		TaskARN:      m["task_arn"],
		Version:      n["version"],
		ForwardPort:  int32(n["forward_port"]),
		ForwardHost:  m["forward_host"],
		creationTime: time.Unix(n["creation_time"], 0),
		updateTime:   time.Unix(n["update_time"], 0),
	}, nil
}

func (t *TaskDB) GetTasks(ctx context.Context) ([]*RunningTask, error) {
	in := &dynamodb.ScanInput{
		TableName: aws.String(t.Config.deployedTasksTable.Get()),
	}
	req, out := t.DynamoClient.ScanRequest(in)
	req.SetContext(ctx)
	if err := req.Send(); err != nil {
		return nil, err
	}
	ret := make([]*RunningTask, 0, len(out.Items))
	for _, item := range out.Items {
		t, err := taskFromDynamo(item)
		if err != nil {
			return nil, err
		}
		ret = append(ret, t)
	}
	return ret, nil
}

func (t *TaskDB) DeleteTask(ctx context.Context, r *RunningTask) error {
	in := &dynamodb.DeleteItemInput{
		TableName: aws.String(t.Config.deployedTasksTable.Get()),
		Key: map[string]*dynamodb.AttributeValue{
			"id": {S: aws.String(r.DynamoID())},
		},
	}
	req, _ := t.DynamoClient.DeleteItemRequest(in)
	req.SetContext(ctx)
	return req.Send()
}

func (t *TaskDB) GetTask(ctx context.Context, team string, service string, imageName string) (*RunningTask, error) {
	rt := RunningTask{
		Team:      team,
		Service:   service,
		ImageName: imageName,
	}
	return t.GetTaskByID(ctx, rt.DynamoID())
}

func (t *TaskDB) GetTaskByID(ctx context.Context, id string) (*RunningTask, error) {
	in := &dynamodb.GetItemInput{
		TableName: aws.String(t.Config.deployedTasksTable.Get()),
		Key: map[string]*dynamodb.AttributeValue{
			"id": {S: &id},
		},
	}
	req, out := t.DynamoClient.GetItemRequest(in)
	req.SetContext(ctx)
	if err := req.Send(); err != nil {
		return nil, err
	}
	return taskFromDynamo(out.Item)
}

func (t *TaskDB) StoreTask(ctx context.Context, task *RunningTask) error {
	// TODO: Check version
	task.creationTime = time.Now()
	in := &dynamodb.PutItemInput{
		Item:      task.toDynamo(),
		TableName: aws.String(t.Config.deployedTasksTable.Get()),
	}
	req, _ := t.DynamoClient.PutItemRequest(in)
	req.SetContext(ctx)
	return req.Send()
}
